/*
 *	cook - file construction tool
 *	Copyright (C) 1990, 1991, 1992, 1993, 1994, 1997 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manipulate the symbol table
 */

#include <ac/stddef.h>
#include <ac/stdlib.h>

#include <id/private.h>
#include <id/variable.h>
#include <mem.h>
#include <option.h>
#include <progname.h>
#include <symtab.h>
#include <version-stmp.h>
#include <str_list.h>
#include <trace.h>

static	symtab_ty	*symtab;
	string_ty	*id_need;
	string_ty	*id_younger;
	string_ty	*id_target;
	string_ty	*id_targets;
	string_ty	*id_search_list;


static void reap _((void *));

static void
reap(p)
	void		*p;
{
	id_ty		*idp;

	idp = p;
	id_instance_delete(idp);
}


/*
 * NAME
 *	id_initialize - start up symbol table
 *
 * SYNOPSIS
 *	void id_initialize(void);
 *
 * DESCRIPTION
 *	The id_initialize function is used to create the hash table.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	Assumes the str_initialize function has been called already.
 */

void
id_initialize()
{
	trace(("init\n"));
	id_need = str_from_c("need");
	id_younger = str_from_c("younger");
	id_target = str_from_c("target");
	id_targets = str_from_c("targets");
	id_search_list = str_from_c("search_list");

	id_reset();
}


void
id_reset()
{
	string_list_ty	wl;
	string_ty	*s;

	if (symtab)
		symtab_free(symtab);
	symtab = symtab_alloc(100);
	symtab->reap = reap;

	/*
	 * set the "version" predefined variable
	 */
	string_list_constructor(&wl);
	s = str_from_c(version_stamp());
	string_list_append(&wl, s);
	str_free(s);
	s = str_from_c("version");
	id_assign(s, id_variable_new(&wl));
	str_free(s);
	string_list_destructor(&wl);

	/*
	 * set the "self" predefined variable
	 */
	s = str_from_c(progname_get());
	string_list_append(&wl, s);
	str_free(s);
	s = str_from_c("self");
	id_assign(s, id_variable_new(&wl));
	str_free(s);
	string_list_destructor(&wl);

#ifdef __STDC__
	/*
	 * This symbol is only defined if we have a conforming C
	 * compiler.  This is support for the C recipes.
	 */
	string_list_append(&wl, str_true);
	s = str_from_c("__STDC__");
	id_assign(s, id_variable_new(&wl));
	str_free(s);
	string_list_destructor(&wl);
#endif
	
	/*
	 * set the default search list
	 * in the "search_list" predefined variable
	 */
	s = str_from_c(".");
	string_list_append(&wl, s);
	str_free(s);
	id_assign(id_search_list, id_variable_new(&wl));
	string_list_destructor(&wl);
}


/*
 * NAME
 *	id_search - search for a variable
 *
 * SYNOPSIS
 *	int id_search(string_ty *name, string_list_ty *value);
 *
 * DESCRIPTION
 *	Id_search is used to reference a variable.
 *
 * RETURNS
 *	If the variable has been defined, the function returns a non-zero value
 *	and the value is returned through the 'value' pointer.
 *	If the variable has not been defined, it returns zero,
 *	and 'value' is unaltered.
 *
 * CAVEAT
 *	The value returned from this function, when returned, is allocated
 *	in dynamic memory (it is a copy of the value remembered by this module).
 *	It is the responsibility of the caller to free it when finished with,
 *	by a string_list_destructor() call.
 */

id_ty *
id_search(name)
	string_ty	*name;
{
	id_ty		*data;

	assert(symtab);
	data = symtab_query(symtab, name);
	return data;
}


id_ty *
id_search_fuzzy(name, name_used)
	string_ty	*name;
	string_ty	**name_used;
{
	id_ty		*data;

	assert(symtab);
	data = symtab_query_fuzzy(symtab, name, name_used);
	return data;
}


/*
 * NAME
 *	id_assign - assign a variable
 *
 * SYNOPSIS
 *	void id_assign(string_ty *name, id_ty *);
 *
 * DESCRIPTION
 *	Id_assign is used to assign a value to a given variable.
 *
 * CAVEAT
 *	The name and value are copied by id_assign, so the user may
 *	modify or free them at a later date without affecting the
 *	variable.
 */

void
id_assign(name, arg)
	string_ty	*name;
	id_ty		*arg;
{
	assert(symtab);
	symtab_assign(symtab, name, arg);
}


/*
 * NAME
 *	id_assign_push - assign a variable, remembering old value
 *
 * SYNOPSIS
 *	void id_assign_push(string_ty *name, id_ty *);
 *
 * DESCRIPTION
 *	Id_assign is used to assign a value to a given variable.
 *
 * CAVEAT
 *	The name and value are copied by id_assign, so the user may
 *	modify or free them at a later date without affecting the
 *	variable.
 */

void
id_assign_push(name, arg)
	string_ty	*name;
	id_ty		*arg;
{
	assert(symtab);
	symtab_assign_push(symtab, name, arg);
}


/*
 * NAME
 *	id_unassign - delete a variable
 *
 * SYNOPSIS
 *	void id_unassign(string_ty *name, id_class_ty class);
 *
 * DESCRIPTION
 *	Id_delete is used to delete variables.
 *
 * CAVEAT
 *	No complaint is made if the variable does not exist,
 *	since the user wanted it to not exist.
 */

void
id_unassign(name)
	string_ty	*name;
{
	assert(symtab);
	symtab_delete(symtab, name);
}


/*
 * NAME
 *	id_dump - dump id table
 *
 * SYNOPSIS
 *	void id_dump(char *title, int mask);
 *
 * DESCRIPTION
 *	The id_dump function is used to dump the contents of the id table.
 *	The title will be used to indicate why the table was dumped.  The mask
 *	may be used to selectively dump the table, 0 means everything, bits
 *	correspond directly with ID_CLASS defines.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	This function is only available when symbol DEBUG is defined.
 */

#ifdef DEBUG

void
id_dump(s)
	char		*s;
{
	assert(symtab);
	symtab_dump(symtab, s);
}

#endif


int
id_interpret(idp, ocp, pp)
	id_ty		*idp;
	struct opcode_context_ty *ocp;
	const struct expr_position_ty *pp;
{
	assert(idp);
	assert(idp->method);
	assert(idp->method->interpret);
	return idp->method->interpret(idp, ocp, pp);
}
