/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994, 1996, 1997, 1998 Peter Miller;
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 *
 * MANIFEST: functions to manage command line options
 *
 * The options may be set at various levels.  The level with the highest
 * precedence which has actually been set is used to determine the option
 * value at any given time.
 *
 * Each level of an option is represented by 2 bits in the flag word.  One bit
 * is used to indicate that the option has been set for that level, the other
 * bit indicates the state.  Determining the least set bit in an expression is
 * cheap (x&-x) so highest priority is the lowest numbered level.
 *
 * The COOK enviroment variable is basically a replacement for the defaults,
 * so that users can change the default behaviour.  The command line overrides
 * almost everything.  The error level is the only level with higher
 * precedence than the command line, and it is used to prevent disasters
 * after parse errors or interrupts have happened.
 */

#include <ac/ctype.h>
#include <ac/limits.h>
#include <ac/stddef.h>
#include <ac/stdio.h>
#include <ac/stdlib.h>
#include <ac/string.h>
#include <ac/time.h>

#include <libdir.h>
#include <mem.h>
#include <option.h>
#include <os.h>
#include <progname.h>
#include <trace.h>


typedef struct flag_ty flag_ty;
struct flag_ty
{
	unsigned long o_flag[OPTION_max];
};

option_ty option;
static flag_ty flag;


/*
 * NAME
 *	option_set - set an option
 *
 * SYNOPSIS
 *	void option_set(option_number_ty num, option_level_ty lvl, int state);
 *
 * DESCRIPTION
 *	The option_set function is used to set the given option at the given
 *	level to the given state.
 *
 * RETURNS
 *	void
 */

void
option_set(o, level, state)
	option_number_ty	o;
	option_level_ty	level;
	int		state;
{
	assert((int)o >= 0 && (int)o < (int)OPTION_max);
	if (state)
		flag.o_flag[(size_t)o] |= 3L << (2 * (int)level);
	else
		flag.o_flag[(size_t)o] |= 1L << (2 * (int)level);
}


/*
 * NAME
 *	option_already - see if an option is already set
 *
 * SYNOPSIS
 *	int option_already(option_number_ty num, option_level_ty lvl);
 *
 * DESCRIPTION
 *	The option_already function is used to test if a given option at a
 *	given level has been set.
 *
 * RETURNS
 *	int: zero if the option has not been set, nonzero if it has.
 */

int
option_already(o, level)
	option_number_ty	o;
	option_level_ty	level;
{
	assert((int)o >= 0 && (int)o < (int)OPTION_max);
	return ((flag.o_flag[(size_t)o] >> (2 * (int)level)) & 1) != 0;
}


/*
 * NAME
 *	option_undo - remove option setting
 *
 * SYNOPSIS
 *	void option_undo(option_number_ty num, option_level_ty lvl);
 *
 * DESCRIPTION
 *	The option_undo function is used to is used to remove the option
 *	setting for the given option at the given level.
 *
 * RETURNS
 *	void
 */

void
option_undo(o, level)
	option_number_ty	o;
	option_level_ty	level;
{
	assert((int)o >= 0 && (int)o < (int)OPTION_max);
	flag.o_flag[(size_t)o] &= ~(3L << (2 * (int)level));
}


/*
 * NAME
 *	option_undo_level - remove options settings
 *
 * SYNOPSIS
 *	void option_undo_level(option_level_ty lvl);
 *
 * DESCRIPTION
 *	The option_undo_level function is used to remove the settings for all
 *	options at a given level.
 *
 * RETURNS
 *	void
 */

void
option_undo_level(level)
	option_level_ty	level;
{
	int		o;

	for (o = 0; o < (int)OPTION_max; ++o)
		option_undo((option_number_ty)o, level);
}


/*
 * NAME
 *	option_test - test an option
 *
 * SYNOPSIS
 *	int option_test(option_number_ty num);
 *
 * DESCRIPTION
 *	The option_test function is used to test the setting of an option.
 *	The level of highest precedence which hash been set is used.
 *
 * RETURNS
 *	int: zero if the option is off, nonzero if the option is on.
 */

int
option_test(o)
	option_number_ty o;
{
	unsigned long	*op;
	unsigned long	mask;

	assert((int)o >= 0 && (int)o < (int)OPTION_max);
	op = &flag.o_flag[(size_t)o];
	mask = *op & 0x55555555;
	mask &= -mask; /* get LSB */
	return (*op & (mask << 1)) != 0;
}


/*
 * NAME
 *	option_tidy_up - mother hen
 *
 * SYNOPSIS
 *	void option_tidy_up(void);
 *
 * DESCRIPTION
 *	The option_tidy_up function is used to set a few defaults, and tidy up
 *	after the command line.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	Must be called after the command line has been parsed.
 */

void
option_tidy_up()
{
	string_ty	*s;
	string_ty	*s1;

	/*
	 * user's library
	 */
	trace(("tidy up\n"));
	s = os_accdir();
	assert(s);
	s1 = str_format("%S/.%s", s, progname_get());
	str_free(s);
	string_list_append_unique(&option.o_search_path, s1);
	str_free(s1);

	/*
	 * cook's library
	 *	architecture-specific, then architecture-neutral
	 */
	s = str_from_c(library_directory_get());
	string_list_append_unique(&option.o_search_path, s);
	str_free(s);
	s = str_from_c(data_directory_get());
	string_list_append_unique(&option.o_search_path, s);
	str_free(s);

	if (!option.o_book)
	{
		static char *name[] =
		{
			".how.to.%s",
			".howto.%s",
			"how.to.%s",
			"howto.%s",
			"%s.file",
			"%sfile",
			".%s.rc",
			".%src",
		};
		int	j;
	
		/*
		 * A huge range of alternative default names is given.
		 * The first found will be used.
		 */
		for (j = 0; j < SIZEOF(name); j++)
		{
			char	format[300];

			strcpy(format, name[j]);
			floop:
			s = str_format(format, progname_get());
			switch (os_exists(s))
			{
			case -1:
				exit(1);

			case 0:
				str_free(s);
				if (islower(format[0]))
				{
					format[0] = toupper(format[0]);
					goto floop;
				}
				continue;

			case 1:
				option.o_book = s;
				break;
			}
			break;
		}
	}

	if (!option.o_logfile && option.o_book)
	{
		char		*sp;
		char		*cp;

		sp = option.o_book->str_text;
		/* skip first char in case it's a '.' */
		cp = strrchr(sp + 1, '.');
		if (cp)
			s = str_n_from_c(sp, cp - sp);
		else
			s = str_copy(option.o_book);
		sp = (option_test(OPTION_CMDFILE) ? "sh" : "list");
		option.o_logfile = str_format("%S.%s", s, sp);
		str_free(s);
	}

	option_set(OPTION_ACTION, OPTION_LEVEL_DEFAULT, 1);
	option_set(OPTION_CASCADE, OPTION_LEVEL_DEFAULT, 1);
	option_set(OPTION_LOGGING, OPTION_LEVEL_DEFAULT, 1);
	option_set(OPTION_TERMINAL, OPTION_LEVEL_DEFAULT, 1);
	option_set(OPTION_STRIP_DOT, OPTION_LEVEL_DEFAULT, 1);
	option_set(OPTION_INCLUDE_COOKED, OPTION_LEVEL_DEFAULT, 1);
	option_set(OPTION_FINGERPRINT_WRITE, OPTION_LEVEL_DEFAULT, 1);
}


/*
 * NAME
 *	option_set_errors - set error flags
 *
 * SYNOPSIS
 *	void option_set_errors(void);
 *
 * DESCRIPTION
 *	The option_set_errors function is used to set the appropriate options
 *	to prevent undesirable side effects when errors occur.
 *
 * RETURNS
 *	void
 */

void
option_set_errors()
{
	option_set(OPTION_SILENT, OPTION_LEVEL_ERROR, 0);
	option_set(OPTION_ACTION, OPTION_LEVEL_ERROR, 0);
	option_set(OPTION_ERROK, OPTION_LEVEL_ERROR, 0);
	option_set(OPTION_METER, OPTION_LEVEL_ERROR, 0);
	option_set(OPTION_PERSEVERE, OPTION_LEVEL_ERROR, 0);
}


void *
option_flag_state_get()
{
	flag_ty		*fp;

	fp = mem_alloc(sizeof(flag_ty));
	*fp = flag;
	return fp;
}


void
option_flag_state_set(p)
	void		*p;
{
	flag_ty		*fp;

	fp = p;
	flag = *fp;
	mem_free(p);
}
