/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "gladeconfig.h"

#include "project.h"
#include "gbwidget.h"
#include "gbwidgetarray.h"
#include "palette.h"
#include "palettebutton.h"

/* Include the selector pixmap */
#include "graphics/selector.xpm"

/* Size of the palette's table, though it should expand OK as widgets are
   added. */
#define PALETTE_ROWS 15
#define PALETTE_COLS 4

/* This key is used for each button in the palette to store the associated
   GtkWidget class name. */
#define GB_CLASS_KEY		"GbClass"

/* The button corresponding to the selector (arrow) icon. */
static GtkWidget *palette_selector = NULL;
/* The currently active button. */
static GtkWidget *palette_widget = NULL;

static GtkWidget *win_palette = NULL;
static GtkWidget *palette_table = NULL;

/* The control button can be used to hold the selected widget, so it
   doesn't automatically revert to the selector as is normally the case. */
static gboolean hold_selected_widget = FALSE;

static GtkTooltips *tooltips;
static GdkColor tooltips_fgcolor =
{0, 0, 0, 0};
static GdkColor tooltips_bgcolor =
{0, 0xffff, 0xffff, 0};

static void palette_create (void);
static void palette_toggle_widget (GtkWidget * widget,
				   gpointer data);


static void
palette_create ()
{
  GtkWidget *button, *label;
  int index, x, y;
  GbWidget *gbwidget;
  gchar **pixmap_struct;
  gchar *tooltip;
  GSList *group = NULL;
  GdkColormap *colormap;

  win_palette = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_uposition (win_palette, 0, 235);
  /* We have to realize win_palette so we can create the icon pixmaps. */
  gtk_widget_realize (win_palette);

  gtk_signal_connect (GTK_OBJECT (win_palette), "delete_event",
		      GTK_SIGNAL_FUNC (palette_hide), NULL);

  gtk_window_set_title (GTK_WINDOW (win_palette), _("Palette"));
  gtk_container_border_width (GTK_CONTAINER (win_palette), 0);

  palette_table = gtk_table_new (PALETTE_ROWS, PALETTE_COLS, FALSE);
  gtk_container_add (GTK_CONTAINER (win_palette), palette_table);
  gtk_widget_show (palette_table);

  tooltips = gtk_tooltips_new ();
  colormap = gtk_widget_get_colormap (win_palette);
  if (gdk_color_alloc (colormap, &tooltips_fgcolor)
      && gdk_color_alloc (colormap, &tooltips_bgcolor))
    gtk_tooltips_set_colors (tooltips, &tooltips_bgcolor, &tooltips_fgcolor);

  index = 0;
  x = y = 0;

  for (;;)
    {
      if (widgets[index] == NULL)
	{
	  /* 2 NULLs signify end of widget array */
	  if (widgets[index + 1] == NULL)
	    break;

	  /* 1 NULL followed by a string starts a new section with a label,
	     but if the section is 'NotShown:' exit the loop */
	  if (!strcmp (widgets[index + 1], "NotShown:"))
	    break;

	  label = gtk_label_new (gettext (widgets[index + 1]));
	  gtk_widget_set_sensitive (label, FALSE);
	  gtk_widget_show (label);
	  /* Make sure the label is on its own row */
	  if (x != 0)
	    {
	      x = 0;
	      y++;
	    }
	  gtk_table_attach (GTK_TABLE (palette_table), label, 0, PALETTE_COLS,
			    y, y + 1,
			    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL,
			    2, 2);
	  y++;
	}
      else
	{
	  button = gtk_palette_button_new (group);
	  group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
	  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (button), FALSE);
	  gtk_widget_show (button);

	  if (index == 0)
	    {
	      gbwidget = NULL;
	      palette_widget = palette_selector = button;
	      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), TRUE);
	      tooltip = _("Selector");
	      pixmap_struct = selector_xpm;
	    }
	  else
	    {
	      /* Look up GbWidget struct using the class name */
	      gbwidget = (GbWidget *) g_hash_table_lookup (gb_widget_table,
							   widgets[index]);
	      g_return_if_fail (gbwidget != NULL);
	      /* Save the class name in the buttons data hash */
	      gtk_object_set_data (GTK_OBJECT (button), GB_CLASS_KEY,
				   widgets[index]);

	      pixmap_struct = gbwidget->pixmap_struct;
	      tooltip = gbwidget->tooltip;
	    }

	  if (pixmap_struct)
	    {
	      GtkWidget *pixmap;
	      GdkPixmap *gdkpixmap;
	      GdkBitmap *mask;
	      gdkpixmap = gdk_pixmap_create_from_xpm_d (win_palette->window,
							&mask,
				  &win_palette->style->bg[GTK_STATE_NORMAL],
							pixmap_struct);
	      pixmap = gtk_pixmap_new (gdkpixmap, mask);
	      gtk_widget_show (pixmap);
	      gtk_container_add (GTK_CONTAINER (button), pixmap);

	      /* Don't unref the gdkpixmap or mask since we keep pointers to
	         them. */
	      if (gbwidget)
		{
		  gbwidget->gdkpixmap = gdkpixmap;
		  gbwidget->mask = mask;
		}
	    }

	  gtk_signal_connect (GTK_OBJECT (button), "clicked",
			   GTK_SIGNAL_FUNC (palette_toggle_widget), button);

	  gtk_tooltips_set_tip (tooltips, button, tooltip, NULL);

	  gtk_table_attach (GTK_TABLE (palette_table), button, x, x + 1,
			    y, y + 1,
			    GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL,
			    0, 0);
	  x++;
	  if (x == 4)
	    {
	      x = 0;
	      y++;
	    }
	}
      index += 2;
    }
}


void
palette_show (GtkWidget * widget,
	      gpointer data)
{
  if (!win_palette)
    palette_create ();
  gtk_widget_show (win_palette);
  /* This maps the window, which also de-iconifies it according to ICCCM. */
  gdk_window_show (GTK_WIDGET (win_palette)->window);
  gdk_window_raise (GTK_WIDGET (win_palette)->window);
}


gint
palette_hide (GtkWidget * widget,
	      gpointer data)
{
  gtk_widget_hide (win_palette);
  return TRUE;
}


gboolean
palette_is_selector_on ()
{
  return (palette_widget == palette_selector) ? TRUE : FALSE;
}


gchar *
palette_get_widget_class ()
{
  if (palette_widget && palette_widget != palette_selector)
    return gtk_object_get_data (GTK_OBJECT (palette_widget), GB_CLASS_KEY);
  else
    return NULL;
}


static void
palette_toggle_widget (GtkWidget * widget,
		       gpointer data)
{
  GdkModifierType modifiers;
  if ((GTK_TOGGLE_BUTTON (widget)->active))
    {
      palette_widget = widget;
      gdk_window_get_pointer (widget->window, NULL, NULL, &modifiers);
      hold_selected_widget = (modifiers & GDK_CONTROL_MASK) ? TRUE : FALSE;
    }
}


void
palette_reset_selection ()
{
  if (!hold_selected_widget)
    {
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (palette_selector), TRUE);
      gtk_container_set_focus_child (GTK_CONTAINER (palette_table),
				     palette_selector);
    }
}
