/*
 *	Software for Humanity
 *	Public Domain
 *	GDB/RBD
 *
 *	This program is freely distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 *	$Id: all_argv.c,v 6.1.1.1 97/02/25 17:18:08 nevin Exp $
 *
 *	Function:	- argument vector library
 */

#include <stdlib.h>
#include <string.h>

#include "lam.h"

#define ARGSIZE		64

/*
 * global functions
 */
int			sfh_argv_add();
int			sfh_argv_count();
unint			sfh_argv_len();
void			sfh_argv_free();
char			**sfh_argv_break();
char			**sfh_argv_dup();
char			*sfh_argv_glue();

/*
 *	sfh_argv_add
 *
 *	Function:	- extends an argv with a new arg
 *	Accepts:	- argc (value returned)
 *			- argv (value returned)
 *			- new arg
 *	Returns:	- 0 or -1 (error)
 */
int
sfh_argv_add(argc, argv, arg)

int			*argc;
char			***argv;
char			*arg;

{
/*
 * Create new argv.
 */
	if (*argv == 0) {
		*argv = (char **) malloc(2 * sizeof(char *));
		if (*argv == 0) return(-1);
		(*argv)[0] = 0;
		(*argv)[1] = 0;
	}
/*
 * Extend existing argv.
 */
	else {
		*argv = (char **) realloc((char *) *argv,
				(unsigned) (*argc + 2) * sizeof(char *));
		if (*argv == 0) return(-1);
	}

	(*argv)[*argc] = malloc((unsigned) strlen(arg) + 1);
	if ((*argv)[*argc] == 0) return(-1);

	strcpy((*argv)[*argc], arg);
	*argc = *argc + 1;
	(*argv)[*argc] = 0;
	return(0);
}

/*
 *	sfh_argv_free
 *
 *	Function:	- frees memory associated with argv
 *	Accepts:	- argv
 */
void
sfh_argv_free(argv)

char			**argv;

{
	char		**p;

	if (argv == 0) return;

	for (p = argv; *p; ++p) {
		free(*p);
	}

	free((char *) argv);
}

/*
 *	sfh_argv_break
 * 
 *	Function:	- breaks single string into argv
 *			- allocates space for argv structure
 *			  which must be freed by the caller
 *	Accepts:	- argument string
 *			- arg delimiter within string
 *	Returns:	- argv or NULL (error)
 */
char **
sfh_argv_break(args, delim)

CONST char		*args;
int			delim;

{
	char		arg[ARGSIZE];	/* single argument */
	char		**argv = 0;	/* argument structure ptr */
	CONST char	*p;		/* walk through args */
	char		*argtemp;	/* temporary single argument */
	int		argc = 0;	/* # arguments */
	unint		arglen;		/* single argument length */

	while (*args) {
		p = args;
		arglen = 0;

		while ((*p != '\0') && (*p != delim)) {
			p++;
			arglen++;
		}
/*
 * zero length argument, skip
 */
		if (args == p) {
			p++;
		}
/*
 * tail argument, add straight from the original string
 */
		else if (*p == '\0') {

			if (sfh_argv_add(&argc, &argv, args)) return(0);
		}
/*
 * long argument, malloc buffer, copy and add
 */
		else if (arglen > (ARGSIZE - 1)) {
			argtemp = malloc((unsigned) (arglen + 1));
			if (argtemp == 0) return(0);

			strncpy(argtemp, args, arglen);
			argtemp[arglen] = '\0';

			if (sfh_argv_add(&argc, &argv, argtemp)) {
				free(argtemp);
				return(0);
			}

			free(argtemp);
		}
/*
 * short argument, copy to buffer and add
 */
		else {
			strncpy(arg, args, arglen);
			arg[arglen] = '\0';

			if (sfh_argv_add(&argc, &argv, arg)) return(0);
		}

		args = p;
	}

	return(argv);
}

/*
 *	sfh_argv_count
 * 
 *	Function:	- count the # of entries in argv
 *	Accepts:	- argv
 *	Returns:	- # of entries
 */
int
sfh_argv_count(argv)

char			**argv;

{
	char		**p;
	int		i;

	if (argv == 0) return(0);

	for (i = 0, p = argv; *p; i++, p++);

	return(i);
}

/*
 *	sfh_argv_glue
 * 
 *	Function:	- glues argv into single string
 *			- delimiter and max string length specified
 *			- allocates space for string which must be
 *			  freed by the caller
 *	Accepts:	- argv
 *			- arg delimiter within string
 *			- max string length (or 0 meaning full argv)
 *	Returns:	- string or NULL (error)
 */
char *
sfh_argv_glue(argv, delim, maxlen)

char			**argv;
int			delim;
unint			maxlen;

{
	char		**p;		/* favorite pointer */
	char		*pp;		/* another pointer */
	char		*str;		/* glued string */
	unint		str_len = 0;	/* string length */
	unint		i;		/* favorite index */
/*
 * Find the total string length in argv including delimiters.
 * The last delimiter is replaced by the NULL character.
 */
	for (p = argv; *p; p++) {
		str_len += strlen(*p) + 1;
	}

	if ((maxlen > 0) && (maxlen + 1 < str_len)) {
		str_len = maxlen + 1;
	}
/*
 * Allocate the string.
 */
	if ((str = malloc((unsigned) str_len)) == 0) return(0);
/*
 * Loop filling in the string.
 */
	str[--str_len] = '\0';
	p = argv;
	pp = *p;

	for (i = 0; i < str_len; i++) {

		if (*pp == '\0') {
/*
 * End of a string, fill in a delimiter and go to the next string.
 */
			str[i] = (char) delim;
			p++;
			pp = *p;
		} else {
			str[i] = *pp++;
		}
	}

	return(str);
}

/*
 *	sfh_argv_len
 * 
 *	Function:	- count the bytes consumed by an argv
 *	Accepts:	- argv
 *	Returns:	- length
 */
unint
sfh_argv_len(argv)

char			**argv;

{
	char		**p;
	unint		length;		/* argv length */

	if (argv == 0) return(0);

	length = sizeof(char *);

	for (p = argv; *p; p++) {
		length += strlen(*p) + 1 + sizeof(char *);
	}

	return(length);
}

/*
 *	sfh_argv_dup
 *
 *	Function:	- create a duplicate copy of an argv
 *	Accepts:	- argv
 *	Returns:	- duplicate argv or NULL
 */
char **
sfh_argv_dup(argv)

char			**argv;

{
	char		**dupv = 0;	/* duplicate argv */
	int		dupc = 0;	/* dupv count */

	if (argv == 0) return(0);

	while (*argv != 0) {

		if (sfh_argv_add(&dupc, &dupv, *argv)) {
			sfh_argv_free(dupv);
			return(0);
		}

		argv++;
	}

	return(dupv);
}

/*
 * backwards compatibility
 */
int addarg(argc, argv, arg) int *argc; char ***argv, *arg;
{ return(sfh_argv_add(argc, argv, arg)); }

int argvadd(argc, argv, arg) int *argc; char ***argv, *arg;
{ return(sfh_argv_add(argc, argv, arg)); }

void argvfree(argv) char **argv;
{ sfh_argv_free(argv); }

char **argvbreak(args, delim) CONST char *args; int delim;
{ return(sfh_argv_break(args, delim)); }

int argvcount(argv) char **argv;
{ return(sfh_argv_count(argv)); }

char *argvglue(argv, delim, maxlen) char **argv; int delim; unint maxlen;
{ return(sfh_argv_glue(argv, delim, maxlen)); }

unint argvlen(argv) char **argv;
{ return(sfh_argv_len(argv)); }

char **argvdup(argv) char **argv;
{ return(sfh_argv_dup(argv)); }
