/*
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/NJN
 *
 *	$Id: mpisys.h,v 6.1.1.1 97/01/21 10:01:14 nevin Exp $
 *
 *	Function:	- MPI system header file
 *			- the real meat is here
 */

#ifndef _MPISYS_H
#define _MPISYS_H

#include <errno.h>
#include <stdio.h>

#include <all_list.h>
#include <all_hash.h>
#include <app_mgmt.h>
#include <kio.h>
#include <mpi.h>
#include <portable.h>
#include <terror.h>

/*
 * LAM constants & macros
 */
#define LAM_PREDEF		0x01		/* flags object as predefined */
#define LAM_COLLMAXLIN		4		/* linear coll. op. threshold */
#define LAM_COLLMAXDIM		16		/* maximum cube dimension */
#define LAM_PORTLEN		16		/* maximum port name length */
#define LAM_MAXTAG		32767		/* maximum tag */

#define RPI_SPLIT(l, c, args) 	((lam_c2c) ? c args : l args)

/*
 * These constants correspond to FORTRAN datatypes.  They are used internally
 * and should not be visible to C MPI user code.
 */
#define	MPI_F_INTEGER		((MPI_Datatype) &lam_mpi_integer)
#define	MPI_F_REAL		((MPI_Datatype) &lam_mpi_real)
#define	MPI_F_DOUBLE_PRECISION	((MPI_Datatype) &lam_mpi_dblprec)
#define	MPI_F_CHARACTER		((MPI_Datatype) &lam_mpi_character)
#define	MPI_F_LOGICAL		((MPI_Datatype) &lam_mpi_logic)
#define MPI_F_COMPLEX		((MPI_Datatype) &lam_mpi_cplex)
#define MPI_F_DOUBLE_COMPLEX	((MPI_Datatype) &lam_mpi_dblcplex)
#define MPI_F_2INTEGER		((MPI_Datatype) &lam_mpi_2integer)
#define MPI_F_2REAL		((MPI_Datatype) &lam_mpi_2real)
#define MPI_F_2DOUBLE_PRECISION	((MPI_Datatype) &lam_mpi_2dblprec)

/*
 * structures
 */
struct _dtype {
	int		dt_format;
#define LAM_DTBASIC	0			/* basic datatype */
#define LAM_DTCONTIG	1			/* contiguous */
#define LAM_DTVECTOR	2			/* vector */
#define LAM_DTHVECTOR	3			/* hvector */
#define LAM_DTINDEXED	4			/* indexed */
#define LAM_DTHINDEXED	5			/* hindexed */
#define LAM_DTSTRUCT	6			/* struct */
	int		dt_flags;
#define LAM_DTHASUB	0x002			/* ub is in the type map? */
#define LAM_DTHASLB	0x004			/* lb is in the type map? */
#define LAM_DTNOPACK	0x008			/* no packing required? */
#define LAM_DTNOXADJ	0x010			/* extent not adjusted? */
#define LAM_DTLOWSET	0x100			/* lower bound set */
#define LAM_DTHIGHSET	0x200			/* upper bound set */
	int		dt_label;		/* flat-form label */
	int		dt_commit;		/* committed? */
	int		dt_refcount;		/* reference count */
	int		dt_align;		/* alignment */
	int		dt_upper;		/* upper extent */
	int		dt_lower;		/* lower extent */
	int		dt_size;		/* basic size */
	int		dt_dataup;		/* data upper limit */
	int		dt_datalow;		/* data lower limit */
	int		dt_nelem;		/* # top elements */
	int		dt_count;		/* count */
	int		dt_length;		/* vector length */
	MPI_Aint	dt_stride;		/* vector stride */
	MPI_Datatype	dt_dtype;		/* c/v/i datatype */
	int		*dt_lengths;		/* i/s lengths */
	MPI_Aint	*dt_disps;		/* i/s displacements */
	MPI_Datatype	*dt_dtypes;		/* struct datatypes */
};

struct _group {
	int		g_nprocs;		/* # processes */
	int		g_myrank;		/* my local rank */
	int		g_refcount;		/* reference count */
	struct _proc	**g_procs;		/* ptr to processes */
};

struct _comm {
	int		c_flags;
#define LAM_CINTER	0x02			/* intercommunicator? */
#define LAM_CLDEAD	0x04			/* local group dead? */
#define LAM_CRDEAD	0x08			/* remote group dead? */	
	int		c_contextid;		/* context ID */
	int		c_refcount;		/* reference count */
	int		c_cube_dim;		/* inscribing cube dim. */
	MPI_Group	c_group;		/* local group */
	MPI_Group	c_rgroup;		/* remote group */
	MPI_Errhandler	c_errhdl;		/* error handle */
	HASH		*c_keys;		/* keys cache hash table */
	int		c_topo_type;		/* topology type */
	int		c_topo_nprocs;		/* # topo. processes */
	int		c_topo_ndims;		/* # cart. dimensions */
	int		c_topo_nedges;		/* # graph edges */
	int		*c_topo_dims;		/* cart. dimensions */
	int		*c_topo_coords;		/* cart. coordinates */
	int		*c_topo_index;		/* graph indices */
	int		*c_topo_edges;		/* graph edges */
};

struct _op {
	void		(*op_func)();		/* reduction function */
	int		op_commute;		/* commutative? */
	int		op_f77dtype;		/* f77 datatype */
	int		op_flags;		/* properties */
};

struct _errhdl {
	int		eh_f77hdl;		/* f77 error handle */
	int		eh_refcount;		/* reference count */
	void		(*eh_func)();		/* error function */
};

struct _attrkey {
	int		(*ak_copy)();		/* copy function */
	int		(*ak_del)();		/* delete function */
	void		*ak_extra;		/* extra info */
	int		ak_refcount;		/* reference count */
	int		ak_f77comm;		/* f77 communicator */
	int		ak_flags;		/* properties */
};

struct _attr {
	int		a_key;			/* attribute key */
	void		*a_value;		/* attribute value */
};

struct _fyimsg {
	int4		fym_src;		/* src global/local rank */
	int4		fym_dest;		/* dest global/local rank */
	int4		fym_count;		/* count */
	int4		fym_dtpid;		/* datatype/pid */
};

struct _fyiproc {
	int4		fyp_me;			/* my global/local rank */
	int4		fyp_peer;		/* peer global/local rank */
	int4		fyp_peergps;		/* peer node/index */
	int4		fyp_root;		/* root global/local rank */
	int4		fyp_rootgps;		/* root node/index */
	int4		fyp_cidtag;		/* context ID & tag */
	int4		fyp_func;		/* top level function */
	int4		fyp_count;		/* count */
	int4		fyp_dtype;		/* datatype */
};

struct _info {
	char		info_key[MPI_INFO_MAX_KEYLEN + 1];
	char		*info_value;
};

struct _port {
	char		prt_name[LAM_PORTLEN + 1];
	int		prt_num;
};

/*
 * for internal purposes
 */
extern struct _proc	*lam_myproc;
extern LIST		*lam_comms;
extern LIST		*lam_ports;
extern float8		lam_clockskew;
extern int		lam_f77init;
extern int		lam_c2c;
extern int		lam_ger;
extern int		lam_homog;
extern int		lam_jobid;
extern int		lam_universe_size;
extern int		lam_rq_flblock;
extern int		lam_rq_nreqs;
extern MPI_Request	lam_rq_top;
extern MPI_Request	lam_rq_bottom;
extern int		lam_topfunc;
extern int		lam_flinit;
extern int		lam_flfinal;

/*
 * polling and locking
 */
extern int		_lock_poll;
extern int		_lock_delay;
extern int		_shm_poll_delay;

extern struct kio_t     _kio;			/* kernel I/O block */

/*
 * state checking macros
 */
#define	LAM_IS_INTER(c)	((c)->c_flags & LAM_CINTER)	/* is an intercomm */
#define LAM_IS_INTRA(c) (!((c)->c_flags & LAM_CINTER))	/* is an intracomm */
#define LAM_IS_CART(c)	((c)->c_topo_type == MPI_CART)	/* is cartesian */
#define LAM_IS_GRAPH(c) ((c)->c_topo_type == MPI_GRAPH)	/* is graph */

/*
 * useful tracing macros
 */
#define LAM_TRACE(a)	if ((_kio.ki_rtf & RTF_TRON) == RTF_TRON) a

#define LAM_TRACE_TOP() \
		(((_kio.ki_rtf & RTF_TRON) == RTF_TRON) && (!lam_tr_incff()))

/*
 * overhead reduction
 */
#define _mpi_req_add_m(req)				\
{							\
	if (lam_rq_top == 0) {				\
		lam_rq_top = (req);			\
	} else {					\
		lam_rq_bottom->rq_next = (req);		\
	}						\
							\
	lam_rq_bottom = (req);				\
	(req)->rq_next = 0;				\
							\
	++lam_rq_nreqs;					\
}

#define _mpi_req_rem_m(req)					\
{								\
	MPI_Request	p, p2;					\
	for (p = lam_rq_top, p2 = 0; p && (p != (req));		\
		p2 = p, p = p->rq_next);			\
	if (p) {						\
	  if (p == lam_rq_top) {				\
	    lam_rq_top = p->rq_next;				\
	  } else {						\
	    p2->rq_next = p->rq_next;				\
	  }							\
	  if (p == lam_rq_bottom) lam_rq_bottom = p2;		\
	  --lam_rq_nreqs;					\
	}							\
}

#define _mpi_req_blkclr_m()				\
{							\
	MPI_Request	p;				\
	for (p = lam_rq_top; p; p = p->rq_next) {	\
		p->rq_flags &= ~LAM_RQFBLOCK;		\
	}						\
	lam_rq_flblock = 0;				\
}

#define _mpi_req_blkset_m(req)			\
{						\
	req->rq_flags |= LAM_RQFBLOCK;		\
	lam_rq_flblock = 1;			\
}

#define lam_setfunc_m(locfunc)				\
{							\
	if (lam_topfunc == 0) {				\
		lam_topfunc = (locfunc);		\
	}						\
}

#define lam_resetfunc_m(locfunc)				\
if (lam_topfunc == (locfunc)) {					\
	((struct _fyiproc *) _kio.ki_fyi)->fyp_func = 0;	\
	lam_topfunc = 0;					\
}

#define lam_inited_m()	(lam_flinit)

#define lam_initerr_m()							\
if (lam_flinit == 0) {							\
	errno = EMPINOINIT;						\
	fprintf(stderr, "MPI error: process not initialized\n");	\
	exit(errno);							\
}

/*
 * profiling
 */
#ifdef PROFILELIB

#define MPI_Abort			PMPI_Abort 
#define MPI_Accept			PMPI_Accept
#define MPI_Address			PMPI_Address
#define MPI_Allgather			PMPI_Allgather 
#define MPI_Allgatherv			PMPI_Allgatherv 
#define MPI_Allreduce			PMPI_Allreduce 
#define MPI_Alltoall			PMPI_Alltoall 
#define MPI_Alltoallv			PMPI_Alltoallv 
#define MPI_Attr_delete			PMPI_Attr_delete 
#define MPI_Attr_get			PMPI_Attr_get 
#define MPI_Attr_put			PMPI_Attr_put 
#define MPI_Barrier			PMPI_Barrier 
#define MPI_Bcast			PMPI_Bcast 
#define MPI_Bsend			PMPI_Bsend 
#define MPI_Bsend_init			PMPI_Bsend_init 
#define MPI_Buffer_attach		PMPI_Buffer_attach 
#define MPI_Buffer_detach		PMPI_Buffer_detach 
#define MPI_Cancel			PMPI_Cancel 
#define MPI_Cart_coords			PMPI_Cart_coords 
#define MPI_Cart_create			PMPI_Cart_create 
#define MPI_Cart_get			PMPI_Cart_get 
#define MPI_Cart_map			PMPI_Cart_map 
#define MPI_Cart_rank			PMPI_Cart_rank 
#define MPI_Cart_shift			PMPI_Cart_shift 
#define MPI_Cart_sub			PMPI_Cart_sub 
#define MPI_Cartdim_get			PMPI_Cartdim_get 
#define MPI_Comm_compare		PMPI_Comm_compare 
#define MPI_Comm_create			PMPI_Comm_create 
#define MPI_Comm_disconnect		PMPI_Comm_disconnect
#define MPI_Comm_dup			PMPI_Comm_dup 
#define MPI_Comm_free			PMPI_Comm_free 
#define MPI_Comm_get_contextid		PMPI_Comm_get_contextid 
#define MPI_Comm_group			PMPI_Comm_group 
#define MPI_Comm_rank			PMPI_Comm_rank 
#define MPI_Comm_remote_group		PMPI_Comm_remote_group 
#define MPI_Comm_remote_size		PMPI_Comm_remote_size 
#define MPI_Comm_size			PMPI_Comm_size 
#define MPI_Comm_split			PMPI_Comm_split 
#define MPI_Comm_test_inter		PMPI_Comm_test_inter 
#define MPI_Connect			PMPI_Connect
#define MPI_Dims_create			PMPI_Dims_create 
#define MPI_Errhandler_create		PMPI_Errhandler_create 
#define MPI_Errhandler_free		PMPI_Errhandler_free 
#define MPI_Errhandler_get		PMPI_Errhandler_get 
#define MPI_Errhandler_set		PMPI_Errhandler_set 
#define MPI_Error_class			PMPI_Error_class 
#define MPI_Error_string		PMPI_Error_string 
#define MPI_Finalize			PMPI_Finalize 
#define MPI_Gather			PMPI_Gather 
#define MPI_Gatherv			PMPI_Gatherv 
#define MPI_Get_count			PMPI_Get_count 
#define MPI_Get_elements		PMPI_Get_elements 
#define MPI_Get_processor_name		PMPI_Get_processor_name 
#define MPI_Get_version			PMPI_Get_version
#define MPI_Graph_create		PMPI_Graph_create 
#define MPI_Graph_get			PMPI_Graph_get 
#define MPI_Graph_map			PMPI_Graph_map 
#define MPI_Graph_neighbors		PMPI_Graph_neighbors 
#define MPI_Graph_neighbors_count	PMPI_Graph_neighbors_count 
#define MPI_Graphdims_get		PMPI_Graphdims_get 
#define MPI_Group_compare		PMPI_Group_compare 
#define MPI_Group_difference		PMPI_Group_difference 
#define MPI_Group_excl			PMPI_Group_excl 
#define MPI_Group_free			PMPI_Group_free 
#define MPI_Group_incl			PMPI_Group_incl 
#define MPI_Group_intersection		PMPI_Group_intersection 
#define MPI_Group_range_excl		PMPI_Group_range_excl 
#define MPI_Group_range_incl		PMPI_Group_range_incl 
#define MPI_Group_rank			PMPI_Group_rank 
#define MPI_Group_size			PMPI_Group_size 
#define MPI_Group_translate_ranks	PMPI_Group_translate_ranks 
#define MPI_Group_union			PMPI_Group_union 
#define MPI_Ibsend			PMPI_Ibsend 
#define MPI_Info_create 		PMPI_Info_create 
#define MPI_Info_delete			PMPI_Info_delete
#define MPI_Info_dup 			PMPI_Info_dup 
#define MPI_Info_free 			PMPI_Info_free 
#define MPI_Info_get 			PMPI_Info_get 
#define MPI_Info_get_nkeys 		PMPI_Info_get_nkeys 
#define MPI_Info_get_nthkey 		PMPI_Info_get_nthkey 
#define MPI_Info_get_valuelen 		PMPI_Info_get_valuelen 
#define MPI_Info_set 			PMPI_Info_set 
#define MPI_Init			PMPI_Init 
#define MPI_Initialized			PMPI_Initialized 
#define MPI_Intercomm_create		PMPI_Intercomm_create
#define MPI_Intercomm_merge		PMPI_Intercomm_merge
#define MPI_Iprobe			PMPI_Iprobe 
#define MPI_Irecv			PMPI_Irecv 
#define MPI_Irsend			PMPI_Irsend 
#define MPI_Isend			PMPI_Isend 
#define MPI_Issend			PMPI_Issend 
#define MPI_Keyval_create		PMPI_Keyval_create 
#define MPI_Keyval_free			PMPI_Keyval_free 
#define MPI_Name_get			PMPI_Name_get
#define MPI_Name_publish		PMPI_Name_publish
#define MPI_Name_unpublish		PMPI_Name_unpublish
#define MPI_Op_create			PMPI_Op_create 
#define MPI_Op_free			PMPI_Op_free 
#define MPI_Pack			PMPI_Pack 
#define MPI_Pack_size			PMPI_Pack_size 
#define MPI_Pcontrol			PMPI_Pcontrol 
#define MPI_Port_open			PMPI_Port_open
#define MPI_Port_close			PMPI_Port_close
#define MPI_Probe			PMPI_Probe 
#define MPI_Recv			PMPI_Recv 
#define MPI_Recv_init			PMPI_Recv_init 
#define MPI_Reduce			PMPI_Reduce 
#define MPI_Reduce_scatter		PMPI_Reduce_scatter 
#define MPI_Request_free		PMPI_Request_free 
#define MPI_Rsend			PMPI_Rsend 
#define MPI_Rsend_init			PMPI_Rsend_init 
#define MPI_Scan			PMPI_Scan 
#define MPI_Scatter			PMPI_Scatter 
#define MPI_Scatterv			PMPI_Scatterv 
#define MPI_Send			PMPI_Send 
#define MPI_Send_init			PMPI_Send_init 
#define MPI_Sendrecv			PMPI_Sendrecv 
#define MPI_Sendrecv_replace		PMPI_Sendrecv_replace 
#define MPI_Spawn			PMPI_Spawn
#define MPI_Spawn_multiple		PMPI_Spawn_multiple
#define MPI_Ssend			PMPI_Ssend 
#define MPI_Ssend_init			PMPI_Ssend_init 
#define MPI_Start			PMPI_Start 
#define MPI_Startall			PMPI_Startall
#define MPI_Test			PMPI_Test 
#define MPI_Test_cancelled		PMPI_Test_cancelled
#define MPI_Testall			PMPI_Testall 
#define MPI_Testany			PMPI_Testany 
#define MPI_Testsome			PMPI_Testsome 
#define MPI_Topo_test			PMPI_Topo_test 
#define MPI_Type_commit			PMPI_Type_commit 
#define MPI_Type_contiguous		PMPI_Type_contiguous 
#define MPI_Type_extent			PMPI_Type_extent 
#define MPI_Type_free			PMPI_Type_free 
#define MPI_Type_hindexed		PMPI_Type_hindexed 
#define MPI_Type_hvector		PMPI_Type_hvector 
#define MPI_Type_indexed		PMPI_Type_indexed 
#define MPI_Type_lb			PMPI_Type_lb 
#define MPI_Type_size			PMPI_Type_size 
#define MPI_Type_struct			PMPI_Type_struct 
#define MPI_Type_ub			PMPI_Type_ub 
#define MPI_Type_vector			PMPI_Type_vector 
#define MPI_Unpack			PMPI_Unpack 
#define MPI_Wait			PMPI_Wait 
#define MPI_Waitall			PMPI_Waitall 
#define MPI_Waitany			PMPI_Waitany 
#define MPI_Waitsome			PMPI_Waitsome 
#define MPI_Wtick			PMPI_Wtick 
#define MPI_Wtime			PMPI_Wtime 

#define MPIL_Spawn			PMPIL_Spawn

#endif	/* PROFILELIB */

/*
 * prototypes of LAM MPI library internal functions
 */
#ifndef __ARGS
#if __STDC__ || defined(c_plusplus) || defined(__cplusplus)
#define __ARGS(a)	a
#else
#define __ARGS(a)	()
#endif
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern void	lam_band __ARGS((void *, void *, int *, MPI_Datatype *));
extern void	lam_bkerr __ARGS((int, int *, int *, int *));
extern void	lam_bor __ARGS((void *, void *, int *, MPI_Datatype *));
extern int	lam_bufattach __ARGS((void *, int));
extern int	lam_bufdetach __ARGS((void *, int *));
extern int	lam_buffreebsend_ __ARGS((MPI_Request));
extern int	lam_bufinit __ARGS((MPI_Request));
extern int	lam_bufinitbsend_ __ARGS((MPI_Request));
extern void	lam_bxor __ARGS((void *, void *, int *, MPI_Datatype *));
extern int	lam_chkarg __ARGS((void *, int, MPI_Datatype,
				int, int, MPI_Comm));
extern int	lam_clocksync __ARGS((int, struct _gps *, float8 *));
extern int	lam_coll2pt __ARGS((int));
extern void	lam_commfault __ARGS((int));
extern int	lam_comm_free __ARGS((MPI_Comm));
extern int	lam_comm_new __ARGS((int, MPI_Group, MPI_Group,
				int, MPI_Comm *));
extern int	lam_cubedim __ARGS((int));
extern int	lam_delkey __ARGS((MPI_Comm, int));
extern void	lam_dtalign __ARGS((MPI_Datatype));
extern void	lam_dtblock __ARGS((MPI_Datatype, MPI_Datatype, int, int));
extern int	lam_dtbuffer __ARGS((MPI_Datatype, int, char **, char **));
extern void	lam_dtcpy __ARGS((char *, char *, int, MPI_Datatype));
extern int	lam_dtsndrcv __ARGS((void *, int, MPI_Datatype,
				void *, int, MPI_Datatype, int, MPI_Comm));
extern void	lam_emptystat __ARGS((MPI_Status *));
extern void	lam_errfatal __ARGS((MPI_Comm *, int *));
extern int	lam_errfunc __ARGS((MPI_Comm, int, int));
extern void	lam_errreturn __ARGS((MPI_Comm *, int *));
extern int	lam_finalized __ARGS((void));
extern int	lam_freekey __ARGS((int));
extern struct _attrkey *lam_getattr __ARGS((int));
extern int	lam_getcid __ARGS((void));
extern int	lam_getfunc __ARGS((void));
extern struct _attr *lam_getkey __ARGS((MPI_Comm, int));
extern void	lam_getparam __ARGS((int *, int*));
extern int	lam_hibit __ARGS((int, int));
extern int	lam_inited __ARGS((void));
extern void	lam_initerr __ARGS((void));
extern int	lam_initfault __ARGS((void));
extern int	lam_isend __ARGS((void *, int, MPI_Datatype, int,
				int, MPI_Comm, MPI_Request *, int));
extern int	lam_isendger_ __ARGS((MPI_Request *));
extern void	lam_land __ARGS((void *, void *, int *, MPI_Datatype *));
extern int	lam_linit __ARGS((char *, int *, int *, int *, 
				struct _gps **, int *));
extern void	lam_lor __ARGS((void *, void *, int *, MPI_Datatype *));
extern void	lam_lxor __ARGS((void *, void *, int *, MPI_Datatype *));
extern void	lam_max __ARGS((void *, void *, int *, MPI_Datatype *));
extern void	lam_maxloc __ARGS((void *, void *, int *, MPI_Datatype *));
extern void	lam_min __ARGS((void *, void *, int *, MPI_Datatype *));
extern void	lam_minloc __ARGS((void *, void *, int *, MPI_Datatype *));
extern void	lam_mkcoll __ARGS((MPI_Comm));
extern int	lam_mkerr __ARGS((int, int));
extern int	lam_mkkey __ARGS((void));
extern void	lam_mkpt __ARGS((MPI_Comm));
extern struct _proc *lam_nextproc __ARGS((void));
extern int	lam_nprocs __ARGS((void));
extern void	lam_nukecids __ARGS((void));
extern void	lam_nukefunc __ARGS((void));
extern void	lam_nukekeys __ARGS((void));
extern void	lam_nukeprocs __ARGS((void));
extern void	lam_nuketrace __ARGS((void));
extern int	lam_pack __ARGS((char *, int, MPI_Datatype, char *, int));
extern int	lam_port_close __ARGS((int));
extern int	lam_port_is_open __ARGS((int));
extern int	lam_port_open __ARGS((void));
extern struct _proc *lam_procadd __ARGS((struct _gps *));
extern int	lam_proccmp __ARGS((struct _proc *, struct _proc *));
extern struct _proc *lam_procfind __ARGS((struct _gps *));
extern int	lam_procfree __ARGS((struct _proc *));
extern int	lam_procrm __ARGS((struct _proc *));
extern void	lam_prod __ARGS((void *, void *, int *, MPI_Datatype *));
extern int	lam_pt2coll __ARGS((int));
extern int	lam_putkey __ARGS((MPI_Comm, int));
extern void	lam_resetfunc __ARGS((int));
extern void	lam_rmcid __ARGS((int));
extern int	lam_send __ARGS((void *, int, MPI_Datatype, int, int,
				MPI_Comm, int));
extern void	lam_setcid __ARGS((int));
extern void	lam_setfunc __ARGS((int));
extern void	lam_setparam __ARGS((int, int, int));
extern void	lam_sum __ARGS((void *, void *, int *, MPI_Datatype *));
extern int	lam_test __ARGS((MPI_Request *, int, int *, MPI_Status *));
extern struct _proc *lam_topproc __ARGS((void));
extern void	lam_tr_cffend __ARGS((int, int, MPI_Comm,
				MPI_Datatype, int));
extern void	lam_tr_cffstart __ARGS((int));
extern int	lam_tr_comm __ARGS((MPI_Comm));
extern int	lam_tr_dtype __ARGS((MPI_Datatype));
extern void	lam_tr_dtypefree __ARGS((MPI_Datatype));
extern int	lam_tr_incff __ARGS((void));
extern int	lam_tr_init __ARGS((char *, float8));
extern void	lam_tr_msg __ARGS((int, double, int, int, int, int,
			MPI_Comm, MPI_Datatype, int, int, int, int, int));
extern int	lam_tr_off __ARGS((void));
extern int	lam_tr_on __ARGS((void));
extern void	lam_tr_startall __ARGS((int, MPI_Request *, double,
				double, double));
extern void	lam_tr_sub __ARGS((int, double, int));
extern double	lam_tr_timer __ARGS((void));
extern void	lam_tr_wrapend __ARGS((int));
extern int	lam_tr_wrapstart __ARGS((int));
extern int	lam_type_free __ARGS((MPI_Datatype));
extern int	lam_unpack __ARGS((char *, int, char *, int, MPI_Datatype));
extern void	lam_unsyncsql __ARGS((int, int, int *, int *,
					int *, int *, int *, int *));
extern void	_m2l_ack __ARGS((struct nmsg *));
extern void	_m2l_fill __ARGS((struct _proc *, struct _proc *, int, int,
				struct nmsg *));
extern struct _proc *_m2l_getsource __ARGS((struct nmsg *));
extern int	_m2l_gettag __ARGS((struct nmsg *));
extern void	_m2l_tail __ARGS((struct nmsg *));
extern void	_mpi_req_add __ARGS((MPI_Request));
extern int	_mpi_req_advance __ARGS((void));
extern void	_mpi_req_blkclr __ARGS((void));
extern void	_mpi_req_blkset __ARGS((MPI_Request));
extern int	_mpi_req_build __ARGS((void *, int, MPI_Datatype, int, int,
				MPI_Comm, int, MPI_Request *));
extern int	_mpi_req_destroy __ARGS((MPI_Request *));
extern int	_mpi_req_end __ARGS((MPI_Request));
extern void	_mpi_req_get __ARGS((int, MPI_Request *));
extern int	_mpi_req_getn __ARGS((void));
extern void	_mpi_req_rem __ARGS((MPI_Request));
extern int	_mpi_req_start __ARGS((MPI_Request));
extern int	_rpi_c2c_addprocs __ARGS((void));
extern int	_rpi_c2c_advance __ARGS((MPI_Request, int));
extern int	_rpi_c2c_build __ARGS((MPI_Request));
extern int	_rpi_c2c_destroy __ARGS((MPI_Request));
extern int	_rpi_c2c_finalize __ARGS((struct _proc *));
extern int	_rpi_c2c_init __ARGS((void));
extern int	_rpi_c2c_iprobe __ARGS((MPI_Request));
extern int	_rpi_c2c_start __ARGS((MPI_Request, MPI_Request));
extern int	_rpi_lamd_addprocs __ARGS((void));
extern int	_rpi_lamd_advance __ARGS((MPI_Request, int));
extern int	_rpi_lamd_build __ARGS((MPI_Request));
extern int	_rpi_lamd_destroy __ARGS((MPI_Request));
extern int	_rpi_lamd_finalize __ARGS((struct _proc *));
extern int	_rpi_lamd_init __ARGS((void));
extern int	_rpi_lamd_iprobe __ARGS((MPI_Request));
extern int	_rpi_lamd_start __ARGS((MPI_Request, MPI_Request));

#ifdef __cplusplus
}
#endif

#endif	/* _MPISYS_H */
