/*
 *	Ohio Trollius
 *	Copyright 1995 The Ohio State University
 *	GDB
 *
 *	$Log:	krecv.c,v $
 * Revision 6.1  96/11/23  22:49:16  nevin
 * Ohio Release
 * 
 * Revision 6.0  96/08/30  18:44:06  nevin
 * Ohio Release
 * 
 * Revision 5.2.1.2  96/01/12  21:55:53  gdburns
 * Remove ETIMEDOUT for TCP kernel.
 * Add block time tracing.
 * 
 * Revision 5.2.1.1  94/10/20  12:11:39  gdburns
 * Defer error checking on krecvback().
 * 
 * Revision 5.2  94/08/22  14:01:41  gdburns
 * Ohio Release
 * 
 * Revision 5.1.1.1  94/08/18  11:37:38  gdburns
 * upgrade for new kernel
 * 
 * Revision 5.1  94/05/18  12:45:27  gdburns
 * Ohio Release
 * 
 * Revision 2.3  94/04/22  12:39:30  gdburns
 * Ohio Release
 * 
 *	Function:	- kernel message receive
 */

#include <errno.h>

#include <kio.h>
#include <kreq.h>
#include <portable.h>
#include <terror.h>
#include <typical.h>

/*
 * external variables
 */
extern struct kio_t	_kio;			/* kernel I/O block */

/*
 * external functions
 */
extern int		_cipc_krecvback();	/* ITB/OTB msg recv */
extern int		_cipc_krecvfront();	/* ITB/OTB msg recv */
extern void		_ksig_follow();		/* check signals */

/*
 *	krecv
 *
 *	Function:	- receives a message from another process
 *			- retries kernel request after signal interruption
 *			  or transfer timeout
 *	Accepts:	- kernel message descriptor ptr
 *	Returns:	- 0 or ERROR
 */
int
krecv(pkmsg)

struct kmsg		*pkmsg;			/* message desc. ptr */

{
	int4		r;
/*
 * Loop through possible signal interruptions.
 */
	do {
		if (krecvfront(pkmsg) < 0) {
			return(LAMERROR);
		}

		r = krecvback(pkmsg);

	} while ((r > 0) && ((r & _kio.ki_sigretry) == r));

	if (r > 0) {
		errno = EINTR;
		return(LAMERROR);
	} else if (r < 0) {
		return(LAMERROR);
	}

	return(0);
}

/*
 *	krecvfront
 *
 *	Function:	- request portion of kernel message receive
 *	Accepts:	- kernel message descriptor ptr
 *	Returns:	- kernel-client socket descriptor or ERROR
 */
int
krecvfront(pkmsg)

struct kmsg		*pkmsg;		/* message desc. ptr */

{
	struct kreq	req;		/* kernel request */
/*
 * Formulate the KQRECV request.
 */
	req.kq_req = KQRECV;
	req.kq_index = _kio.ki_index;
	req.kq_msg.k_event = pkmsg->k_event;
	req.kq_msg.k_type = pkmsg->k_type;
	req.kq_msg.k_flags = pkmsg->k_flags;
	req.kq_msg.k_length = pkmsg->k_length;
	req.kq_msg.k_msg = pkmsg->k_msg;

	return(_cipc_krecvfront(&req));
}

/*
 *	krecvback
 *
 *	Function:	- reply portion of kernel message receive
 *	Accepts:	- kernel message descriptor ptr
 *	Returns:	- signal if request was interrupted
 *			- 0 or ERROR or signal
 */
int4
krecvback(pkmsg)

struct kmsg		*pkmsg;

{
	struct kreq	req;		/* kernel request */
	struct kreply	reply;		/* kernel reply */

	req.kq_msg.k_flags = pkmsg->k_flags;
	req.kq_msg.k_msg = pkmsg->k_msg;

	if (_cipc_krecvback(&req, &reply)) {
		return(LAMERROR);
	}

	if (pkmsg->k_flags & KTRACE) {
		_kio.ki_blktime += reply.kr_blktime;
	}

	if (reply.kr_signal) {
		_kio.ki_signal |= reply.kr_signal;
		_ksig_follow();

		if (reply.kr_reply == EINTR) {
			errno = EINTR;
			return(reply.kr_signal);
		}
	}

	if (reply.kr_reply) {
		errno = reply.kr_reply;
		return(LAMERROR);
	}

	pkmsg->k_type = reply.kr_type;
	pkmsg->k_length = reply.kr_length;
	return(0);
}
