/*
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: waitany.c,v 6.1 96/11/23 22:54:55 nevin Rel $
 *
 *	Function:	- wait for first request to finish
 *	Accepts:	- # of requests
 *			- array of requests
 *			- index of completed request (returned)
 *			- status (returned)
 *	Returns:	- MPI_SUCCESS or error code
 *			- in the case of an error return index is set to
 *			- MPI_UNDEFINED if the error is not associated with
 *			- the completed request, e.g. an internal error
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <rpisys.h>

/*
 * external functions
 */
extern void		lam_emptystat();
extern void		lam_initerr();
extern void		lam_resetfunc();
extern void		lam_setfunc();
extern void		lam_tr_msg();
extern void		_mpi_req_blkclr();
extern void		_mpi_req_blkset();
extern int		lam_tr_incff();
extern int		lam_errfunc();
extern int		lam_mkerr();
extern int		_mpi_req_advance();
extern int		_mpi_req_destroy();
extern int		_mpi_req_end();


int
MPI_Waitany(count, reqs, index, stat)

int			count;
MPI_Request		*reqs;
int			*index;
MPI_Status		*stat;

{
	int		i;			/* favourite index */
	int		nactive;		/* # active requests */
	int		err;			/* error code */
	MPI_Comm	comm;			/* communicator */
	MPI_Request	req;			/* request */
	int		fl_trace;		/* do tracing? */
	double		startt;			/* start time */
	double		finisht;		/* finish time */

	lam_initerr();
	lam_setfunc(BLKMPIWAITANY);

	*index = MPI_UNDEFINED;
	
	if (count < 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITANY, 
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if ((count > 0) && (reqs == 0)) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITANY,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if ((index == 0) || (stat == 0)) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIWAITANY,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if ((fl_trace = LAM_TRACE_TOP())) {
		startt = MPI_Wtime();
		_kio.ki_blktime = 0.0;
	}

	lam_emptystat(stat);
/*
 * While no request is done.
 */
	while (1) {
/*
 * Loop over the requests.
 */
		_mpi_req_blkclr();

		for (i = 0, nactive = 0; i < count; ++i) {
			req = reqs[i];
/*
 * Skip inactive requests.
 */
			if ((req == MPI_REQUEST_NULL) ||
					(req->rq_state == LAM_RQSINIT))
					continue;

			if (req->rq_state == LAM_RQSDONE) break;
/*
 * It's an active request.
 */
			++nactive;
			_mpi_req_blkset(req);
		}
/*
 * We found a completed request.
 */
		if (i < count) {
			*index = i;
			break;
		}
/*
 * If no active requests were found, we're done.
 */
		else if (nactive == 0) {
			*index = MPI_UNDEFINED;
			lam_resetfunc(BLKMPIWAITANY);
			return(MPI_SUCCESS);
		}
/*
 * Otherwise block on any and loop again.
 */
		else {
			err = _mpi_req_advance();
			if (err != MPI_SUCCESS) {
				*index = MPI_UNDEFINED;
				return(lam_errfunc(MPI_COMM_WORLD,
						BLKMPIWAITANY, err));
			}
		}
	}
/*
 * Finish the completed request.
 */
	_mpi_req_rem(req);

	err = _mpi_req_end(req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIWAITANY, err));
	}

	*stat = req->rq_status;
	comm = req->rq_comm;
/*
 * Generate a run time trace except in the case of cancelled request or
 * a request where the peer is MPI_PROC_NULL.
 */
	if (fl_trace && (req->rq_rank != MPI_PROC_NULL)
			&& !(req->rq_flags & LAM_RQFCANCEL)) {
		finisht = MPI_Wtime();

		lam_tr_msg(
			(req->rq_type == LAM_RQIRECV) ? TRTINPUT : TRTNOIO,
			startt, LAM_S2US(finisht - startt - _kio.ki_blktime),
			LAM_S2US(_kio.ki_blktime), req->rq_rank, req->rq_tag,
			req->rq_comm, req->rq_dtype, req->rq_count,
			stat->MPI_SOURCE, stat->MPI_TAG, req->rq_seq,
			req->rq_type);
	}
/*
 * Reset persistent requests.
 * Destroy requests that are not persistent.
 */
	if (req->rq_marks & LAM_RQFPERSIST) {
		req->rq_state = LAM_RQSINIT;
	} else {
		err = _mpi_req_destroy(&reqs[i]);
		if (err != MPI_SUCCESS) {
			return(lam_errfunc(MPI_COMM_WORLD,
					BLKMPIWAITANY, err));
		}
	}

	lam_resetfunc(BLKMPIWAITANY);

	if (stat->MPI_ERROR != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWAITANY, stat->MPI_ERROR));
	}

	return(MPI_SUCCESS);
}
