/*
 * GSDPLT.C - domain mesh plot routines
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

#define SAVE_SEG(jn1, jn2, nn)                                               \
    x1 = rx[jn1];                                                            \
    x2 = rx[jn2];                                                            \
    y1 = ry[jn1];                                                            \
    y2 = ry[jn2];                                                            \
    z1 = rz[jn1];                                                            \
    z2 = rz[jn2];                                                            \
    if ((xmn <= x1) && (x1 <= xmx) &&                                        \
        (xmn <= x2) && (x2 <= xmx) &&                                        \
        (ymn <= y1) && (y1 <= ymx) &&                                        \
	(ymn <= y2) && (y2 <= ymx) &&                                        \
	(zmn <= z1) && (z1 <= zmx) &&                                        \
	(zmn <= z2) && (z2 <= zmx))                                          \
       {tx[nn] = x1;                                                         \
        ty[nn] = y1;                                                         \
        tz[nn] = z1;                                                         \
        nn++;                                                                \
        tx[nn] = x2;                                                         \
        ty[nn] = y2;                                                         \
        tz[nn] = z2;                                                         \
        nn++;}

static int
 _cntrg,
 _ptype,
 _scatter;

static double
 _chi,
 _phi,
 _phi_l,
 _theta,
 _theta_l;

static char
 *_palette;

int
 _PG_bnd_color      = 0,
 _PG_bnd_style      = SOLID,
 _PG_ref_mesh       = FALSE,
 _PG_ref_mesh_color = 0;

REAL
 _PG_bnd_width = 0.0;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DOM_1D - draw a 1d domain mesh */

static void _PG_dom_1d(dev, dom, ran)
   PG_device *dev;
   PM_set *dom, *ran;
   {int i, nn, npts;
    REAL xv, xmx, xmn, ymn, ymx;
    REAL **r, *rx, *f;

    npts = dom->n_elements;
    r    = (REAL **) dom->elements;
    rx   = PM_array_real(dom->element_type, r[0], npts, NULL);

    if (ran != NULL)
       {npts = ran->n_elements;
	r    = (REAL **) ran->elements;
        f    = PM_array_real(ran->element_type, r[0], npts, NULL);}
    else
       f = NULL;

/* do work here */
    PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);

    switch (_cntrg)
       {case Z_CENT :
	     nn = npts - 2;

	     xv = rx[0];
	     PG_draw_line(dev, xv, ymn, xv, ymx);

	     for (i = 1; i < nn; i++)
	         {xv = 0.5*(rx[i] + rx[i+1]);
		  PG_draw_line(dev, xv, ymn, xv, ymx);};

	     xv = rx[npts-1];
	     PG_draw_line(dev, xv, ymn, xv, ymx);

	     break;

	case N_CENT :
	     for (i = 0; i < npts; i++)
	         {xv = rx[i];
		  PG_draw_line(dev, xv, ymn, xv, ymx);};
	     break;};

    SFREE(rx);
    SFREE(f);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DOM_LR_2D - draw a 2d logically rectangular domain mesh */

static void _PG_dom_lr_2d(dev, rx, ry, cnnct, alist)
   PG_device *dev;
   REAL *rx, *ry;
   byte *cnnct;
   pcons *alist;
   {int i, imp, k, l, km, lm, i1, i2, n, nmap;
    int *maxes, kmax, lmax, eflag, refl;
    char *emap;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    n     = kmax*lmax;
    nmap  = (kmax - 1) * (lmax - 1);

    refl = 1;
    LR_MAPPING_INFO(alist, nmap);

    PM_CHECK_EMAP(alist, nmap, eflag, emap);

/* This isn't the most efficient algorithm, but it's simple and it works! */
    lm = lmax - 1;
    km = kmax - 1;
    for (l = 0; l < lm; l++)
        for (k = 0; k < km; k++)
	    {i   = l*kmax + k;
             imp = l*km + k;
	     if (emap[imp] == 0)
	        continue;

	     i1 = i;
	     i2 = i + 1;
	     PG_draw_line(dev, rx[i1], refl*ry[i1],
			  rx[i2], refl*ry[i2]);
                
	     i1 = i + 1;
	     i2 = i + 1 + kmax;
	     PG_draw_line(dev, rx[i1], refl*ry[i1],
			  rx[i2], refl*ry[i2]);

	     i1 = i + kmax + 1;
	     i2 = i + kmax;
	     PG_draw_line(dev, rx[i1], refl*ry[i1],
			  rx[i2], refl*ry[i2]);
                
	     i1 = i + kmax;
	     i2 = i;
	     PG_draw_line(dev, rx[i1], refl*ry[i1],
			  rx[i2], refl*ry[i2]);};

    if (eflag)
	SFREE(emap);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DOM_AC_2D - draw a 2d arbitrarily connected domain mesh */

static void _PG_dom_ac_2d(dev, rx, ry, cnnct, alist)
   PG_device *dev;
   REAL *rx, *ry;
   byte *cnnct;
   pcons *alist;
   {int nd,is, os, iz, oz, is1, is2, in1, in2;
    int *nc, nz, *np, nzp, nsp;
    long zt[2];
    long **cells, *zones, *sides;
    PM_mesh_topology *mt;

    mt = (PM_mesh_topology *) cnnct;

    nd    = mt->n_dimensions;
    nc    = mt->n_cells;
    np    = mt->n_bound_params;
    cells = mt->boundaries;
	
    switch (nd)
       {case 1 :
	     zt[0] = 0;
	     zt[1] = nc[1] - 1;
	     zones = zt;
	     nzp   = 1;
	     nz    = 1;
	     break;

	default :
        case 2  :
	     zones = cells[2];
	     nzp   = np[2];
	     nz    = nc[2];
	     break;};

    nsp   = np[1];
    sides = cells[1];

    for (iz = 0; iz < nz; iz++)
        {oz  = iz*nzp;
	 is1 = zones[oz];
	 is2 = zones[oz+1];

	 for (is = is1; is <= is2; is++)
	     {os  = is*nsp;
	      in1 = sides[os];
	      in2 = sides[os+1];

              PG_draw_line(dev, rx[in1], ry[in1], rx[in2], ry[in2]);};};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DOM_2D - draw a 2d domain mesh */

static void _PG_dom_2d(dev, dom, ran)
   PG_device *dev;
   PM_set *dom, *ran;
   {int npts;
    REAL **r, *rx, *ry, *f;
    byte *cnnct;
    pcons *alst;

    npts = dom->n_elements;
    alst = (pcons *) dom->info;
    r    = (REAL **) dom->elements;

    rx = PM_array_real(dom->element_type, r[0], npts, NULL);
    ry = PM_array_real(dom->element_type, r[1], npts, NULL);

    if (ran != NULL)
       {npts = ran->n_elements;
	r    = (REAL **) ran->elements;
        f    = PM_array_real(ran->element_type, r[0], npts, NULL);}
    else
       f = NULL;

    if (dom->topology == NULL)
       {cnnct = (byte *) dom->max_index;
        _PG_dom_lr_2d(dev, rx, ry, cnnct, alst);}

    else if (strcmp(dom->topology_type, PM_MESH_TOPOLOGY_P_S) == 0)
       {cnnct = dom->topology;
        _PG_dom_ac_2d(dev, rx, ry, cnnct, alst);};

    SFREE(rx);
    SFREE(ry);
    SFREE(f);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DOM_LR_3D - draw a 3d logically rectangular domain mesh */

static void _PG_dom_lr_3d(dev, rx, ry, rz, dextr, cnnct, alist)
   PG_device *dev;
   REAL *rx, *ry, *rz, *dextr;
   byte *cnnct;
   pcons *alist;
   {int k, l, m, i1, i2, n, np, ns;
    int *maxes, kmax, lmax, mmax, eflag, stride;
    char *emap;
    double xmn, xmx, ymn, ymx, zmn, zmx;
    double x1, x2, y1, y2, z1, z2;
    REAL *tx, *ty, *tz;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];
    mmax  = maxes[2];
    n     = kmax*lmax*mmax;

    np = 6*n - 2*(lmax*mmax + kmax*mmax + kmax*lmax);

    tx = FMAKE_N(REAL, np, "_PG_DOM_LR_3D:tx");
    ty = FMAKE_N(REAL, np, "_PG_DOM_LR_3D:ty");
    tz = FMAKE_N(REAL, np, "_PG_DOM_LR_3D:tz");

    emap  = NULL;
    SC_assoc_info(alist,
		  "EXISTENCE", &emap,
		  NULL);

    eflag = (emap == NULL);
    if (eflag)
       {emap = FMAKE_N(char, n, "_PG_DOM_LR_3D:emap");
	memset(emap, 1, n);}
    else
       PM_CHECK_EMAP(alist, n, eflag, emap);

    xmn = dextr[0];
    xmx = dextr[1];
    ymn = dextr[2];
    ymx = dextr[3];
    zmn = dextr[4];
    zmx = dextr[5];

    PG_ADJUST_LIMITS_3D(xmn, xmx, ymn, ymx, zmn, zmx);

    np = 0;

/* lines in the k direction */
    stride = 1;
    for (m = 1; m <= mmax; m++)
        for (l = 1; l <= lmax; l++)
            for (k = 1; k < kmax; k++)
	        {i1 = ((m - 1)*lmax + (l - 1))*kmax + k - 1;
		 i2 = i1 + stride;
		 SAVE_SEG(i1, i2, np);};

/* lines in the l direction */
    stride *= kmax;
    for (m = 1; m <= mmax; m++)
        for (k = 1; k <= kmax; k++)
	    for (l = 1; l < lmax; l++)
	        {i1 = ((m - 1)*lmax + (l - 1))*kmax + k - 1;
		 i2 = i1 + stride;
		 SAVE_SEG(i1, i2, np);};
                
/* lines in the m direction */
    stride *= lmax;
    for (l = 1; l <= lmax; l++)
        for (k = 1; k <= kmax; k++)
	    for (m = 1; m < mmax; m++)
	        {i1 = ((m - 1)*lmax + (l - 1))*kmax + k - 1;
		 i2 = i1 + stride;
		 SAVE_SEG(i1, i2, np);};

    if (eflag)
       SFREE(emap);

    ns = np >> 1;

    PG_draw_disjoint_polyline_3(dev, tx, ty, tz,
				_theta, _phi, _chi, ns, TRUE, FALSE);

/* GOTCHA: the normalization was being done this way for quite a while
 *         but it is wrong for recent test cases - SAB
 *
				_theta, _phi, _chi, ns, TRUE, TRUE);
 */

    SFREE(tx);
    SFREE(ty);
    SFREE(tz);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DOM_AC_3D - draw a 3d arbitrarily connected domain mesh */

static void _PG_dom_ac_3d(dev, rx, ry, rz, dextr, cnnct, alist)
   PG_device *dev;
   REAL *rx, *ry, *rz, *dextr;
   byte *cnnct;
   pcons *alist;
   {int jz, jf, je, jf1, jf2, je1, je2, jn1, jn2;
    int oz, of, oe, nn, ns, nz, nf, ne;
    int *nc, *np, nzp, nfp, nep;
    long **cells, *zones, *faces, *edges;
    REAL *tx, *ty, *tz;
    double xmn, xmx, ymn, ymx, zmn, zmx;
    double x1, x2, y1, y2, z1, z2;
    PM_mesh_topology *mt;

    mt = (PM_mesh_topology *) cnnct;

    cells = mt->boundaries;
    zones = cells[3];
    faces = cells[2];
    edges = cells[1];

    nc = mt->n_cells;
    nz = nc[3];
    nf = nc[2];
    ne = nc[1];

    tx = FMAKE_N(REAL, 2*ne, "_PG_DOM_AC_3D:tx");
    ty = FMAKE_N(REAL, 2*ne, "_PG_DOM_AC_3D:ty");
    tz = FMAKE_N(REAL, 2*ne, "_PG_DOM_AC_3D:tz");

    xmn = dextr[0];
    xmx = dextr[1];
    ymn = dextr[2];
    ymx = dextr[3];
    zmn = dextr[4];
    zmx = dextr[5];

    PG_ADJUST_LIMITS_3D(xmn, xmx, ymn, ymx, zmn, zmx);

    np  = mt->n_bound_params;
    nzp = np[3];
    nfp = np[2];
    nep = np[1];
    nn  = 0;
    switch (mt->n_dimensions)
        {case 3 :
	      for (jz = 0; jz < nz; jz++)
		  {oz  = jz*nzp;
		   jf1 = zones[oz];
		   jf2 = zones[oz+1];
		   for (jf = jf1; jf <= jf2; jf++)
		       {of  = jf*nfp;
			je1 = faces[of];
			je2 = faces[of+1];
			for (je = je1; je <= je2; je++)
			    {oe  = je*nep;
			     jn1 = edges[oe];
			     jn2 = edges[oe+1];
			     SAVE_SEG(jn1, jn2, nn);};};};
	      break;

	 case 2 :
	      for (jf = 0; jf < nf; jf++)
		  {of  = jf*nfp;
		   je1 = faces[of];
		   je2 = faces[of+1];
		   for (je = je1; je <= je2; je++)
		       {oe  = je*nep;
			jn1 = edges[oe];
			jn2 = edges[oe+1];
			SAVE_SEG(jn1, jn2, nn);};};
	      break;

	 case 1 :
	      for (je = 0; je < ne; je++)
		  {oe  = je*nep;
		   jn1 = edges[oe];
		   jn2 = edges[oe+1];
		   SAVE_SEG(jn1, jn2, nn);};
	      break;};

    ns = nn >> 1;

    dev->autodomain = FALSE;

    PG_draw_disjoint_polyline_3(dev, tx, ty, tz,
				_theta, _phi, _chi, 
				ns, dev->autodomain, FALSE);

    SFREE(tx);
    SFREE(ty);
    SFREE(tz);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DOM_AC_3D_SHADED - draw shaded faces of a 3d domain mesh */

static void _PG_dom_ac_3d_shaded(dev, rx, ry, rz, dextr, cnnct, alist)
   PG_device *dev;
   REAL *rx, *ry, *rz, *dextr;
   byte *cnnct;
   pcons *alist;
   {int i, jn1, jn2, jn3;
    int nt, color, ok;
    int area_weighted;
    REAL px[20], py[20], ext[2], v;
    REAL *tx, *ty, *tz;
    double ctl, stl, cpl, spl;
    double ix, iy, iz;
    double x1, x2, y1, y2, z1, z2, x3, y3, z3;
    double dx1, dy1, dz1, dx2, dy2, dz2, s1, s2, nrm;
    double s3x, s3y, s3z;

    area_weighted = FALSE;

    ok = PG_hls_remove(PM_MESH_TOPOLOGY_S,
		       rx, ry, rz, dextr, NULL,
		       _theta, _phi, _chi, cnnct, alist,
		       &tx, &ty, &tz, NULL, &nt);
    if (!ok)
       return;

    ctl = cos(_theta_l);
    cpl = cos(_phi_l);
    stl = sin(_theta_l);
    spl = sin(_phi_l);

/* NOTE: we rotate by phi from negative y axis! */
    ix  =  stl*spl;
    iy  = -stl*cpl;
    iz  = ctl;

    ext[0] = 0.0;
    ext[1] = 1.0;
    for (i = 0; i < nt; i++)
        {jn1 = i*3;
         jn2 = jn1 + 1;
         jn3 = jn2 + 1;
         x1  = px[0] = px[3] = tx[jn1];
	 y1  = py[0] = py[3] = ty[jn1];
	 z1  = tz[jn1];
	 x2  = px[1] = tx[jn2];
	 y2  = py[1] = ty[jn2];
	 z2  = tz[jn2];
	 x3  = px[2] = tx[jn3];
	 y3  = py[2] = ty[jn3];
	 z3  = tz[jn3];
	 dx1 = x2 - x1;
	 dy1 = y2 - y1;
	 dz1 = z2 - z1;
	 dx2 = x3 - x2;
	 dy2 = y3 - y2;
	 dz2 = z3 - z2;
	 s3x = dy1*dz2 - dz1*dy2;
	 s3y = dz1*dx2 - dx1*dz2;
	 s3z = dx1*dy2 - dy1*dx2;
	 if (area_weighted)
	    {s1  = dx1*dx1 + dy1*dy1 + dz1*dz1;
	     s2  = dx2*dx2 + dy2*dy2 + dz2*dz2;
	     nrm = 1.0/sqrt(s1*s2);}
	 else
	    nrm = 1.0/sqrt(s3x*s3x + s3y*s3y + s3z*s3z + SMALL);

	 v = nrm*(s3x*ix + s3y*iy + s3z*iz);
	 v = max(v, 0.0);

	 color = PG_select_color(dev, 1, &v, ext);

	 PG_set_color_fill(dev, color, TRUE);
	 PG_shade_poly(dev, px, py, 4);

	 if (dev->draw_fill_bound)
	    {PG_set_line_color(dev, dev->WHITE);
	     PG_draw_polyline(dev, px, py, 4, TRUE);};};

    SFREE(tx);
    SFREE(ty);
    SFREE(tz);
/*
    PG_set_line_color(dev, dev->WHITE);
    _PG_dom_ac_3d(dev, rx, ry, rz, dextr, cnnct, alist);
*/
    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DOM_0D_3D - do a 3d scatter plot */

static void _PG_dom_0d_3d(dev, npts, rx, ry, rz, dextr, alst)
   PG_device *dev;
   long npts;
   REAL *rx, *ry, *rz, *dextr;
   pcons *alst;
   {REAL *pms;
    REAL *tx, *ty, *tz, ux, uy, uz;
    REAL xmn, xmx, ymn, ymx, zmn, zmx;
    int i, n, *pmk, mrk;

    pmk = NULL;
    pms = NULL;
    SC_assoc_info(alst,
		  "MARKER-INDEX", &pmk,
		  "MARKER-SCALE", &pms,
		  NULL);

    mrk = (pmk != NULL) ? *pmk : 0;

    dev->marker_scale = (pms != NULL) ? *pms : dev->marker_scale;

    tx = FMAKE_N(REAL, npts, "_PG_DOM_0D_3D:tx");
    ty = FMAKE_N(REAL, npts, "_PG_DOM_0D_3D:ty");
    tz = FMAKE_N(REAL, npts, "_PG_DOM_0D_3D:tz");

    PG_rotate_3d_WC(rx, ry, rz,
		    dextr[0], dextr[1],
		    dextr[2], dextr[3],
		    dextr[4], dextr[5],
                    _theta, _phi, _chi, tx, ty, tz, npts, FALSE);

/* get rid of points outside the domain limits */
    xmn = dextr[0];
    xmx = dextr[1];
    ymn = dextr[2];
    ymx = dextr[3];
    zmn = dextr[4];
    zmx = dextr[5];
    for (i = 0, n = 0; i < npts; i++)
        {ux = rx[i];
         uy = ry[i];
         uz = rz[i];
         if ((xmn <= ux) && (ux <= xmx) &&
             (ymn <= uy) && (uy <= ymx) &&
             (zmn <= uz) && (uz <= zmx))
            {tx[n] = tx[i];
             ty[n] = ty[i];
             n++;};};

    PG_draw_markers(dev, n, tx, ty, mrk);

    SFREE(tx);
    SFREE(ty);
    SFREE(tz);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_LABEL_NODES_3 - label the nodes in a mesh plot */

static void PG_label_nodes_3(dev, x, y, z, n, f, type)
   PG_device *dev;
   REAL *x, *y, *z;
   long n;
   byte *f;
   char *type;
   {int i;
    REAL *rx, *ry, *rz;
    REAL xmn, xmx, ymn, ymx, zmn, zmx;
    REAL xv, yv, zv;
    char ltype[MAXLINE], *s;

    PG_make_device_current(dev);

    rx = FMAKE_N(REAL, n, "PG_LABEL_NODES_3:rx");
    ry = FMAKE_N(REAL, n, "PG_LABEL_NODES_3:ry");
    rz = FMAKE_N(REAL, n, "PG_LABEL_NODES_3:rz");

    xmn =  HUGE_REAL;
    ymn =  HUGE_REAL;
    zmn =  HUGE_REAL;
    xmx = -HUGE_REAL;
    ymx = -HUGE_REAL;
    zmx = -HUGE_REAL;
    for (i = 0; i < n; i++)
        {xv = x[i];
         yv = y[i];
         zv = z[i];
         xmn = min(xmn, xv);
         xmx = max(xmx, xv);
         ymn = min(ymn, yv);
         ymx = max(ymx, yv);
         zmn = min(zmn, zv);
         zmx = max(zmx, zv);};

    PG_rotate_3d_WC(x, y, z, xmn, xmx, ymn, ymx, zmn, zmx,
                    _theta, _phi, _chi, rx, ry, rz, n,
		    dev->autodomain);

    strcpy(ltype, type);
    SC_strtok(ltype, " *", s);
    if ((strcmp(ltype, SC_INTEGER_S) == 0) ||
	(strcmp(ltype, SC_LONG_S) == 0) ||
	(strcmp(ltype, SC_SHORT_S) == 0))
       {int *fi;
	char s[MAXLINE];

	fi = NULL;
	CONVERT(SC_INTEGER_S, &fi, ltype, f, n, FALSE);
	for (i = 0; i < n; i++)
	    {sprintf(s, "%d", fi[i]);
	     PG_write_WC(dev, rx[i], ry[i], s);};

	SFREE(fi);}

    else if ((strcmp(ltype, SC_DOUBLE_S) == 0) ||
	     (strcmp(ltype, SC_FLOAT_S) == 0) ||
	     (strcmp(ltype, SC_REAL_S) == 0))
       {REAL *fi;
	char s[MAXLINE];

	fi = PM_array_real(type, f, n, NULL);
	for (i = 0; i < n; i++)
	    {sprintf(s, _PG_text_format, fi[i]);
	     PG_write_WC(dev, rx[i], ry[i], s);};

	SFREE(fi);}

    else if (strcmp(ltype, SC_CHAR_S) == 0)
       {char t[2], *fi;

	t[1] = '\0';
	fi   = f;
	for (i = 0; i < n; i++)
	    {t[0] = fi[i];
	     PG_write_WC(dev, rx[i], ry[i], t);};

	SFREE(fi);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DRAW_3D_DOMAIN - draw a single domain set */

static void _PG_draw_3d_domain(dev, dom, ran, dextr, rx, ry, rz)
   PG_device *dev;
   PM_set *dom, *ran;
   REAL *dextr, *rx, *ry, *rz;
   {long ne;
    byte *cnnct;
    pcons *alst;

/* get the options for this mesh */
    alst = NULL;
    if (dom->info_type != NULL)
       if (strcmp(dom->info_type, SC_PCONS_P_S) == 0)
	  alst = (pcons *) dom->info;

    ne = dom->n_elements;

    if (_scatter || (dom->dimension == 0))
       _PG_dom_0d_3d(dev, ne, rx, ry, rz, dextr, alst);

    else if (dom->topology == NULL)
       {cnnct = (byte *) dom->max_index;
        _PG_dom_lr_3d(dev, rx, ry, rz, dextr, cnnct, alst);}

    else if (strcmp(dom->topology_type, PM_MESH_TOPOLOGY_P_S) == 0)
       {cnnct = dom->topology;
        if (_ptype == PLOT_SURFACE)
	   {PG_set_palette(dev, _palette);
	    _PG_dom_ac_3d_shaded(dev, rx, ry, rz, dextr, cnnct, alst);}
        else
           _PG_dom_ac_3d(dev, rx, ry, rz, dextr, cnnct, alst);};

    if ((_ptype == PLOT_MESH) && (ran != NULL))
       {byte **r;

        r = (byte **) ran->elements;
	PG_label_nodes_3(dev, rx, ry, rz, ne, r[0],
			 ran->element_type);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DOM_3D - draw a 3d domain mesh */

static void _PG_dom_3d(dev, dom, ran)
   PG_device *dev;
   PM_set *dom, *ran;
   {int npts;
    REAL extr[6], *dextr;
    REAL **r, *rx, *ry, *rz;
    REAL p1[9], p2[9], p3[9];
    PM_set *p;

    if (dev->autodomain == TRUE)
       dextr = PM_array_real(dom->element_type, dom->extrema, 6, extr);
    else
       dextr = PM_get_limits(dom);

    for (p = dom; p != NULL; p = p->next)
        {npts = p->n_elements;
	 r    = (REAL **) p->elements;

	 rx = PM_array_real(p->element_type, r[0], npts, NULL);
	 ry = PM_array_real(p->element_type, r[1], npts, NULL);
	 rz = PM_array_real(p->element_type, r[2], npts, NULL);

         if (p == dom)
	    PG_axis_3d_limits(dev, rx, ry, rz, npts,
			      _theta, _phi, _chi,
			      FALSE, TRUE, TRUE,
			      dextr[0], dextr[1],
			      dextr[2], dextr[3],
			      dextr[4], dextr[5],
			      p1, p2, p3);

	 _PG_draw_3d_domain(dev, p, ran, dextr, rx, ry, rz);

	 SFREE(rx);
	 SFREE(ry);
	 SFREE(rz);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_PICTURE_MESH - setup a window for a mesh rendering
 *                       - NOTE: no drawing of any kind is to be done here
 */

PG_picture_desc *PG_setup_picture_mesh(dev, data, save, clear)
   PG_device *dev;
   PG_graph *data;
   int save, clear;
   {int nde, clr, sty, axfl, clip, change;
    int *pc, *ps, *pcent, *psc;
    char *ppal;
    REAL wid;
    REAL *dpex, *ddex, *pdx;
    REAL *vwprt, *pw, *ptl, *ppl;
    PM_set *dom;
    PG_picture_desc *pd;
    PG_par_rend_info *pri;
    PG_device *dd;
    pcons *alst;

    dom = data->f->domain;

    change = !dev->supress_setup;

    pd = PG_get_rendering_properties(dev, data);

    alst  = pd->alist;
    pc    = NULL;
    ps    = NULL;
    pw    = NULL;
    pcent = NULL;
    ptl   = NULL;
    ppl   = NULL;
    psc   = NULL;
    ppal  = NULL;
    SC_assoc_info(alst,
		  "LINE-COLOR", &pc,
		  "LINE-WIDTH", &pw,
		  "LINE-STYLE", &ps,
		  "THETA-LIGHT", &ptl,
		  "PHI-LIGHT", &ppl,
		  "CENTERING", &pcent,
		  "PALETTE", &ppal,
		  "SCATTER", &psc,
		  NULL);

    pd->palette_fl = FALSE;
    pd->legend_fl  = FALSE;
    pri            = dev->pri;
    if (pri != NULL)
       {dd = pri->dd;
	if (dd != NULL)
	   {dd->pri->alist  = alst;
	    dd->pri->render = PLOT_MESH;};};

    if (change)
       {vwprt = pd->viewport;
	if (vwprt != NULL)
	   PG_set_viewport(dev, vwprt[0], vwprt[1], vwprt[2], vwprt[3]);

/* find the extrema for this frame */
	PG_find_extrema(data, 0.0, &dpex, NULL, &nde, &ddex, NULL, NULL);

/* setup the domain limits */
	pdx = ((dev->autodomain == TRUE) || (dpex == NULL)) ? ddex : dpex;
	PG_set_window(dev, pdx[0], pdx[1], pdx[2], pdx[3]);

	SFREE(ddex);

	axfl = pd->axis_fl;
	if (axfl && (nde == 3))
	   pd->ax_type = CARTESIAN_3D;

	if (nde == 3)
	   {clip = FALSE;
	    dev->autodomain = (dpex == NULL);
	    dev->autorange  = (dpex == NULL);}
	else
	   clip = TRUE;

	PG_set_clipping(dev, clip);};

/* set the file static constants */
    _cntrg   = (pcent == NULL) ? N_CENT : *pcent;
    _theta_l = (ptl == NULL) ? 45.0 : *ptl;
    _phi_l   = (ppl == NULL) ? 45.0 : *ppl;
    _theta_l *= DEG_RAD;
    _phi_l   *= DEG_RAD;
    _palette  = (ppal == NULL) ? dev->current_palette->name : ppal;
    _scatter  = (psc != NULL) ? *psc : _PG_scatter_plot;

    PG_set_palette(dev, "standard");

    clr = (pc == NULL) ? dev->WHITE : *pc;
    wid = (pw == NULL) ? 0.0 : *pw;
    sty = (ps == NULL) ? SOLID : *ps;

    PG_set_color_text(dev, dev->WHITE, TRUE);
    PG_set_line_color(dev, clr);
    PG_set_line_width(dev, wid);
    PG_set_line_style(dev, sty);

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_DOMAIN_PLOT - plot the mesh of the given domain set */

void PG_domain_plot(dev, dom, ran)
   PG_device *dev;
   PM_set *dom, *ran;
   {PM_mapping f;
    PG_graph g, *data;
    PG_picture_desc *pd;

    memset(&f, 0, sizeof(PM_mapping));
    f.domain = dom;
    f.name   = dom->name;

    data = &g;
    memset(data, 0, sizeof(PG_graph));
    g.f         = &f;
    g.rendering = PLOT_MESH;
    g.info_type = SC_PCONS_P_S;
    g.info      = dom->info;

    pd = PG_setup_picture(dev, data, TRUE, TRUE, TRUE);

    _theta = pd->theta;
    _phi   = pd->phi;
    _chi   = pd->chi;

/* plot the domain */
    switch (dom->dimension_elem)
       {case 1  : 
	     _PG_dom_1d(dev, dom, ran);
	     break;

        case 2  :
	     _PG_dom_2d(dev, dom, ran);
	     break;

        case 3 :
	     _PG_dom_3d(dev, dom, ran);
	     break;};

    PG_finish_picture(dev, data, pd);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_MESH_PLOT - plot the domain mesh and maybe the range values */

#ifdef PCC

void PG_mesh_plot(dev, data, va_alist)
   PG_device *dev;
   PG_graph *data;
   va_dcl

#endif

#ifdef ANSI

void PG_mesh_plot(PG_device *dev, PG_graph *data, ...)

#endif

   {PG_graph *g;
    PM_mapping *f;
    PM_set *domain;
    byte *info;
    pcons *mlst, *dlst, *tlst;
    REAL *extr, *dextr;
    int color, *pc;

    for (g = data; g != NULL; g = g->next)
        {for (f = data->f; f != NULL; f = f->next)
	     {domain = f->domain;
	      info   = domain->info;

	      mlst = SC_copy_alist((pcons *) f->map);
	      dlst = SC_copy_alist((pcons *) domain->info);

	      tlst = SC_copy_alist((pcons *) data->info);
	      tlst = SC_append_alist(tlst, mlst);
	      tlst = SC_append_alist(tlst, dlst);

	      if (f == data->f)
		 {extr = PM_get_limits(domain);
		  pc = NULL;
		  SC_assoc_info(tlst, "LINE-COLOR", &pc, NULL);
		  color = (pc == NULL) ? dev->BLUE : *pc;}
	      else
		 {SC_CHANGE_VALUE_ALIST(tlst, int, SC_INTEGER_P_S,
					"DRAW-AXIS", FALSE);
		  SC_CHANGE_VALUE_ALIST(tlst, int, SC_INTEGER_P_S,
					"DRAW-LABEL", FALSE);
		  SC_CHANGE_VALUE_ALIST(tlst, int, SC_INTEGER_P_S,
					"LINE-COLOR", ++color);};

	      domain->info = (byte *) tlst;
	      if (f != data->f)
		 {dextr = FMAKE_N(REAL, 4, "PG_MESH_PLOT:dextr");
		  memcpy(dextr, extr, 4*sizeof(REAL));
		  PM_set_limits(domain, dextr);};

	      PG_domain_plot(dev, domain, f->range);

/* NOTE: because of the way SC_append_alist works we don't need
 *       to free mlst and dlst explicitly
 */
	      SC_free_alist(tlst, 3);

	      domain->info = info;};};

    return;}

/*--------------------------------------------------------------------------*/

/*                   DOMAIN BOUNDARY PLOT ROUTINES                          */

/*--------------------------------------------------------------------------*/

/* _PG_DRAW_BND_LR_2D - draw boundary around 2d LR domain */

static void _PG_draw_bnd_lr_2d(dev, domain)
   PG_device *dev;
   PM_set *domain;
   {int i, k, l, npts, km, lm, kmax, lmax;
    int *maxes;
    REAL x1, y1, x2, y2;
    REAL *rx, *ry, **r;

    npts  = domain->n_elements;
    maxes = domain->max_index;
    r     = (REAL **) domain->elements;

    kmax = maxes[0];
    lmax = maxes[1];
    km   = kmax - 1;
    lm   = lmax - 1;

    rx = PM_array_real(domain->element_type, r[0], npts, NULL);
    ry = PM_array_real(domain->element_type, r[1], npts, NULL);

/* k = 0 side */
    for (l = 0; l < lm; l++)
        {i = l*kmax;
	 x1 = rx[i];
	 y1 = ry[i];
	 x2 = rx[i + kmax];
	 y2 = ry[i + kmax];
	 PG_draw_line(dev, x1, y1, x2, y2);};

/* l = lmax-1 side */
    for (k = 0; k < km; k++)
        {i = lm*kmax + k;
	 x1 = rx[i];
	 y1 = ry[i];
	 x2 = rx[i + 1];
	 y2 = ry[i + 1];
	 PG_draw_line(dev, x1, y1, x2, y2);};

/* k = kmax-1 side */
    for (l = 0; l < lm; l++)
        {i = l*kmax + km;
	 x1 = rx[i];
	 y1 = ry[i];
	 x2 = rx[i + kmax];
	 y2 = ry[i + kmax];
	 PG_draw_line(dev, x1, y1, x2, y2);};

/* l = 0 side */
    for (k = 0; k < km; k++)
        {i = k;
	 x1 = rx[i];
	 y1 = ry[i];
	 x2 = rx[i + 1];
	 y2 = ry[i + 1];
	 PG_draw_line(dev, x1, y1, x2, y2);};

    SFREE(rx);
    SFREE(ry);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_DOMAIN_BOUNDARY - draw the domain boundary of the mapping F
 *                         - if the attribute of the graph DATA demands
 */

void PG_draw_domain_boundary(dev, f)
   PG_device *dev;
   PM_mapping *f;
   {int nd, cl, clr, sty, dbclr, dbsty;
    int *pdbc, *pdbs;
    REAL xmn, xmx, rmn, rmx;
    REAL wid, dbwid, *pdbw;
    REAL extr[6];
    PM_set *domain;
    PG_palette *pl;

    domain = f->domain;
    nd     = domain->dimension;
    pl     = dev->current_palette;

    PG_set_palette(dev, "standard");

    pdbw = NULL;
    pdbc = NULL;
    pdbs = NULL;
    if (domain->info_type != NULL)
       {if (strcmp(domain->info_type, SC_PCONS_P_S) == 0)
	   SC_assoc_info((pcons *) domain->info,
			 "DOMAIN-BORDER-WIDTH", &pdbw,
			 "DOMAIN-BORDER-COLOR", &pdbc,
			 "DOMAIN-BORDER-STYLE", &pdbs,
			 NULL);};

    dbwid = (pdbw == NULL) ? -1.0 : *pdbw;
    dbclr = (pdbc == NULL) ? dev->WHITE : *pdbc;
    dbsty = (pdbs == NULL) ? SOLID : *pdbs;

    if (dbwid > -1.0)
       {PG_get_line_color(dev, &clr);
	PG_get_line_style(dev, &sty);
	PG_get_line_width(dev, &wid);
	PG_get_clipping(dev, &cl);

	PG_set_clipping(dev, TRUE);
	PG_set_line_color(dev, dbclr);
	PG_set_line_style(dev, dbsty);
	PG_set_line_width(dev, dbwid);

	PM_array_real(domain->element_type, domain->extrema,
		      2*nd, extr);

/* GOTCHA: this needs to be reworked for domains other than
 *         rectangular ones
 */
	switch (nd)
	   {case 1 :
		 PG_get_viewport_WC(dev, &xmn, &xmx, &rmn, &rmx);
		 PG_draw_box(dev, extr[0], extr[1], rmn, rmx);
	         break;

	    case 2 :
	         if (strcmp(f->category, PM_LR_S) == 0)
		    _PG_draw_bnd_lr_2d(dev, domain);
		 break;

	    case 3 :
	         break;};

	PG_set_clipping(dev, cl);
	PG_set_line_color(dev, clr);
	PG_set_line_style(dev, sty);
	PG_set_line_width(dev, wid);};

    dev->current_palette = pl;

   return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_REF_MESH - plot the domain mesh and maybe the range values */

void PG_ref_mesh(dev, data, ndims, nvxmn, nvxmx, nvymn, nvymx, vwprt)
   PG_device *dev;
   PG_graph *data;
   int ndims;
   REAL nvxmn, nvxmx, nvymn, nvymx; 
   REAL *vwprt;
   {int save_axis_on, save_labels_on, mshf, clp;
    PM_mapping *f;
    PM_set *domain;
    REAL *pv;
    int *pc;
    pcons *alst;
    
    mshf = data->mesh;
    if (mshf || _PG_ref_mesh)
       {data->mesh = FALSE;

/* set the viewport to incoming values */
	if (vwprt == NULL)
	   {pv = FMAKE_N(REAL, 4, "PG_ref_mesh:pv");
	    pv[0] = nvxmn;
	    pv[1] = nvxmx;
	    pv[2] = nvymn;
	    pv[3] = nvymx;
	    data->info = (byte *) SC_change_alist(data->info, "VIEW-PORT",
						  SC_REAL_P_S, pv);}

/* save some state */
	save_axis_on   = _PG_axis_on;
	save_labels_on = PG_plot_labels;
	PG_get_clipping(dev, &clp);
	PG_set_clipping(dev, TRUE);

/* don't plot axis or labels for reference mesh */
	_PG_axis_on    = FALSE;
	PG_plot_labels = FALSE;

	f      = data->f;
	domain = f->domain;

/* save the line color */
	alst = NULL;
	if ((domain->info_type != NULL) &&
	    (strcmp(domain->info_type, SC_PCONS_P_S) == 0))
	   {alst = (pcons *) domain->info;
	    SC_assoc_info(alst,
			  "LINE-COLOR", &pc,
			  NULL);};

	_PG_ref_mesh_color = _PG_trans_color(dev, _PG_ref_mesh_color);

/* set the line color for the reference mesh */
	SC_CHANGE_VALUE_ALIST(alst, int, SC_INTEGER_S,
			      "LINE-COLOR", _PG_ref_mesh_color);

	domain->info = (byte *) alst;

	PG_mesh_plot(dev, data, ndims);

/* restore the state */
	_PG_axis_on    = save_axis_on;
	PG_plot_labels = save_labels_on;
	PG_set_clipping(dev, clp);

	if (vwprt == NULL)
	   {data->info = (byte *) SC_rem_alist(data->info, "VIEW-PORT");
	    SFREE_N(pv, 4);};

	if (pc == NULL)
	   alst = SC_rem_alist(alst, "LINE-COLOR");
	else
	   alst = SC_change_alist(alst, "LINE-COLOR", "int *", (byte *) pc);

	domain->info = (byte *) alst;
	data->mesh   = mshf;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
