/*
 * GSIMAG.C - image handling routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "pgs.h"
#include "gsrast.h"

/*--------------------------------------------------------------------------*/

/* Note that if z is within range the scaled pixel value is shifted up by
 * two colors accounting for the black and white mandatory in each palette
 */
#define PG_TRANS_IMAGE(_d, _bf, _km, _lm, _n, _nc, _z, _zmn, _zmx, _t, _vn, _vx) \
    {_t *_lz, _lzt, _lzmn, _lzmx;                                            \
     int _i, _cmid;                                                          \
     REAL _sc;                                                               \
     _lz = (_t *) _z;                                                        \
     if (_zmx <= _zmn)                                                       \
        {_lzmn = _vx;                                                        \
         _lzmx = _vn;                                                        \
         for (_i = 0; _i < _n; _i++)                                         \
             {_lzt = _lz[_i];                                                \
              _lzmn = min(_lzmn, _lzt);                                      \
              _lzmx = max(_lzmx, _lzt);};                                    \
         _zmn = _lzmn;                                                       \
         _zmx = _lzmx;};                                                     \
     _sc = (_zmn == _zmx) ? HUGE : 0.9999*_nc/(_zmx - _zmn);                 \
     _cmid  = 0.5*_nc + 2;                                                   \
     for (_i = 0; _i < _n; _i++, _lz++)                                      \
         {if (_sc == HUGE)                                                   \
             _bf[_i] = _cmid;                                                \
          else                                                               \
             _bf[_i] = _sc*(*_lz - _zmn) + 2;};}

/*--------------------------------------------------------------------------*/

int
 _PG_palette_orientation = HORIZONTAL;

REAL
 _PG_palette_width = 0.02;

static REAL
 *_pol_extr;

static char
 *PG_IMAGE_S = "PG_image";

static PG_graph
 *im_temp = NULL;

static int
 ok_change_viewport = TRUE;

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_MAP_TYPE_IMAGE - map the image data from the specified
 *                    - TYPE in Z to UNSIGNED CHAR in BF
 *                    - this also inverts the image in INV is TRUE
 */

static void _PG_map_type_image(dev, type, bf, kmax, lmax, n, nc,
			       z, zmin, zmax)
   PG_device *dev;
   char *type;
   unsigned char *bf;
   int kmax, lmax, n, nc;
   byte *z;
   double zmin, zmax;
   {

/* scale the data into the buffer */
    if (type == NULL)
       memcpy(bf, z, n);

    else if (strcmp(type, SC_STRING_S) == 0)
       {PG_TRANS_IMAGE(dev, bf, kmax, lmax, n, nc, z, zmin, zmax,
		       unsigned char, 0, 255);}

    else if (strcmp(type, SC_DOUBLE_P_S) == 0)
       {PG_TRANS_IMAGE(dev, bf, kmax, lmax, n, nc, z, zmin, zmax,
		       double, -DBL_MAX, DBL_MAX);}

    else if (strcmp(type, SC_FLOAT_P_S) == 0)
       {PG_TRANS_IMAGE(dev, bf, kmax, lmax, n, nc, z, zmin, zmax,
		       float, -FLT_MAX, FLT_MAX);}

    else if (strcmp(type, SC_LONG_P_S) == 0)
       {PG_TRANS_IMAGE(dev, bf, kmax, lmax, n, nc, z, zmin, zmax,
		       long, LONG_MIN, LONG_MAX);}

    else if (strcmp(type, SC_INTEGER_P_S) == 0)
       {PG_TRANS_IMAGE(dev, bf, kmax, lmax, n, nc, z, zmin, zmax,
		       int, INT_MIN, INT_MAX);}

    else if (strcmp(type, SC_SHORT_P_S) == 0)
       {PG_TRANS_IMAGE(dev, bf, kmax, lmax, n, nc, z, zmin, zmax,
		       short, SHRT_MIN, SHRT_MAX);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_INVERT_IMAGE_DATA - flip the image throught the y direction */

void PG_invert_image_data(bf, kx, lx, bpi)
   unsigned char *bf;
   int kx, lx, bpi;
   {int i, l, lm, lh, bc;
    long nb;
    unsigned char *ri, *ro;

    nb = kx*bpi;

    lm = lx - 1;
    lh = lx >> 1;

    for (l = 0; l < lh; l++)
        {ri = bf + l*nb;
	 ro = bf + (lm - l)*nb;
	 for (i = 0; i < nb; i++)
	     {bc    = ri[i];
	      ri[i] = ro[i];
	      ro[i] = bc;};};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_INVERT_IMAGE - flip the image throught the y direction */

void PG_invert_image(im)
   PG_image *im;
   {int kx, lx;
    unsigned char *pb;
    char type[MAXLINE];

    strcpy(type, im->element_type);
    PD_dereference(type);

    kx = im->kmax;
    lx = im->lmax;
    pb = im->buffer;

    PG_invert_image_data(pb, kx, lx, SIZEOF(type));

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_IMAGE_HAND - image plot handler */

static void PG_image_hand(dev, g, fmin, fmax, fnc_zc, fnc_nc)
   PG_device *dev;
   PG_graph *g;
   double fmin, fmax;
   PFByte fnc_zc, fnc_nc;
   {int npts, ndpts, centering, autodomain;
    char *name;
    REAL *f, x[4], xmax, ymax, xmin, ymin;
    PM_mapping *h;
    PM_set *domain, *range;
    byte *cnnct;
    pcons *alst;

    autodomain = dev->autodomain;
    dev->autodomain = FALSE;

    for (h = g->f; h != NULL; h = h->next)
	{PG_get_viewport_WC(dev, &xmin, &xmax, &ymin, &ymax);

	 domain = h->domain;
	 ndpts  = domain->n_elements;

	 PM_array_real(domain->element_type, domain->extrema, 4, x);
	 xmin = x[0];
	 xmax = x[1];
	 ymin = x[2];
	 ymax = x[3];

	 range = h->range;
	 npts  = range->n_elements;

	 f = PM_array_real(range->element_type, DEREF(range->elements), npts, NULL);

/* find the additional mapping information */
	 centering = N_CENT;
	 alst = PM_mapping_info(h,
				"CENTERING", &centering,
				NULL);

	 cnnct = PM_connectivity(h);

         if (h == g->f)
	    name = h->name;
	 else
	    {SC_CHANGE_VALUE_ALIST(alst, int, SC_INTEGER_P_S,
				   "DRAW-AXIS", FALSE);
	     SC_CHANGE_VALUE_ALIST(alst, int, SC_INTEGER_P_S,
				   "DRAW-LABEL", FALSE);
	     SC_CHANGE_VALUE_ALIST(alst, int, SC_INTEGER_P_S,
				   "DRAW-PALETTE", FALSE);};

/* this is done consistently with PG_draw_vector */
	 switch (centering)
	    {case Z_CENT :
                  (*fnc_zc)(dev, name, SC_DOUBLE_S, f,
			    xmin, xmax, ymin, ymax, fmin, fmax, cnnct, alst);
		  break;

             case N_CENT :
                  (*fnc_nc)(dev, name, SC_DOUBLE_S, f,
			    xmin, xmax, ymin, ymax, fmin, fmax, cnnct, alst);
		  break;

             case F_CENT :
             case U_CENT :
             default     : break;};

	 PG_draw_domain_boundary(dev, h);

	 SFREE(f);};

    dev->autodomain = autodomain;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_RENDER_DATA_TYPE - return TRUE iff the specified graph DATA has
 *                     - a mapping for its data and FALSE
 *                     - if it has a cast image
 */

int PG_render_data_type(data)
   PG_graph *data;
   {char *s;

    s = data->use;

    return((s == NULL) || (strcmp(s, PG_IMAGE_S) != 0));}
    
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_IMAGE_PICTURE_INFO - return the picture info that an image might
 *                       - contain
 */

void PG_image_picture_info(data, pnd, pxmn, pxmx, pymn, pymx, pzmn, pzmx)
   PG_graph *data;
   int *pnd;
   REAL *pxmn, *pxmx, *pymn, *pymx, *pzmn, *pzmx;
   {REAL xmn, xmx, ymn, ymx;
    PG_image *im;

    im = (PG_image *) data->f;

    xmn = im->xmin;
    xmx = im->xmax;
    ymn = im->ymin;
    ymx = im->ymax;
    if ((xmn == 0) && (xmx == 0) &&
	(ymn == 0) && (ymx == 0))
       {xmx = im->kmax;
	ymx = im->lmax;};

    *pxmn = xmn;
    *pxmx = xmx;
    *pymn = ymn;
    *pymx = ymx;

    if (pnd != NULL)
       *pnd  = 1;
    if (pzmn != NULL)
       *pzmn = im->zmin;
    if (pzmx != NULL)
       *pzmx = im->zmax;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SETUP_PICTURE_IMAGE - setup a window for a image rendering
 *                        - NOTE: no drawing of any kind is to be done here
 */

PG_picture_desc *PG_setup_picture_image(dev, data, save, clear)
   PG_device *dev;
   PG_graph *data;
   int save, clear;
   {int nde, nre, hvf, change;
    REAL vxmn, vxmx, vymn, vymx;
    REAL *dpex, *ddex, *pdx, *rpex, *rdex, *prx, *vwprt;
    PG_picture_desc *pd;
    PG_device *dd;
    PG_par_rend_info *pri;
    pcons *alst;

    change = !dev->supress_setup;

    pd   = PG_get_rendering_properties(dev, data);

    alst = pd->alist;
    pri  = dev->pri;
    if (pri != NULL)
       {dd = pri->dd;
	if (dd != NULL)
	   {dd->pri->alist  = alst;
	    dd->pri->render = PLOT_IMAGE;};};

    pd->legend_fl = FALSE;

/* check on any special attributes */
    SC_assoc_info(alst,
		  "POLAR-IMAGE", &_pol_extr,
		  NULL);

    if (change)

/* setup the viewport */
       {if (ok_change_viewport)
	   {hvf   = _PG_palette_orientation;
	    vwprt = pd->viewport;
	    if (vwprt != NULL)
	       {vxmn = vwprt[0];
		vxmx = vwprt[1];
		vymn = vwprt[2];
		vymx = vwprt[3];}
	    else
	       {if (hvf == VERTICAL)
		   {vxmn = 0.175;
		    vxmx = 0.725 - _PG_palette_width;
		    vymn = 0.175;
		    vymx = 0.825;}

	        else if (hvf == HORIZONTAL)
		   {vxmn = 0.175;
		    vxmx = 0.85;
		    vymn = 0.23 + _PG_palette_width;
		    vymx = 0.825;};};

	    PG_set_viewport(dev, vxmn, vxmx, vymn, vymx);};

/* find the extrema for this frame */
	PG_find_extrema(data, 0.0, &dpex, &rpex, &nde, &ddex, &nre, &rdex);

/* setup the domain limits */
	pdx = ((dev->autodomain == TRUE) || (dpex == NULL)) ? ddex : dpex;
	PG_set_window(dev, pdx[0], pdx[1], pdx[2], pdx[3]);

/* setup the range limits */
	prx = ((dev->autorange == TRUE) || (rpex == NULL)) ? rdex : rpex;
	PG_register_range_extrema(dev, nre, prx);

	SFREE(ddex);
	SFREE(rdex);

	PG_set_clipping(dev, FALSE);};

    return(pd);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_IMAGE_CORE - image plot core routine */

static void _PG_image_core(dev, data, fnc_zc, fnc_nc)
   PG_device *dev;
   PG_graph *data;
   PFByte fnc_zc, fnc_nc;
   {REAL amin, amax, *extr;
    PG_graph *g;
    PG_picture_desc *pd;

    pd = PG_setup_picture(dev, data, FALSE, TRUE, TRUE);

    ok_change_viewport = FALSE;

    extr = dev->range_extrema;
    amin = extr[0];
    amax = extr[1];

/* plot all of the current functions */
    for (g = data; g != NULL; g = g->next)
        PG_image_hand(dev, g, amin, amax, fnc_zc, fnc_nc);

    SFREE(pd);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_IMAGE_PLOT - main image plot control routine */

#ifdef PCC

void PG_image_plot(dev, data, va_alist)
   PG_device *dev;
   PG_graph *data;
   va_dcl

#endif

#ifdef ANSI

void PG_image_plot(PG_device *dev, PG_graph *data, ...)

#endif

   {

    im_temp = data;

    if (strcmp(data->f->category, PM_LR_S) == 0)
       _PG_image_core(dev, data, PG_draw_image_zc_lr, PG_draw_image_nc_lr);

    ok_change_viewport = TRUE;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_IMAGE_ZC_LR - draw an image base on an zone centered set of values
 *                     - on a Logical-Rectangular mesh
 */

void PG_draw_image_zc_lr(dev, name, type, f, xmn, xmx, ymn, ymx, fmn, fmx,
			 cnnct, alist)
   PG_device *dev;
   char *name, *type;
   byte *f;
   double xmn, xmx, ymn, ymx, fmn, fmx;
   byte *cnnct;
   pcons *alist;
   {REAL *fp;

    fp = PM_zone_node_lr_2d(f, cnnct, alist);

    PG_draw_image_nc_lr(dev, name, type, fp, xmn, xmx, ymn, ymx, fmn, fmx,
			cnnct, alist);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_IMAGE_NC_LR - draw an image base on a node centered set of values
 *                     - on a Logical-Rectangular mesh
 */

void PG_draw_image_nc_lr(dev, name, type, f, xmn, xmx, ymn, ymx, fmn, fmx,
			 cnnct, alist)
   PG_device *dev;
   char *name, *type;
   byte *f;
   double xmn, xmx, ymn, ymx, fmn, fmx;
   byte *cnnct;
   pcons *alist;
   {int *maxes, kmax, lmax, bpp;
    PG_image *im;

    maxes = (int *) cnnct;
    kmax  = maxes[0];
    lmax  = maxes[1];

    bpp = log((double) (dev->absolute_n_color))/log(2.0) + 0.5;

    im  = PG_make_image(name, type, f,
                        xmn, xmx, ymn, ymx, fmn, fmx,
                        kmax, lmax, bpp, NULL);

    PG_draw_image(dev, im, name, alist);
    PG_rl_image(im);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_POLAR_IMAGE - convert the image whose data is on a polar mesh
 *                - from cartesian to polar form
 *                - the limits in IM are already in cartesian form
 */

static void PG_polar_image(im, pol_extr, bc)
   PG_image *im;
   REAL *pol_extr;
   int bc;
   {int i, j, k, l, kp, lp, ns, ir, ith;
    REAL xmn, xmx, ymn, ymx;
    REAL rmn, rmx, tmn, tmx;
    REAL dr, dth, dx, dy;
    REAL r, th, x, y;
    unsigned char *cart, *polar;

    xmn = im->xmin;
    ymn = im->ymin;
    xmx = im->xmax;
    ymx = im->ymax;

    rmn = pol_extr[0];
    rmx = pol_extr[1];
    tmn = pol_extr[2];
    tmx = pol_extr[3];

    kp = im->kmax;
    lp = im->lmax;
    ns = kp*lp;

    dr  = (rmx - rmn)/((REAL) kp - 1.0);
    dth = (tmx - tmn)/((REAL) lp - 1.0);

    dx = (xmx - xmn)/((REAL) kp - 1.0);
    dy = (ymx - ymn)/((REAL) lp - 1.0);

    polar = im->buffer;
    cart  = im->buffer = FMAKE_N(unsigned char, ns,
				 "PG_POLAR_IMAGE:cart");

    memset(cart, bc, ns);

/* poll the dst points and load the src points */
    for (l = 0; l < lp; l++)
        {y = (ymn + l*dy);
	 for (k = 0; k < kp; k++)
             {x = (xmn + k*dx);

	      r  = sqrt(x*x + y*y);
	      th = PM_atan(x, y)*RAD_DEG;

	      ir  = (r - rmn)/dr;
	      ith = (th - tmn)/dth;
	      if (((ir < 0) || (kp <= ir)) ||
		  ((ith < 0) || (lp <= ith)))
		 continue;

	      i = ith*kp + ir;
	      j = (lp - 1 - l)*kp + k;

              cart[j] = polar[i];};};

    SFREE(polar);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_CROP_IMAGE - crop the image to the domain limits */

static void PG_crop_image(dev, im)
   PG_device *dev;
   PG_image *im;
   {int k, k1, k2, kx, l, l1, l2, lx;
    int i, kmax, lmax;
    REAL xmn, xmx, ymn, ymx, xmin, ymin;
    REAL dx, dy;
    unsigned char *nb, *bf;

    xmin = im->xmin;
    ymin = im->ymin;
    kmax = im->kmax;
    lmax = im->lmax;

    dx = 1.00001*(kmax - 1)/(ABS(im->xmax - xmin));
    dy = 1.00001*(lmax - 1)/(ABS(im->ymax - ymin));

    PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);
    k1 = (xmn - xmin)*dx;
    k2 = (xmx - xmin)*dx;
    l1 = (ymn - ymin)*dy;
    l2 = (ymx - ymin)*dy;

/* the above assumes the domain limits are outside the viewport
 * they may in fact be inside - so compensate for that case
 */
    k1 = max(k1, 0);
    l1 = max(l1, 0);
    k2 = min(k2, kmax - 1);
    l2 = min(l2, lmax - 1);

    kx = k2 - k1 + 1;
    lx = l2 - l1 + 1;

    im->kmax = kx;
    im->lmax = lx;
    im->size = kx*lx;
    im->xmin = xmn;
    im->xmax = xmx;
    im->ymin = ymn;
    im->ymax = ymx;

    bf = im->buffer;
    nb = im->buffer = FMAKE_N(unsigned char, im->size,
			      "PG_CROP_IMAGE:nb");
    for (l = l1; l <= l2; l++)
        for (k = k1; k <= k2; k++)
            {i = l*kmax + k;
             *nb++ = bf[i];};

    SFREE(bf);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_BUILD_IMAGE - map arrays to an image and return it */

PG_image *PG_build_image(dev, name, type, z, kmax, lmax, xmin, xmax,
                         ymin, ymax, zmin, zmax)
   PG_device *dev;
   char *name, *type;
   byte *z;
   int kmax, lmax;
   double xmin, xmax, ymin, ymax, zmin, zmax;
   {int n, nc, bpp;
    unsigned char *bf;
    PG_image *im;

    bpp = log((double) (dev->absolute_n_color))/log(2.0) + 0.5;

    im  = PG_make_image(name, SC_CHAR_S, NULL,
                        xmin, xmax, ymin, ymax, zmin, zmax,
                        kmax, lmax, bpp, NULL);

/* scale the data into the buffer */
    bf = im->buffer;
    n  = kmax*lmax;
    nc = dev->current_palette->n_pal_colors;

    _PG_map_type_image(dev, type, bf, kmax, lmax, n, nc,
		       z, zmin, zmax);

    if (dev->quadrant == QUAD_FOUR)
       PG_invert_image(im);

/* crop the image to the domain limits */
    PG_crop_image(dev, im);

    im->zmin = zmin;
    im->zmax = zmax;

    return(im);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_GET_VIEWPORT_SIZE - return the origin and size of the viewport
 *                      - for the purpose of image display and extraction
 */

void PG_get_viewport_size(dev, pix, piy, pnx, pny)
   PG_device *dev;
   int *pix, *piy, *pnx, *pny;
   {int ix, iy, nx, ny;

    PG_get_viewport_PC(dev, &ix, &nx, &iy, &ny);

    nx = nx - ix + 1;
    if (ny > iy)
       ny = ny - iy + 1;
    else
       ny = iy - ny + 1;

    *pix = ix;
    *piy = iy;
    *pnx = nx;
    *pny = ny;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_PREP_IMAGE - take a raw image of any type and return an image 
 *               - of pixels ready to be rendered in the current viewport
 *               - return a pointer to the new image iff successful
 *               - return NULL otherwise
 */

static PG_image *PG_prep_image(dev, im, nx, ny)
   PG_device *dev;
   PG_image *im;
   int nx, ny;
   {int kmax, lmax, n, nc;
    REAL xmn, xmx, ymn, ymx, zmin, zmax;
    byte *z;
    char *type;
    unsigned char *bf;
    PG_palette *pal;
    PG_image *nim, wrk;

/* NOTE: don't use the zmin and zmax in the image those are only to label
 * the axis by the palette
 */
/*
    zmin = 0.0;
    zmax = 0.0;
*/

/* NOTE: for a linked list mapping you must get the limits this way */
    zmin = im->zmin;
    zmax = im->zmax;

/* make a temporary copy of the image */
    wrk  = *im;
    kmax = im->kmax;
    lmax = im->lmax;
    type = im->element_type;
    z    = (byte *) im->buffer;
    n    = kmax*lmax;

    pal = im->palette;
    if (pal == NULL)
       pal = dev->current_palette;

    nc = pal->n_pal_colors;

    bf = FMAKE_N(unsigned char, n, "PG_PREP_IMAGE:bf");
    wrk.buffer = bf;

    PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);
    if (wrk.xmax == wrk.xmin)
       {wrk.xmax = xmx;
	wrk.xmin = xmn;};

    if (wrk.ymax == wrk.ymin)
       {wrk.ymax = ymx;
	wrk.ymin = ymn;};

    wrk.bits_pixel = log((double) (dev->absolute_n_color))/log(2.0) + 0.5;

    nim  = PG_make_image(dev->title, SC_STRING_S, NULL,
                         wrk.xmin, wrk.xmax, wrk.ymin, wrk.ymax,
			 wrk.zmin, wrk.zmax,
			 nx, ny, wrk.bits_pixel, wrk.palette);

    _PG_map_type_image(dev, type, bf, kmax, lmax, n, nc,
		       z, zmin, zmax);

    PG_place_image(nim, &wrk, TRUE);

/* crop the image to the domain limits */
    PG_crop_image(dev, nim);

    SFREE(wrk.buffer);

    return(nim);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_IMAGE - given an image structure map the pixels to the
 *               - screen
 */

void PG_draw_image(dev, im, label, alist)
   PG_device *dev;
   PG_image *im;
   char *label;
   pcons *alist;
   {int ix, iy, nx, ny, do_par_set;
    REAL *vwprt;
    PG_image *nim;
    PG_graph g, *data;
    PG_picture_desc *pd;

    im->palette = dev->current_palette;

    if (im_temp == NULL)
       {data = &g;
	memset(data, 0, sizeof(PG_graph));
	g.f         = (PM_mapping *) im;
	g.use       = PG_IMAGE_S;
	g.rendering = PLOT_IMAGE;
	g.info_type = SC_PCONS_P_S;
	g.info      = (byte *) alist;
        do_par_set  = TRUE;}

    else
       {data        = im_temp;
        do_par_set  = FALSE;};

    pd    = PG_setup_picture(dev, data, TRUE, TRUE, do_par_set);
    vwprt = pd->viewport;

    PG_get_viewport_size(dev, &ix, &iy, &nx, &ny);

    if (!_PG_allocate_image_buffer(dev, NULL, &nx, &ny))
       return;

/* scale the image to fit its part of the viewport */
    nim = PG_prep_image(dev, im, nx, ny);

    if (_pol_extr != NULL)
       PG_polar_image(nim, _pol_extr, dev->BLACK);

    PG_put_image(dev, nim->buffer, ix, iy, nx, ny);

    PG_rl_image(nim);

    pd->mesh_fl &= (im_temp != NULL);
    PG_finish_picture(dev, data, pd);

    im_temp = NULL;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* PG_EXTRACT_IMAGE - extract a PG_image from the region of DEV
 *                  - specified by the origin (IX, IY) and
 *                  - the extent (NX, NY)
 */
 
PG_image *PG_extract_image(dev, ix, iy, nx, ny, zmn, zmx)
   PG_device *dev;
   int ix, iy, nx, ny;
   double zmn, zmx;
   {int bpp;
    unsigned char *bf;
    REAL xmn, xmx, ymn, ymx;
    PG_image *im;

    if ((nx == 0) || (ny == 0))
       PG_get_viewport_size(dev, &ix, &iy, &nx, &ny);

    PG_get_viewport_WC(dev, &xmn, &xmx, &ymn, &ymx);

    bpp = log((double) (dev->absolute_n_color))/log(2.0) + 0.5;
    im  = PG_make_image(dev->title, SC_CHAR_S, NULL,
                        xmn, xmx, ymn, ymx, zmn, zmx,
                        nx, ny, bpp, NULL);

/* load up the image data */
    bf = im->buffer;

    PG_get_image(dev, bf, ix, iy, nx, ny);
 
    return(im);}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PAR_RESET_VIEWPORT - reset the viewport to the actual image size
 *                        - NOTE: temporary until roundoff problems in
 *                        -       original parallel viewport computation
 *                        -       are fixed
 */

static void _PG_par_reset_viewport(dev, im, ix, iy, pnx, pny)
   PG_device *dev;
   PG_image *im;
   int ix, iy, *pnx, *pny;
   {int mx, my, nx, ny;
    REAL vxmn, vxmx, vymn, vymx;

    nx = im->kmax;
    ny = im->lmax;
    
    mx = ix + nx;
    my = iy + ny;

    if (dev->quadrant == QUAD_FOUR)
       {PtoS(dev, ix, my, vxmn, vymn);
	PtoS(dev, mx, iy, vxmx, vymx);}
    else
       {PtoS(dev, ix, iy, vxmn, vymn);
	PtoS(dev, mx, my, vxmx, vymx);};

    PG_set_viewport(dev, vxmn, vxmx, vymn, vymx);

    *pnx = nx;
    *pny = ny;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_RENDER_PARALLEL - complete the parallel rendering of a picture */

void PG_render_parallel(dd, nim, np, pim)
   PG_device *dd;
   PG_image *nim;
   int np;
   PG_image *pim;
   {int ix, iy, nx, ny;
    PG_graph g, *data;
    PG_picture_desc *pd;
    PG_palette *pl;

    data = &g;
    memset(data, 0, sizeof(PG_graph));
    g.f         = (PM_mapping *) nim;
    g.use       = PG_IMAGE_S;
    g.info_type = SC_PCONS_P_S;
    g.info      = (byte *) dd->pri->alist;
    g.rendering = dd->pri->render;

    pd = PG_setup_picture(dd, data, TRUE, FALSE, FALSE);

    PG_get_viewport_size(dd, &ix, &iy, &nx, &ny);

/* GOTCHA: this is only necessary because the round off issues
 *         are not dealt with completely in PG_setup_parallel
 *         hence nx and nim->kmax et. al. differ by 2-5 pixels
 */
    _PG_par_reset_viewport(dd, nim, ix, iy, &nx, &ny);

    pl = dd->current_palette;
    dd->current_palette = dd->color_table;

    PG_put_image(dd, nim->buffer, ix, iy, nx, ny);

    dd->current_palette = pl;

    PG_finish_picture(dd, data, pd);

/* release nim only now because the label printed in PG_finish_picture
 * comes from here
 */
    PG_rl_image(nim);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_FILL_PALETTE_IMAGE - fill in the pixels for the palette image
 *                        - add 2 to each color to allow for the mandatory
 *                        - black and white included in each palette
 */

static void _PG_fill_palette_image(dev, bf, nx, ny, hvf, nc, ncx, ncy)
   PG_device *dev;
   unsigned char *bf;
   int nx, ny, hvf, nc, ncx, ncy;
   {int l, k, ls;
    unsigned char *pbf;
    REAL a[2], extr[4];
    double scale;

    pbf = bf;

    if (ncx == nc) 
       /* 1-d color table */
       {if (hvf == VERTICAL)
           {scale = 0.9999*nc/((double) ny);
            for (l = 0; l < ny; l++)
                {ls = (ny - l)*scale;
                 for (k = 0; k < nx; k++)
                     *pbf++ = ls + 2;};}

        else if (hvf == HORIZONTAL)
           {scale = 0.9999*nc/((double) nx);
            for (l = 0; l < ny; l++)
                {for (k = 0; k < nx; k++)
                     *pbf++ = k*scale + 2;};};}

    else if((ncx*ncy == nc) && (ncy > 1)) 
       /* 2-d color table */
       {extr[0] = 0.0; extr[1] = (REAL) (nx - 1);
        extr[2] = 0.0; extr[3] = (REAL) (ny - 1);
        for (l = ny - 1; l >= 0; l--)
            {a[1] = (REAL) ((ny - 1) - l);
             for (k = 0; k < nx; k++)
                {a[0] = (REAL) k;
                 pbf[l*nx + k] = (unsigned char)PG_select_color(dev, 2, a, extr);};};}

#ifdef MSC

/* compute a palette image */
       {scale = 0.9999*nc/((double) ny);
        for (l = 0; l < ny; l++)
            {ls = l*scale;
             for (k = 0; k < nx; k++)
                 *pbf++ = ls + 2;};};

#endif

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_PALETTE - display the palette next to the image */

void PG_draw_palette(dev, xmn, ymn, xmx, ymx, zmn, zmx, wid)
   PG_device *dev;
   double xmn, ymn, xmx, ymx, zmn, zmx, wid;
   {int ix, iy, nx, ny;
    int n_color, hvf;
    unsigned char *bf;
    char format[20];
    double sxmn, symn, sxmx, symx, scale, toler;
    REAL fxmn, fxmx, fymn, fymx;
    PG_palette *pal;

    fxmn = xmn;
    fxmx = xmx;
    fymn = ymn;
    fymx = ymx;

    PG_frame_viewport(dev, &fxmn, &fxmx, &fymn, &fymx);

/* reduce number of off-by-one pixel cases */
    StoP(dev, fxmn, fymn, ix, iy);
    PtoS(dev, ix, iy, fxmn, fymn);
    StoP(dev, fxmx, fymx, ix, iy);
    PtoS(dev, ix, iy, fxmx, fymx);

    sxmn = fxmn;
    sxmx = fxmx;
    symn = fymn;
    symx = fymx;

/* set up the palette image */
    StoW(dev, fxmn, fymn);
    StoW(dev, fxmx, fymx);

    pal = dev->current_palette;
    dev->current_palette = dev->palettes;

    PG_set_color_line(dev, dev->WHITE, TRUE);
    strcpy(format, "%10.2g");

/* NOTE: this value is chosen to avoid problems with StoP and
 *       clipping for arbitrary size windows
 */
    toler = 10.0*dev->window_width/((REAL) INT_MAX);
    if (ABS(zmx - zmn) < toler*(ABS(zmn) + ABS(zmx)))
       {if (zmn == 0.0)
	   {zmn -= 0.1;
	    zmx += 0.1;}
        else
	   {zmn -= 0.5*toler*zmn;
	    zmx += 0.5*toler*zmx;};};

    if (sxmn == sxmx)
       {sxmn  = sxmx - wid;
        hvf   = VERTICAL;
        scale = (fymx - fymn)/(zmx - zmn + SMALL);

        PG_draw_axis(dev, fxmn, fymn, fxmn, fymx,
                     0.0, 1.0, zmn, zmx, scale,
                     format,
                     RIGHT_OF_AXIS, RIGHT_OF_AXIS, FALSE,
                     MAJOR, MINOR, LABEL, 0);

        dev->current_palette = pal;}

    else if (symn == symx)
       {symx  = symn + wid;
        hvf   = HORIZONTAL;
    
        scale = (fxmx - fxmn)/(zmx - zmn + SMALL);

        PG_draw_axis(dev, fxmn, fymn, fxmx, fymn,
                     0.0, 1.0, zmn, zmx, scale,
                     format,
                     RIGHT_OF_AXIS, RIGHT_OF_AXIS, FALSE,
                     MAJOR, MINOR, LABEL, 0);

        dev->current_palette = pal;}

    else
       {dev->current_palette = pal;
        return;};

    if (dev->quadrant == QUAD_ONE)
       {StoP(dev, sxmn, symn, ix, iy);
	StoP(dev, sxmx, symx, nx, ny);}

    else if (dev->quadrant == QUAD_FOUR)
       {StoP(dev, sxmn, symx, ix, iy);
	StoP(dev, sxmx, symn, nx, ny);};

    nx = nx - ix + 1;
    if (ny > iy)
       ny = ny - iy + 1;
    else
       ny = iy - ny + 1;

    if (!_PG_allocate_image_buffer(dev, &bf, &nx, &ny))
       return;

/* compute a palette image */
    n_color = pal->n_pal_colors;
    _PG_fill_palette_image(dev, bf, nx, ny, hvf, n_color, n_color, 0);

/* move image off axis */
    if (hvf == VERTICAL)
       ix -= 1;
    else
       {if (dev->quadrant == QUAD_ONE)
	   iy += 1;
        else
	   iy -= 1;};

/* draw the palette image */
    PG_put_image(dev, bf, ix, iy, nx, ny);

    SFREE(bf);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_DRAW_2DPALETTE - display the palette next to the image */

void PG_draw_2dpalette(dev, xmn, ymn, xmx, ymx, z1mn, z1mx,
                       z2mn, z2mx, wid)
   PG_device *dev;
   double xmn, ymn, xmx, ymx, z1mn, z1mx, z2mn, z2mx, wid;
   {int ix, iy, nx, ny, ncx, ncy;
    int n_color, **pd, *shape, hvf, fsize;
    unsigned char *bf;
    char format[20], *face, *style;
    double sxmn, symn, sxmx, symx, scale, toler;
    REAL fxmn, fxmx, fymn, fymx;
    PG_palette *pal;

    pal = dev->current_palette;
    pd  = pal->pal_dims;

    if ((pd == NULL) || (pal->max_pal_dims != 2))
       PG_draw_palette(dev, xmn, ymn, xmx, ymx,
                       z1mn, z1mx, wid);

    fxmn = xmn;
    fxmx = xmx;
    fymn = ymn;
    fymx = ymx;

    PG_frame_viewport(dev, &fxmn, &fxmx, &fymn, &fymx);

/* reduce number of off-by-one pixel cases */
    StoP(dev, fxmn, fymn, ix, iy);
    PtoS(dev, ix, iy, fxmn, fymn);
    StoP(dev, fxmx, fymx, ix, iy);
    PtoS(dev, ix, iy, fxmx, fymx);

    sxmn = fxmn;
    sxmx = fxmx;
    symn = fymn;
    symx = fymx;

/* set up the palette image */
    StoW(dev, fxmn, fymn);
    StoW(dev, fxmx, fymx);

    dev->current_palette = dev->palettes;

    PG_set_color_line(dev, dev->WHITE, TRUE);
    strcpy(format, "%10.2g");

/* NOTE: this value is chosen to avoid problems with StoP and
 *       clipping for arbitrary size windows
 */
    toler = 10.0*dev->window_width/((REAL) INT_MAX);
    if (ABS(z1mx - z1mn) < toler*(ABS(z1mn) + ABS(z1mx)))
       {if (z1mn == 0.0)
	   {z1mn -= 0.1;
	    z1mx += 0.1;}
        else
	   {z1mn -= 0.5*toler*z1mn;
	    z1mx += 0.5*toler*z1mx;};};

    if (ABS(z2mx - z2mn) < toler*(ABS(z2mn) + ABS(z2mx)))
       {if (z2mn == 0.0)
	   {z2mn -= 0.1;
	    z2mx += 0.1;}
        else
	   {z2mn -= 0.5*toler*z2mn;
	    z2mx += 0.5*toler*z2mx;};};

    shape = pal->pal_dims[pal->max_pal_dims - 1];
    ncx = shape[0];
    ncy = shape[1];

    if (sxmn == sxmx)

/*
       {sxmn  = sxmx - wid;
        hvf   = VERTICAL;
        scale = (fymx - fymn)/(zmx - zmn + SMALL);

        PG_draw_axis(dev, fxmn, fymn, fxmn, fymx,
                     0.0, 1.0, zmn, zmx, scale,
                     format,
                     RIGHT_OF_AXIS, RIGHT_OF_AXIS, FALSE,
                     MAJOR, MINOR, LABEL, 0);

*/
       {dev->current_palette = pal;
        return;} /* VERTICAL not supported yet */

    else
       {hvf   = HORIZONTAL;
    
        scale = (fxmx - fxmn)/(z1mx - z1mn + SMALL);

        PG_get_font(dev, &face, &style, &fsize);
        PG_set_font(dev, face, style, 8);
        
        PG_draw_axis(dev, fxmn, fymn, fxmx, fymn,
                     0.0, 1.0, z1mn, z1mx, scale,
                     format,
                     RIGHT_OF_AXIS, RIGHT_OF_AXIS, FALSE,
                     MAJOR, MINOR, LABEL, 0);

        scale = (fymx - fymn)/(z2mx - z2mn + SMALL);   

        PG_draw_axis(dev, fxmn, fymn, fxmn, fymx,
                     0.0, 1.0, z2mn, z2mx, scale,
                     format,
                     LEFT_OF_AXIS, LEFT_OF_AXIS, FALSE,
                     MAJOR, NOTHING_ON_AXIS, LABEL, 0);

        PG_set_font(dev, face, style, fsize);

        dev->current_palette = pal;}

    if (dev->quadrant == QUAD_ONE)
       {StoP(dev, sxmn, symn, ix, iy);
	StoP(dev, sxmx, symx, nx, ny);}

    else if (dev->quadrant == QUAD_FOUR)
       {StoP(dev, sxmn, symx, ix, iy);
	StoP(dev, sxmx, symn, nx, ny);};

    nx = nx - ix + 1;
    if (ny > iy)
       ny = ny - iy + 1;
    else
       ny = iy - ny + 1;

    if (!_PG_allocate_image_buffer(dev, &bf, &nx, &ny))
       return;

/* compute a palette image */
    n_color = pal->n_pal_colors;
    _PG_fill_palette_image(dev, bf, nx, ny, hvf, n_color, ncx, ncy);

/* move image off axis */
    if (hvf == VERTICAL)
       ix -= 1;
    else
       {if (dev->quadrant == QUAD_ONE)
	   iy += 1;
        else
	   iy -= 1;};

/* draw the palette image */
    PG_put_image(dev, bf, ix, iy, nx, ny);

    SFREE(bf);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_BYTE_BIT_MAP - compress a byte map to a bit map
 *                  - return the effective row width of the bit map
 */

int _PG_byte_bit_map(bf, nx, ny, complmnt)
   unsigned char *bf;
   int nx, ny, complmnt;
   {int i, k, l;
    unsigned int mask, xbyte;
    unsigned char *pbt, *pbf;

    pbf = bf;
    pbt = bf;
    mask  = 0x80;
    xbyte = 0;

    if (complmnt)
       {for (l = 0; l < ny; l++)
            {for (k = 0; k < nx; k++)
                 {i = *pbf++;
                  xbyte |= mask*i;
                  mask >>= 1;
                  if (mask == 0)
                     {mask   = 0x80;
                      *pbt++ = ~xbyte;
                      xbyte  = 0;};};
             mask   = 0x80;
             *pbt++ = ~xbyte;
             xbyte  = 0;};}

    else
       {for (l = 0; l < ny; l++)
            {for (k = 0; k < nx; k++)
                 {i = *pbf++;
                  xbyte |= mask*i;
                  mask >>= 1;
                  if (mask == 0)
                     {mask   = 0x80;
                      *pbt++ = xbyte;
                      xbyte  = 0;};};
             mask   = 0x80;
             *pbt++ = xbyte;
             xbyte  = 0;};};

    return((nx + 7) >> 3);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_MAKE_IMAGE - initialize an image */

PG_image *PG_make_image(label, type, z, xmn, xmx, ymn, ymx,
			zmn, zmx, k, l,
                        bits_pix, palette)
   char *label, *type;
   byte *z;
   double xmn, xmx, ymn, ymx, zmn, zmx;
   int k, l, bits_pix;
   PG_palette *palette;
   {PG_image *im;
    long width, height, size;
    char bf[MAXLINE];

    width  = k;
    height = l;
    size   = width*height;

    im = FMAKE(PG_image, "PG_MAKE_IMAGE:im");
    if (im == NULL)
       return(NULL);

    if (z == NULL)
       {im->buffer = FMAKE_N(unsigned char, size*SC_sizeof(type),
                             "PG_MAKE_IMAGE:imbuffer");
	if (im->buffer == NULL)
	   {SFREE(im);
	    return(NULL);};}
    else
       im->buffer = (unsigned char *) z;

    sprintf(bf, "%s *", type);

    im->version_id   = PG_IMAGE_VERSION;
    im->label        = SC_strsavef(label, "char*:PG_MAKE_IMAGE:label");
    im->element_type = SC_strsavef(bf, "char*:PG_MAKE_IMAGE:type");
    im->xmin         = xmn;
    im->xmax         = xmx;
    im->ymin         = ymn;
    im->ymax         = ymx;
    im->zmin         = zmn;
    im->zmax         = zmx;
    im->kmax         = k;
    im->lmax         = l;
    im->size         = size;
    im->bits_pixel   = bits_pix;
    im->palette      = palette;

    SC_mark(im->label, 1);
    SC_mark(im->element_type, 1);
    SC_mark(im->buffer, 1);

    return(im);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_ALLOCATE_IMAGE_BUFFER - allocate an appropriately sized buffer */

int _PG_allocate_image_buffer(dev, pbf, pnx, pny)
   PG_device *dev;
   unsigned char **pbf;
   int *pnx, *pny;
   {int i, nx, ny, ix, iy;
    unsigned char *bf;

    nx = *pnx;
    ny = *pny;

    bf = NULL;
    for (i = dev->resolution_scale_factor; i < 512; i <<= 2)
        {ix = nx/i;
         iy = ny/i;

	 if (CGM_DEVICE(dev))
	    {if (ix & 1)
	        ix++;

	     if (iy & 1)
	        iy++;};

	 bf = FMAKE_N(unsigned char, ix*iy,
                      "_PG_ALLOCATE_IMAGE_BUFFER:bf");
         if (bf != NULL)
            break;};

    if (bf != NULL)
       memset(bf, dev->BLACK, ix*iy);

    dev->resolution_scale_factor = i;

    if (pbf != NULL)
       *pbf = bf;
    else
       SFREE(bf);

    *pnx = ix;
    *pny = iy;

    return(i < 512);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_RL_IMAGE - free the space associated with an image */

void PG_rl_image(im)
   PG_image *im;
   {

    if (im != NULL)
       {SFREE(im->label);
	SFREE(im->element_type);
	SFREE(im->buffer);
	SFREE(im);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_SHIFT_IMAGE_RANGE - shift the data values of B to match
 *                      - the range of A and add on offset OFF
 */

int PG_shift_image_range(a, b, off)
   PG_image *a, *b;
   int off;
   {int i, nb, ok;
    unsigned int ac;
    unsigned char *pb;
    REAL azn, azx, bzn, bzx;
    REAL a0, b0, as, ai, bc, nca;
    PG_palette *pal;

    ok = TRUE;

    azn = a->zmin;
    azx = a->zmax;
    pal = a->palette;
    nca = pal->n_pal_colors - off;

    nb  = b->size;
    pb  = b->buffer;
    bzn = b->zmin;
    bzx = b->zmax;

    if (azn == azx)
       {as = 0.0;
	ai = off;}
    else
       {a0 = nca/(azx - azn);
	b0 = (bzx - bzn)/nca;
	as = a0*b0;
	ai = a0*(bzn - azn) + off;};

    for (i = 0; i < nb; i++)
        {bc = (REAL) (pb[i] - off);
	 ac = (unsigned char) (as*bc + ai);

	 pb[i] = ac;};

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_PLACE_IMAGE - translate, scale, and copy the data from SIM to DIM
 *                - in the region of overlap
 */

int PG_place_image(dim, sim, scale)
   PG_image *dim, *sim;
   int scale;
   {int sx, sy, dx, dy;
    int k, kmn, kmx, l, lmn, lmx;
    int ix, iy, id, is, z1, z2, z3, z4;
    unsigned char *iz1, *iz2, *iz3, *iz4;
    unsigned char *sp, *dp;
    REAL xmn, xmx, ymn, ymx;
    REAL sxmn, sxmx, symn, symx;
    REAL dxmn, dxmx, dymn, dymx;
    REAL x, y, fx, fy, dxp, dyp;

    sx = sim->kmax;
    sy = sim->lmax;
    sp = sim->buffer;

    dx = dim->kmax;
    dy = dim->lmax;
    dp = dim->buffer;

/* define the region of physical overlap */
    sxmn = sim->xmin;
    sxmx = sim->xmax;
    symn = sim->ymin;
    symx = sim->ymax;

    dxmn = dim->xmin;
    dxmx = dim->xmax;
    dymn = dim->ymin;
    dymx = dim->ymax;

    xmn = max(sxmn, dxmn);
    xmx = min(sxmx, dxmx);
    ymn = max(symn, dymn);
    ymx = min(symx, dymx);

    dxp = (dxmx - dxmn)/((REAL) dx);
    dyp = (dymx - dymn)/((REAL) dy);

    kmn = (xmn - dxmn)/dxp;
    kmx = (xmx - dxmn)/dxp;
    lmn = (ymn - dymn)/dyp;
    lmx = (ymx - dymn)/dyp;

    if (scale)
       {fx = ((REAL) (sx-2))/((REAL) (dx-1));
	fy = ((REAL) (sy-2))/((REAL) (dy-1));}
    else
       {fx = 1.0;
	fy = 1.0;};

/* set up the cell corner pointers of the source data */
    iz1 = sp - sx;
    iz2 = sp;
    iz3 = iz2 - 1;
    iz4 = iz1 - 1;

/* map it all in */
    for (l = lmn; l < lmx; l++)
        {for (k = kmn; k < kmx; k++)
             {x  = k*fx;
              y  = l*fy;
              ix = (int) floor(x) + 1;
              iy = (int) floor(y) + 1;

              id = l*dx + k;
              is = iy*sx + ix;

              x -= floor(x);
              y -= floor(y);

              z1 = iz1[is];
              z2 = iz2[is];
              z3 = iz3[is];
              z4 = iz4[is];

              dp[id] = ((z4 - z3 + z2 - z1)*x + (z3 - z4))*y +
                       (z1 - z4)*x + z4;};};

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_COPY_IMAGE - copy the data from SIM to DIM
 *               - in the region of overlap
 *               - NOTE: do not copy background color (BCK) pixels from
 *               -       the source image SIM
 */

int PG_copy_image(dim, sim, bck)
   PG_image *dim, *sim;
   int bck;
   {int id, is, od, os, c;
    int k, kmnd, kmxd, lmnd, lmxd, dk;
    int l, kmns, kmxs, lmns, lmxs, dl;
    unsigned char *sp, *dp;
    REAL xmn, xmx, ymn, ymx;
    REAL sxmn, sxmx, symn, symx;
    REAL dxmn, dxmx, dymn, dymx;
    REAL dx, dy, dxp, dyp, sx, sy, sxp, syp;

    sx = sim->kmax;
    sy = sim->lmax;
    sp = sim->buffer;

    dx = dim->kmax;
    dy = dim->lmax;
    dp = dim->buffer;

/* define the region of physical overlap */
    sxmn = sim->xmin;
    sxmx = sim->xmax;
    symn = sim->ymin;
    symx = sim->ymax;

    dxmn = dim->xmin;
    dxmx = dim->xmax;
    dymn = dim->ymin;
    dymx = dim->ymax;

    xmn = max(sxmn, dxmn);
    xmx = min(sxmx, dxmx);
    ymn = max(symn, dymn);
    ymx = min(symx, dymx);

    dxp = (dxmx - dxmn)/dx;
    dyp = (dymx - dymn)/dy;
    sxp = (sxmx - sxmn)/sx;
    syp = (symx - symn)/sy;

    kmnd = (xmn - dxmn)/dxp;
    kmxd = (xmx - dxmn)/dxp;
    lmnd = (ymn - dymn)/dyp;
    lmxd = (ymx - dymn)/dyp;

    kmns = (xmn - sxmn)/sxp;
    kmxs = (xmx - sxmn)/sxp;
    lmns = (ymn - symn)/syp;
    lmxs = (ymx - symn)/syp;

    dk = kmxd - kmnd;
    dl = lmxd - lmnd;
    od = lmnd*dx + kmnd;
    os = lmns*sx + kmns;

    for (l = 0; l < dl; l++)
        {for (k = 0; k < dk; k++)
             {id = l*dx + k + od;
              is = l*sx + k + os;

	      c = sp[is];
	      if (c != bck)
		 dp[id] = c;};};

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_INTP_BYTE - intepolate one single byte array into another
 *               - the length of each array is provided as well as
 *               - a stride for each array
 */

void _PG_intp_byte(op, np, ox, nx, os, ns)
   unsigned char *op, *np;
   int ox, nx, os, ns;
   {int i, io1, ax;
    double scale, si, ov1, ov2;

    scale = 0.999*((double) ((ox-os)*ns)) / ((double) ((nx-ns)*os) + SMALL);
    ax    = nx/ns;
    for (i = 0; i < ax; i++)
        {si  = i*scale;
         io1 = si;
         ov1 = op[io1*os];
         ov2 = op[(io1 + 1)*os];

         np[i*ns] = ov1*(io1 + 1 - si) + ov2*(si - io1) + 0.5;};

    return;}

/*--------------------------------------------------------------------------*/

/*                            FORTRAN API ROUTINES                          */

/*--------------------------------------------------------------------------*/

/* PGPLIM - low level image plot routine */

FIXNUM F77_ID(pgplim_, pgplim, PGPLIM)(devid, pnc, name, pnct, type,
				       pz, pk, pl,
				       pxn, pxx, pyn, pyx, pzn, pzx, pal)
   FIXNUM *devid, *pnc;
   F77_string name;
   FIXNUM *pnct;
   F77_string type;
   REAL *pz;
   FIXNUM *pk, *pl;
   REAL *pxn, *pxx, *pyn, *pyx, *pzn, *pzx;
   FIXNUM *pal;
   {PG_device *dev;
    char lname[MAXLINE], ltype[MAXLINE];
    int maxes[2];
    pcons *alst;

    SC_FORTRAN_STR_C(lname, name, *pnc);
    SC_FORTRAN_STR_C(ltype, type, *pnct);

    maxes[0] = *pk;
    maxes[1] = *pl;

    dev  = SC_GET_POINTER(PG_device, *devid);
    alst = SC_GET_POINTER(pcons, *pal);

    PG_draw_image_nc_lr(dev, lname, ltype, pz,
			*pxn, *pxx, *pyn, *pyx, *pzn, *pzx, maxes, alst);

    return((FIXNUM) TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

