/*
 * PPC.C - Portable Process Control system
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "ppc.h"

/* ISSUES and PROBLEMS
 *
 *  (1) Buffer flushing has been a plague since day 1.  The result
 *      is lost characters.
 *         (a) flush sometimes means pass the contents on and
 *             sometimes it means throw the contents away
 *         (b) PPC wants "pass it on" semantics or nothing
 *  (2) The IOCTL semantic variations are hard to follow. The vanilla
 *      BSD and SYSV are bad enough, but toss in vendor differences
 *      and fun, fun, fun is the result.
 *  (3) The differing PTY interface, esp. wrt. open, is a sore point.
 *  (4) Partial ports to DOS are imaginable, especially since some
 *      commercial products do some useful things along these lines
 *      which illustrate that it is possible to do some of what
 *      PPC aims to do.
 *  (5) Partial ports to MAC should likewise be investigated.
 *  (6) WINDOWS should be able to support a reasonably complete
 *      port of PPC.
 *  (7) On some platforms (SGI) when communicating via a local
 *      socket, the parent does not notice the SIGCHILD and
 *      major hangs happen.
 *  (8) One would like to detect the death of a remote child
 *      but rexec doesn't seem to be too helpful there.
 *  (9) When using PC_poll keep in mind that a signal may cause
 *      it to return without detecting that there is something
 *      ready! For example, it is a good idea to do a PC_gets
 *      before PC_close'ing in case a SIGCHILD tossed the
 *      latest PC_poll out.
 * (10) On some systems (eg AIX) the scheduler lets the child run to
 *      completion (after the fork) before letting the parent set
 *      pp->id to cp->id which causes PC_signal_handler to fail to
 *      recognize that the child is a process which it knows about.
 *      The work around is to have the parent and child exchange one
 *      agreed upon byte before exec'ing the child task in the child
 *      process.
 */

#define DATA_BLOCK 4096

int
 _PC_n_processes = 0,
 _PC_max_processes = 0;

PROCESS
 **_PC_processes = NULL;

int
 _PC_current_flushed_process = -1;

TERMINAL_STATE
 *PC_original_terminal_state = NULL;

PROCESS
 *PC_terminal = NULL;

char
 PC_err[MAXLINE];

static void
 SC_DECLARE(PC_remote_handler, (int signo, int code)),
 SC_DECLARE(_PC_latch_child, (PROCESS *pp));

static int
 SC_DECLARE(_PC_init_ipc, (PROCESS *pp, PROCESS *cp)),
 SC_DECLARE(_PC_complete_messagep, (PROCESS *pp));

static PROCESS
 SC_DECLARE(*_PC_open_remote,
         (char *host, char *proc, char **argv, char **envp, char *mode)),
 SC_DECLARE(*_PC_open_local, (char **argv, char **envp, char *mode)),
 SC_DECLARE(*_PC_open_proc,
         (int rcpu, char *name, char **argv, char **envp, char *mode));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_OPEN - start up a process and open a pipe to it for further
 *         - communications
 *         - there are the following variants for the "name" of the process
 *         - to be opened:
 *         -
 *         -  1) <name>              - process "name" to be forked on
 *         -                         - the current CPU/host
 *         -  2) <host>:<name>       - process "name" to be exec'd
 *         -                         - on remote "host"
 *         -  3) <CPU>@<name>        - process "name" to be forked
 *         -                         - on processor number "CPU" of
 *         -                         - the current, parallel machine
 *         -  4) <host>:<CPU>@<name> - process "name" to be forked on
 *         -                         - processor number "CPU" of the
 *         -                         - remote, parallel "host"
 *         - the legal modes are:
 *         -  r    - read only (child stdout only connected to parent)
 *         -  w    - write only (child stdin only connected to parent)
 *         -  a    - read/write (child stdin and stdout connnect to parent)
 *         -
 *         - after rwa can come (in any order)
 *         -  b    - binary data exchanged
 *         -  p    - communicate via pipe
 *         -  s    - communicate via socket
 *         -  t    - communicate via pty
 *         - only one of "pst" can be used and only for current CPU/host
 *         - processes
 *         - (NOTE: for ftp or telnet the "t" option is useful)
 */

PROCESS *PC_open(argv, envp, mode)
   char **argv, **envp, *mode;
   {char name[MAXLINE], *host, *proc, *s;
    static int first = TRUE;

    if (first)
       {
#ifdef HAVE_PROCESS_CONTROL
        SIGNAL(SIGCHLD, PC_signal_handler);
#endif
        first = FALSE;};

    PC_ERR_TRAP(NULL);

    if ((argv == NULL) || (argv[0] == NULL))
       return(NULL);

    strcpy(name, argv[0]);

    if (PC_OTHER_HOSTP(name))
       {host = SC_strtok(name, ":", s);
        proc = SC_strtok(NULL, "\n", s);
        return(_PC_open_remote(host, proc, argv, envp, mode));}

    else
       return(_PC_open_local(argv, envp, mode));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_OPEN_REMOTE - start up a process on a remote host
 *                 - and open a connection to it for further communications
 */

static PROCESS *_PC_open_remote(host, proc, argv, envp, mode)
   char *host, *proc, **argv, **envp, *mode;
   {PROCESS *pp;

#ifdef HAVE_SOCKETS_P

    char command[MAXLINE], bf[MAXLINE];
    char *h, *u, *p, **pa, *t;
    struct servent *sp;
    int s;
/*
    int err_fd;

    err_fd = 0;
 */
    SIGNAL(SIGPIPE, PC_remote_handler);
    SIGNAL(SIGHUP, PC_remote_handler);
    SIGNAL(SIGABRT, PC_remote_handler);
    SIGNAL(SIGURG, PC_remote_handler);

    pp = PC_mk_process(argv, mode, PC_PARENT);

    sp = getservbyname("exec", "tcp");
    if (sp == NULL)
       PC_error("GETSERVBYNAME CALL FAILED - _PC_OPEN_REMOTE");

/* build up the command string from argv */
    pa = argv + 1;
    strcpy(command, proc);
    while (*pa != NULL)
        {strcat(command," ");
         strcat(command, *pa++);};

    strcpy(bf, host);
    h = SC_strtok(bf, ",", t);
    u = SC_strtok(NULL, ",", t);
    p = SC_strtok(NULL, "\n", t);

/* on some systems rexec's prompt for info is caught by the interrupt
 * handler - so disable it while rexec is at work
 */
    PC_catch_io_interrupts(FALSE);

/*    s = rexec(&h, sp->s_port, u, p, command, &err_fd); */
    s = rexec(&h, sp->s_port, u, p, command, NULL);

    PC_catch_io_interrupts(PC_io_interrupts_on);

    if (s <= 0)
       PC_error("REXEC CALL FAILED - _PC_OPEN_REMOTE");

    if (PC_BINARY_MODEP(mode))
       {pp->data_type = BINARY;

/* if this is a binary connection inform the parent of the binary formats */
        PC_send_formats();}

    else
       pp->data_type = ASCII;

    pp->medium = USE_SOCKETS;

    pp->in  = s;
    pp->out = s;

#else

    pp = NULL;

#endif

    return(pp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_OPEN_LOCAL - start up a process on the current CPU
 *                - and open a connection to it for further communications
 */

static PROCESS *_PC_open_local(argv, envp, mode)
   char **argv, **envp, *mode;
   {PROCESS *pp;
    int rcpu;
    char name[MAXLINE], *token, *proc, *s;

    strcpy(name, argv[0]);

/* if requesting another CPU/node */
    if (PC_OTHER_CPUP(name))
       {token = SC_strtok(name, "@", s);
        proc  = SC_strtok(NULL, "\n", s);

        rcpu  = SC_stol(token);

        PC_open_node(rcpu, proc, argv, envp, mode);}

    else
       pp = _PC_open_proc(-1, name, argv, envp, mode);

    return(pp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_OPEN_PROC - start up a process on the current CPU
 *               - and open a connection to it for further communications
 *               - NOTE: on some systems the scheduler lets the child run
 *               - to completion (after the fork) before letting the parent
 *               - set pp->id to cp->id which causes PC_signal_handler to
 *               - fail to recognize that the child is a process which it
 *               - knows about - the work around is to have the parent
 *               - and child exchange one agreed upon byte before exec'ing
 *               - the child task in the child process
 */

static PROCESS *_PC_open_proc(rcpu, name, argv, envp, mode)
   int rcpu;
   char *name, **argv, **envp, *mode;
   {PROCESS *pp, *cp;
    char *av;
    int pid, flag;

#ifdef HAVE_PROCESS_CONTROL

/* allocate PROCESS for the parent (pp) and the child (cp) */

    av      = argv[0];
    argv[0] = name;

    cp = PC_mk_process(argv, mode, PC_CHILD);
    pp = PC_mk_process(argv, mode, PC_PARENT);

    switch (mode[1])
       {case 't' :
             pp->medium = USE_PTYS;
             cp->medium = USE_PTYS;
             break;
        case 's' :
             pp->medium = USE_SOCKETS;
             cp->medium = USE_SOCKETS;
             break;
        default  :
        case 'p' :
             pp->medium = USE_PIPES;
             cp->medium = USE_PIPES;
             break;};

    if (PC_BINARY_MODEP(mode))
       {pp->data_type = BINARY;
        cp->data_type = BINARY;}
    else
       {pp->data_type = ASCII;
        cp->data_type = ASCII;};

/* mixing PROCESS's using PTYs and other media is a bad idea at
 * this point
 */
    if ((pp->medium == USE_PTYS) && (PC_original_terminal_state == NULL))
       PC_original_terminal_state = PC_get_term_state(0);

/* set up the communications pipe */
    flag = _PC_init_ipc(pp, cp);
    if (flag == FALSE)
       {SFREE(pp);
        SFREE(cp);
        _PC_n_processes--;

        PC_error("COULDN'T CREATE IPC CHANNELS - _PC_OPEN_PROC");};

/* fork the process */
    pid = fork();
    pp->id = cp->id = pid;
    switch (pid)
       {case -1 :
             pp->release(pp);
	     pp = NULL;
	     cp->release(cp);
	     cp = NULL;

	     _PC_n_processes--;

	     PC_error("COULDN'T FORK PROCESS - _PC_OPEN_PROC");

	     break;

/* the child process comes here and if successful will never return */
        case 0 :
	     pp->release(pp);
	     pp = NULL;
	     if (cp->exec != NULL)
	        cp->exec(cp, argv, envp, mode);

             else
	        {cp->release(cp);
		 cp = NULL;

		 _PC_n_processes--;
		 PC_error("NO EXEC - _PC_OPEN_PROC");};

	     break;

/* the parent process comes here */
        default :
             pp->id     = cp->id;
	     pp->rcpu   = rcpu;
	     pp->status = RUNNING;
	     pp->reason = 0;

# ifdef F_SETOWN
	     if (pp->medium == USE_SOCKETS)
	        fcntl(pp->in, F_SETOWN, getpid());
# endif

             _PC_latch_child(pp);

	     if (PC_BINARY_MODEP(mode))
	        {if (!PC_recv_formats(pp))
		    pp = NULL;};

	     if ((pp != NULL) && (pp->medium == USE_PTYS))
	        {if (PC_original_terminal_state != NULL)
		    PC_set_raw_state(0, FALSE);
		 cp->in  = -1;
		 cp->out = -1;};

	     cp->release(cp);
	     cp = NULL;

	     break;};

#else

    pp = NULL;

#endif

    argv[0] = av;

    return(pp);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_LATCH_CHILD - latch up with the child
 *                 - this is to prevent the child from running to 
 *                 - completion before the parent even has a chance to
 *                 - register the child so as to be able to catch the
 *                 - SIGCHLD that the child will send when exiting
 */

static void _PC_latch_child(pp)
   PROCESS *pp;
   {int in, out, medium, nw;
    char exch[10];

    in     = pp->in;
    out    = pp->out;
    medium = pp->medium;

#ifdef NO_LATCH_PTYS
    if (medium == USE_PTYS)
       return;
#endif

    exch[0] = 'A';
    exch[1] = '\n';
    exch[2] = '\0';

    nw = -1;
    while (nw <= 0)
       {PC_unblock_fd(out);
	nw = write(out, exch, 2);};
    if (nw != 2)
       PC_error("PARENT WRITE FAILED (%d) - _PC_LATCH_CHILD\n", errno);

/* set the alarm */
    PC_alarm(5, NULL);

    PC_block_fd(in);
    while (exch[0] != 'B')
       read(in, exch, 1);

    read(in, exch, 1);
    PC_unblock_fd(in);

/* a PTY will have converted '\n' into '\n''\r' */
    if (medium == USE_PTYS)
       read(in, exch, 1);

/* reset the alarm */
    PC_alarm(0, NULL);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_MANAGE_PROCESS - record the process in the list
 *                    - of remembered processes
 */

void _PC_manage_process(pp)
   PROCESS *pp;
   {if (_PC_n_processes >= _PC_max_processes)
       {_PC_max_processes += 10;
        if (_PC_processes == NULL)
           _PC_processes = FMAKE_N(PROCESS *,
			  _PC_max_processes,
                          "_PC_MANAGE_PROCESS:processes");
        else
           REMAKE_N(_PC_processes,
                    PROCESS *,
                    _PC_max_processes);};
    _PC_processes[_PC_n_processes++] = pp;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_DELETE_PROCESS - remove the process from the list
 *                    - of remembered processes
 */

void _PC_delete_process(pp)
   PROCESS *pp;
   {int i;

    for (i = 0; i < _PC_n_processes; i++)
        if (pp == _PC_processes[i])
           {_PC_processes[i] = _PC_processes[--_PC_n_processes];
            break;};

    if ((_PC_n_processes == 0) && (PC_original_terminal_state != NULL))
       {PC_set_term_state(PC_original_terminal_state, -1);
        SFREE(PC_original_terminal_state);};

    PC_rl_io_callback(pp->in);
    PC_rl_io_callback(pp->out);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_RL_PROCESS - release a PROCESS structure */

void _PC_rl_process(pp)
   PROCESS *pp;
   {

    if (pp->vif != NULL)
       _PD_rl_pdb(pp->vif);

    SFREE(pp->spty);

    SFREE(pp->in_ring);
    SFREE(pp);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_INIT_IPC - establish two inter-process communications channels
 *              - use one of three media:
 *              -  1) open an available pseudo terminal, pty
 *              -     connect the child to the master side and
 *              -     the parent to the slave side (thanks to Neale Smith
 *              -     for the explanation of PTY's)
 *              -  2) socket
 *              -  3) pipe
 *              - the input channel should always be unblocked
 *              - return TRUE iff successful
 */

static int _PC_init_ipc(pp, cp)
   PROCESS *pp, *cp;
   {int ret;

    ret = TRUE;

#ifdef HAVE_PROCESS_CONTROL

    switch (pp->medium)
       {case USE_PTYS    :
             {char *pc;
              int c, pty;

              c = ' ';

# ifdef SGI
              pc = _getpty(&pty, O_RDWR | O_NDELAY, 0600, FALSE);
              if (pc != NULL)
                 {cp->spty = SC_strsavef(pc, "char*:_PC_INIT_IPC:pc");

                  cp->in  = pty;
                  cp->out = pty;

                  pp->in  = pty;
                  pp->out = pty;}

              else
                 PC_error("CAN'T OPEN PTY - _PC_INIT_IPC");

# define PTY_OPEN_DEFINED
# endif

#if 0
/* The first pty test times out trying to latch up with the child process */
# ifdef HPUX
	      pty = -1;
	      pty = open("/dev/ptmx", O_RDWR | O_NDELAY);
              if (pty >= 0)
                 {grantpt(pty);
		  unlockpt(pty);
		  pc = (char *) ptsname(pty);

		  cp->spty = SC_strsavef(pc, "char*:_PC_INIT_IPC:spty");

                  cp->in  = pty;
                  cp->out = pty;

                  pp->in  = pty;
                  pp->out = pty;}

              else
                 PC_error("CAN'T OPEN PTY - _PC_INIT_IPC");

# define PTY_OPEN_DEFINED
# endif
#endif

# ifdef AIX
	      pty = -1;
	      pty = open("/dev/ptc", O_RDWR | O_NDELAY);
	      if (pty >= 0)
                 {char bf[MAXLINE];

		  cp->spty = SC_strsavef(SC_ttyname(pty, bf, MAXLINE),
                                         "char *:_PC_INIT_IPC:spty");

		  cp->in  = pty;
		  cp->out = pty;

		  pp->in  = pty;
		  pp->out = pty;}

	      else
		PC_error("CAN'T OPEN PTY - _PC_INIT_IPC");

# define PTY_OPEN_DEFINED
# endif

# ifdef UNICOS
              {char *pd, mpty[MAXLINE], spty[MAXLINE];
               int d;

	       pty = -1;
	       for (pc = PC_MASTER_PTY_LETTERS;
                    ((c = *pc) != '\0') && (pty < 0);
                    pc++)
                   {for (d = 0; d < 256; d++)
                        {sprintf(mpty, "/dev/pty/%03d", d);
                         pty = open(mpty, O_RDWR | O_NDELAY);
                         if (pty >= 0)
                            {sprintf(spty, "/dev/%s%c%03d",
                                     PC_SLAVE_PTY_NAME, c, d);
                  
			     cp->spty = SC_strsavef(spty,
                                        "char*:_PC_INIT_IPC:spty");

			     cp->in  = pty;
			     cp->out = pty;

			     pp->in  = pty;
			     pp->out = pty;

			     break;};};};};

              c = (pty >= 0);

# define PTY_OPEN_DEFINED
# endif

# ifndef PTY_OPEN_DEFINED
              {char *pd, mpty[MAXLINE], spty[MAXLINE];
               int d;

	       pty = -1;
	       for (pc = PC_MASTER_PTY_LETTERS;
                    ((c = *pc) != '\0') && (pty < 0);
                    pc++)
                   {for (pd = PC_MASTER_PTY_DIGITS; (d = *pd) != '\0'; pd++)
                        {sprintf(mpty, "/dev/%s%c%c",
                                 PC_MASTER_PTY_NAME, c, d);
			 pty = open(mpty, O_RDWR | O_NDELAY);
			 if (pty >= 0)
                            {sprintf(spty, "/dev/%s%c%c",
				     PC_SLAVE_PTY_NAME, c, d);
                  
			     cp->spty = SC_strsavef(spty,
                                        "char*:_PC_INIT_IPC:spty");

			     cp->in  = pty;
			     cp->out = pty;

			     pp->in  = pty;
			     pp->out = pty;

			     break;};};};};

# define PTY_OPEN_DEFINED
# endif

              if (c == '\0')
		 PC_error("CAN'T OPEN PTY - _PC_INIT_IPC");};

             break;

        case USE_PIPES   :
             {int ports[2];

              if (pipe(ports) < 0)
                 PC_error("COULDN'T CREATE PIPE #1 - _PC_INIT_IPC");

              pp->in  = ports[0];
              cp->out = ports[1];

              if (pipe(ports) < 0)
                 {close(pp->in);
                  close(cp->out);
                  PC_error("COULDN'T CREATE PIPE #2 - _PC_INIT_IPC");};

              cp->in  = ports[0];
              pp->out = ports[1];

              PC_set_fd_attr(pp->in, O_RDONLY | O_NDELAY, TRUE);
              PC_set_fd_attr(pp->out, O_WRONLY, TRUE);
              PC_set_fd_attr(cp->in, O_RDONLY & ~O_NDELAY, TRUE);
              PC_set_fd_attr(cp->out, O_WRONLY, TRUE);};

              break;

# ifdef HAVE_SOCKETS_P

        case USE_SOCKETS :
             {int ports[2];

              if (socketpair(PF_UNIX, SOCK_STREAM, 0, ports) < 0)
                 PC_error(PC_err,
                          "COULDN'T CREATE SOCKET PAIR #1 - _PC_INIT_IPC");

              pp->in  = ports[0];
              cp->out = ports[1];

              if (socketpair(PF_UNIX, SOCK_STREAM, 0, ports) < 0)
                 {close(pp->in);
                  close(cp->out);
                  PC_error(PC_err,
                           "COULDN'T CREATE SOCKET PAIR #2 - _PC_INIT_IPC");};

              cp->in  = ports[0];
              pp->out = ports[1];

              PC_set_fd_attr(pp->in, O_RDONLY | O_NDELAY, TRUE);
              PC_set_fd_attr(pp->out, O_WRONLY, TRUE);
              PC_set_fd_attr(cp->in, O_RDONLY & ~O_NDELAY, TRUE);
              PC_set_fd_attr(cp->out, O_WRONLY, TRUE);};

             break;

# endif

        default          :
             ret = FALSE;};

#endif

    return(ret);}

/*--------------------------------------------------------------------------*/

#ifdef HAVE_PROCESS_CONTROL

/*--------------------------------------------------------------------------*/

/* _PC_RECORD_WAIT_RESULT - process the aftermath of some variant of
 *                        - a wait system call
 */

static void _PC_record_wait_result(w, pid)
    WAITTYPE w;
    int pid;
    {int i;
     PROCESS *pp;

     for (i = 0; i < _PC_n_processes; i++)
         {pp = _PC_processes[i];
	  if (pp == NULL)
	     {PRINT(stderr, "\n\nPROCESS LIST CORRUPT\n\n");
              PC_block_file(stdin);
	      exit(2);};

	  if (pp->id == pid)
	     {if (WIFSTOPPED(w))
		 {pp->status = STOPPED | CHANGED;
		  pp->reason = WSTOPSIG(w);}

	      else if (WIFEXITED(w))
		 {pp->status = EXITED | CHANGED;
		  if (WCOREDUMP(w))
		     pp->status |= COREDUMPED;
		  pp->reason = WRETCODE(w);}

	      else if (WIFSIGNALED(w))
		 {pp->status = SIGNALED | CHANGED;
		  if (WCOREDUMP(w))
		     pp->status |= COREDUMPED;
		  pp->reason = (WTERMSIG(w) & 0x7F);};

	      break;};};

     if (i >= _PC_n_processes)
        {PRINT(stderr, "\nSIGCHILD FROM UNKNOWN PROCESS - %d\n\n", pid);
         PC_block_file(stdin);
	 exit(2);};

     return;}

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/* PC_SIGNAL_HANDLER - on receipt of a signal that a child status has
 *                   - changed record the new status and the reason for
 *                   - the change
 *                   - BSD wait semantics:
 *                   - loop forever asking about children with
 *                   - changed statuses until the system says there are
 *                   - no more (WAIT does not block and has a return
 *                   - indicating that nobody changed status - signals
 *                   - may be lost so run a tight loop to get them all)
 *                   - SYSV wait semantics:
 *                   - ask about children with changed statuses once only
 *                   - (WAIT blocks but SYSV guarantees that no signals
 *                   - will be lost)
 */

void PC_signal_handler(signo)
   int signo;
   {

#ifdef HAVE_PROCESS_CONTROL

    int pid;
    WAITTYPE w;
    int old_errno = errno;

# ifdef BSD_WAIT

    while (TRUE)
       {pid = WAITING(w);
        if (pid > 0)
           _PC_record_wait_result(w, pid);
        else if (errno == EINTR)
	   errno = 0;
	else
           break;};

# endif

# ifdef SYSV_WAIT

    pid = WAITING(w);
    if (pid > 0)
       _PC_record_wait_result(w, pid);

# endif

    SIGNAL(signo, PC_signal_handler);
    errno = old_errno;

#endif

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_REMOTE_HANDLER - on receipt of a signal that a remote child is done */

static void PC_remote_handler(signo, code)
   int signo, code;
   {
#ifdef HAVE_PROCESS_CONTROL

    int i;
    PROCESS *pp;

    for (i = 0; i < _PC_n_processes; i++)
        {pp = _PC_processes[i];
         if (pp == NULL)
            {PRINT(stderr, "\n\nPROCESS LIST CORRUPT\n\n");
	     PC_block_file(stdin);
             exit(2);};

         if (pp->id == _PC_current_flushed_process)
            {pp->status = EXITED | CHANGED;
             pp->reason = signo;
             break;};};

    _PC_current_flushed_process = -1;
    SIGNAL(signo, PC_remote_handler);

#endif

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_PRINTF - do an fprintf style output to a process */

#ifdef PCC

int PC_printf(pp, fmt, va_alist)
   PROCESS *pp;
   char *fmt;
   va_dcl

#endif

#ifdef ANSI

int PC_printf(PROCESS *pp, char *fmt, ...)

#endif

   {char buffer[LRG_TXT_BUFFER];
    int ret;

    PC_ERR_TRAP(FALSE);

    if (pp == NULL)
       PC_error("BAD PROCESS - PC_PRINTF");

    SC_VA_START(fmt);
    SC_VSPRINTF(buffer, fmt);
    SC_VA_END;

    ret = FALSE;
    if ((pp != NULL) && (pp->printf != NULL))
       ret = pp->printf(pp, buffer);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_GETS - get a string from the specified process
 *         - if none are available or there is an return NULL
 */

char *PC_gets(bf, len, pp)
   char *bf;
   int len;
   PROCESS *pp;

#ifdef HAVE_PROCESS_CONTROL

   {int status, blck, reset, nb;
    char *pbf;

    PC_ERR_TRAP(NULL);

    if (pp == NULL)
       PC_error("BAD PROCESS - PC_GETS");

    blck   = pp->blocking;
    status = pp->in_ready(pp);
    reset  = -1;

/* if unblocked when should be blocked */
    if (status && blck)
       {PC_block_fd(pp->in);
        status = FALSE;
        reset  = FALSE;}

/* if blocked when should be unblocked */
    else if (!status && !blck)
       {PC_unblock_fd(pp->in);
        status = TRUE;
        reset  = TRUE;};

/* if we are unblocked check incoming messages */
    pbf = bf;
    if (status)
       {nb = pp->gets(bf, len, pp);
        if (!_PC_buffer_in(bf, nb, pp))
           PC_error("NOT ENOUGH ROOM IN BUFFER - PC_GETS");

        if (pp->line_mode)
           {if (_PC_complete_messagep(pp))
               pbf = _PC_buffer_out(bf, pp, len, FALSE) ? bf : NULL;
            else
               pbf = NULL;}
        else
           pbf = _PC_buffer_out(bf, pp, len, FALSE) ? bf : NULL;}

/* if we are blocked and there are already messages return the next one */
    else if (_PC_complete_messagep(pp))
       pbf = _PC_buffer_out(bf, pp, len, FALSE) ? bf : NULL;

/* if we are blocked and there are no messages go wait for one with
 * full FGETS semantics (i.e. terminated with \n)
 */
    else
       {while (TRUE)
          {nb = pp->gets(bf, len, pp);
           if (!_PC_buffer_in(bf, nb, pp))
              PC_error("NOT ENOUGH ROOM IN BUFFER - PC_GETS");

            if (_PC_complete_messagep(pp))
               {pbf = _PC_buffer_out(bf, pp, len, FALSE) ? bf : NULL;
                break;};};};

/* reset the blocking to its state upon entry */
    switch (reset)
       {case 0 : PC_unblock_fd(pp->in);
                 break;
        case 1 : PC_block_fd(pp->in);
                 break;};

    return(pbf);}

#else

   {return(bf);}

#endif

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_COMPLETE_MESSAGEP - return TRUE iff there is a:
 *                       -   \n terminated string
 *                       -   EOF
 *                       - in the PROCESS ring buffer
 */

static int _PC_complete_messagep(pp)
   PROCESS *pp;
   {unsigned int ob, ls;
    unsigned char *ring, c;

    ob   = pp->ob_in;
    ring = pp->in_ring;    
    ls   = pp->line_sep;

    while ((c = ring[ob]) != '\0')
       {ob = (ob + 1) & SIZE_MASK;
        if ((c == ls) || (c == (unsigned char) EOF))
           return(TRUE);};

    return(FALSE);}

/*--------------------------------------------------------------------------*/

#if 0

/*--------------------------------------------------------------------------*/

/* _PC_BUFFER_MESSAGE - put new messages on the ring (if any) and pop the
 *                    - oldest message off the ring (if any)
 *                    - the return message is to be passed in BF and the
 *                    - incoming messages which are in bf are pointed to by
 *                    - PBF
 *                    - the pointer to BF is returned
 *                    - or NULL if there are no messages
 *                    - if OUTF is TRUE return text upto the first newline
 */

static char *_PC_buffer_message(bf, pbf, pp, outf)
   char *bf, *pbf;
   PROCESS *pp;
   int outf;
   {unsigned int ib, ob, ls;
    unsigned char *ring, c;

/* buffer the traffic */
    ib   = pp->ib_in;
    ob   = pp->ob_in;
    ring = pp->in_ring;    
    ls   = pp->line_sep;

/* copy the input to the ring */
    if (pbf != NULL)
       {while ((c = *pbf++) != '\0')
           {if (((c <= 0x1F) ||
                 (0x7F <= c)) &&
                (c != ls))
               continue;

            ring[ib] = c;
            ib       = (ib + 1) & SIZE_MASK;};

        pp->ib_in = ib;};

    if ((ib == ob) || (outf == FALSE))
       return(NULL);

/* copy the output to the buffer */
    pbf = bf;
    while ((c = ring[ob]) != '\0')
       {ring[ob] = '\0';
        ob       = (ob + 1) & SIZE_MASK;
        *pbf++   = c;
        if (c == ls)
           break;};

    pp->ob_in = ob;

    if (bf == pbf)
       {bf[0] = '\0';
        bf    = NULL;}
    else
       *pbf++ = '\0';

    return(bf);}

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/* _PC_BUFFER_IN - put new bytes on the ring
 *               - everything must go into the buffer
 *               - the routine that extracts from the buffer will
 *               - decide whether ASCII or BINARY information is expected
 *               - return TRUE iff there is enough room to hold the bytes
 */

int _PC_buffer_in(bf, n, pp)
   char *bf;
   int n;
   PROCESS *pp;
   {int i;
    unsigned int ib, ob, ab;
    unsigned char *ring;

    ib   = pp->ib_in;
    ob   = pp->ob_in;
    ring = pp->in_ring;    

    if (ib >= ob)
       ab = SIZE_BUF - ib + ob;
    else
       ab = ob - ib;

    if (ab < n)
       return(FALSE);

    if (bf != NULL)
       {for (i = 0; i < n; i++)
            {ring[ib] = *bf++;
             ib       = (ib + 1) & SIZE_MASK;};

        pp->ib_in = ib;};

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PC_BUFFER_OUT - return the oldest message from the ring in BF
 */

int _PC_buffer_out(bf, pp, nb, binf)
   char *bf;
   PROCESS *pp;
   int nb, binf;
   {int i;
    unsigned int ib, ob, ls;
    unsigned char *ring, c;
    char *pbf;

/* buffer the traffic */
    ib   = pp->ib_in;
    ob   = pp->ob_in;
    ring = pp->in_ring;    
    ls   = pp->line_sep;

    pbf = bf;

/* copy binary data out */
    if (binf)
       {for (i = 0; (ob != ib) && (i < nb); i++, ob = (ob + 1) & SIZE_MASK)
            {*pbf++   = ring[ob];
             ring[ob] = '\0';};}

/* copy ascii data out */
    else
       {for (i = 0; (ob != ib) && (i < nb); i++, ob = (ob + 1) & SIZE_MASK)
            {c = ring[ob];
             ring[ob] = '\0';

             if (((c <= 0x1F) || (0x7F <= c)) &&
                 (c != ls) &&                      /* line feed */
                 (c != '\f') &&
                 (c != '\r') &&
                 (c != '\t'))
                continue;

             *pbf++ = c;
             if (c == ls)
                {ob++;
                 i++;
                 break;};};

        if (bf == pbf)
           bf[0] = '\0';

        else if (i < nb)
           *pbf++ = '\0';};


    pp->ob_in = ob;

    return(i);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_BUFFER_DATA_IN - read from the data connection of the PROCESS
 *                   - and save them in an internal buffer
 */

int PC_buffer_data_in(pp)
   PROCESS *pp;
   {int fd, nbr, nbs;
    unsigned long nb, nx, nbt;
    char s[DATA_BLOCK], *bf;

    fd = pp->data;
    bf = pp->data_buffer;
    nb = pp->nb_data;
    nx = pp->nx_data;
    if (bf == NULL)
       {nx = DATA_BLOCK;
	nb = 0L;
	bf = FMAKE_N(char, nx, "PC_BUFFER_DATA_IN:bf");};

    PC_unblock_fd(fd);

    nbs = 0;
    while (TRUE)
       {nbr = read(fd, s, DATA_BLOCK);
        if (nbr < 0)
	   break;

        nbt = nb + nbr;
        nbs += nbr;

        if (nbt > nx)
           {nx += DATA_BLOCK;
	    REMAKE_N(bf, char, nx);};

	memcpy(bf+nb, s, nbr);
	nb = nbt;};

    pp->data_buffer = bf;
    pp->nb_data     = nb;
    pp->nx_data     = nx;

    return(nbs > -1 ? nbs : -errno);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PC_BUFFER_DATA_OUT - get data from the data connection of the given
 *                    - process including any buffered data
 *                    - and put it NBI bytes in BF
 */

int PC_buffer_data_out(pp, bf, nbi, block_state)
   PROCESS *pp;
   char *bf;
   int nbi, block_state;
   {int nbe, nb0, fd;
    char *pbf, *dbf;

    pbf = bf;
    fd  = pp->data;
    dbf = pp->data_buffer;    
    nbe = pp->nb_data;
    nb0 = nbi;
    if (nbe < nbi)
       {memcpy(pbf, dbf, nbe);
        pbf += nbe;
	nbi -= nbe;
        memset(dbf, 0, nbe);
        pp->nb_data = 0;

        if (block_state)
	   PC_block_fd(fd);
        else
	   PC_unblock_fd(fd);
	    
	while (nbi > 0)
	   {nbe = read(fd, pbf, nbi);
	    if (nbe < 0)
	       break;
	    pbf += nbe;
	    nbi -= nbe;};}

    else
       {memcpy(pbf, dbf, nbi);
        pp->nb_data = nbe - nbi;
        memcpy(dbf, dbf+nbi, pp->nb_data);};

    return(nb0 - nbi);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
