/* libdpkg - Debian packaging suite library routines
 * vercmp.c - comparison of version numbers
 *
 * Cut to shreds and altered for Quinn Diff.
 *
 * Copyright (C) 1995 Ian Jackson <iwj10@cus.cam.ac.uk>
 * Copyright (C) 1997, 1998 James Troup <james@nocrew.org>
 *
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with dpkg; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <ctype.h>
#include <string.h>
#include "common.h"
#include "error.h"
#include "xmalloc.h"
#include "nfmalloc.h"
#include "vercmp.h"
#include "main.h"

struct versionrevision primary_version;
struct versionrevision secondary_version;

static int verrevcmp(const char *val, const char *ref) 
{
  int vc, rc;
  long vl, rl;
  const char *vp, *rp;
  
  if (!val) val= "";
  if (!ref) ref= "";
  for (;;) {
    vp= val;  while (*vp && !isdigit(*vp)) vp++;
    rp= ref;  while (*rp && !isdigit(*rp)) rp++;
    for (;;) {
      vc= val == vp ? 0 : *val++;
      rc= ref == rp ? 0 : *ref++;
      if (!rc && !vc) break;
      if (vc && !isalpha(vc)) vc += 256; /* assumes ASCII character set */
      if (rc && !isalpha(rc)) rc += 256;
      if (vc != rc) return vc - rc;
    }
    val= vp;
    ref= rp;
    vl=0;  if (isdigit(*vp)) vl= strtol(val,(char**)&val,10);
    rl=0;  if (isdigit(*rp)) rl= strtol(ref,(char**)&ref,10);
    if (vl != rl) return vl - rl;
    if (!*val && !*ref) return 0;
    if (!*val) return -1;
    if (!*ref) return +1;
  }
}

int versioncompare(const struct versionrevision *version,
                   const struct versionrevision *refversion) {
  int r;
  
  if (version->epoch > refversion->epoch) return 1;
  if (version->epoch < refversion->epoch) return -1;
  r= verrevcmp(version->version,refversion->version);  if (r) return r;
  return verrevcmp(version->revision,refversion->revision);
}

const char *parseversion(struct versionrevision *rversion, const char *string) {
  char *hyphen, *colon, *eepochcolon;
  unsigned long epoch;

  if (!*string) return ("version string is empty");
  
  colon= strchr(string,':');
  if (colon) {
    epoch= strtoul(string,&eepochcolon,10);
    if (colon != eepochcolon) return ("epoch in version is not number");
    if (!*++colon) return ("nothing after colon in version number");
    string= colon;
    rversion->epoch= epoch;
  } else {
    rversion->epoch= 0;
  }
  rversion->version= nfstrsave(string);
  hyphen= strrchr(rversion->version,'-');
  if (hyphen) *hyphen++= 0;
  rversion->revision= hyphen ? hyphen : nfstrsave("");
  
  return 0;
}

/* ======= End of iwj's stuff, start of my rubbish ====== */

int
compare_versions (const char *primary_version_string,
		  const secondary_package_info *secondary) /* RUARI: bad name choice */
{
  const char *error_msg;

  error_msg = parseversion (&primary_version, primary_version_string);
  if (error_msg) 
    fubar (NONSYS, "parseversion of %s (primary): %s", secondary->name, error_msg); 
   
  error_msg = parseversion (&secondary_version, secondary->version);
  if (error_msg) 
    fubar (NONSYS, "parseversion of %s (secondary): %s", secondary->name, error_msg); 
   
  debug (debug_versions, 
	 "comparing %s: [Primary] %ld:%s-%s (was %s) versus [Secondary] %ld:%s-%s (was %s), result is %d", 
	 secondary->name, 
	 primary_version.epoch, primary_version.version, primary_version.revision,
	 primary_version_string,
	 secondary_version.epoch, secondary_version.version, secondary_version.revision, 
	 secondary->version, 
	 versioncompare (&secondary_version, &primary_version));
 
  return (versioncompare (&secondary_version, &primary_version));
}
