#
# TCL Library for tkCVS
#

#
# $Id: logcanvas.tcl,v 1.13.2.6 1998/09/14 04:52:30 dorothyr Exp $
# 
# Contains procedures used for the log canvas for tkCVS.
#

#
# Globals used in drawing canvases
#

# Height and width to draw boxes
set cvscanv(boxx) 60
set cvscanv(boxy) 30
# Gaps between boxes
set cvscanv(gapx) [expr $cvscanv(boxx) + 100]
set cvscanv(gapy) [expr $cvscanv(boxy) + 10]
# Indent at top left of canvas
set cvscanv(indx) 20
set cvscanv(indy) 20
# Static type variables used while drawing on the canvas.
set cvscanv(xhigh) [expr $cvscanv(boxx) + $cvscanv(indx)]
set cvscanv(yhigh) [expr $cvscanv(boxy) + $cvscanv(indy)]
set cvscanv(xlow) 0
set cvscanv(ylow) 0
set cvscanv(maxy) $cvscanv(indy)


proc new_logcanvas {localfile filelog} {
  #
  # Creates a new log canvas.  filelog must be the output of a cvs
  # log or rlog command.  If localfile is not "no file" then it is
  # the file name in the local directory that this applies to.
  #
  global cvs
  global revdate
  global revwho
  global revcomment
  global revlist
  global cvscanv
  global tags

  if [info exists revlist] { unset revlist }
  if [info exists revdate] { unset revdate }
  if [info exists revwho] { unset revwho }
  if [info exists revcomment] { unset revcomment }
  if [info exists tags] { unset tags }

  static {canvasnum 0}

  # Make the canvas

  incr canvasnum
  set logcanvas ".logcanvas$canvasnum"
  toplevel $logcanvas

  frame $logcanvas.up -relief groove -border 2
  frame $logcanvas.up.left
  frame $logcanvas.up.right
  frame $logcanvas.up1 -relief groove -border 2
  frame $logcanvas.up1.left
  frame $logcanvas.up1.right
  frame $logcanvas.up2 -relief groove -border 2
  frame $logcanvas.up2.left
  frame $logcanvas.up2.right
  frame $logcanvas.down -relief groove -border 2

  pack $logcanvas.up -side top -fill x
  pack $logcanvas.up.left -side left -fill both
  pack $logcanvas.up.right -side left -fill both -expand 1
  pack $logcanvas.up1 -side top -fill x
  pack $logcanvas.up1.left -side left -fill both
  pack $logcanvas.up1.right -side left -fill both -expand 1
  pack $logcanvas.up2 -side top -fill x
  pack $logcanvas.up2.left -side left -fill both
  pack $logcanvas.up2.right -side left -fill both -expand 1
  pack $logcanvas.down -side bottom -fill x

  label $logcanvas.lfname -anchor w -text "RCS File Name"
  entry $logcanvas.tfname
  # This is a hidden entry that stores the local file name.
  entry $logcanvas.tlocalfile
  $logcanvas.tlocalfile delete 0 end
  $logcanvas.tlocalfile insert end $localfile

  label $logcanvas.lvers1 -anchor w -text "Revision A"
  label $logcanvas.lwho1 -anchor w -text "Committed by"
  label $logcanvas.ldate1 -anchor w -text "Date"
  label $logcanvas.lcomment1 -anchor w -text "Log"

  entry $logcanvas.tvers1 -relief sunken
  label $logcanvas.twho1 -anchor w -text "--"
  label $logcanvas.tdate1 -anchor w -text "--"
  text  $logcanvas.tcomment1 -height 5 -width 75

  label $logcanvas.lvers2 -anchor w -text "Revision B"
  label $logcanvas.lwho2 -anchor w -text "Committed by"
  label $logcanvas.ldate2 -anchor w -text "Date"
  label $logcanvas.lcomment2 -anchor w -text "Log"

  entry $logcanvas.tvers2 -relief sunken
  label $logcanvas.twho2 -anchor w -text "--"
  label $logcanvas.tdate2 -anchor w -text "--"
  text  $logcanvas.tcomment2 -height 5 -width 75

  pack $logcanvas.lfname \
    -in $logcanvas.up.left \
    -side top -fill x -pady 2
  pack $logcanvas.tfname \
    -in $logcanvas.up.right \
    -side top -fill x -pady 0
  pack $logcanvas.lvers1 $logcanvas.lwho1 \
      $logcanvas.ldate1 $logcanvas.lcomment1 \
    -in $logcanvas.up1.left \
    -side top -fill x -pady 0
  pack $logcanvas.tvers1 $logcanvas.twho1 \
      $logcanvas.tdate1 $logcanvas.tcomment1 \
    -in $logcanvas.up1.right \
    -side top -fill x -pady 0
  pack $logcanvas.lvers2 $logcanvas.lwho2 \
      $logcanvas.ldate2 $logcanvas.lcomment2 \
    -in $logcanvas.up2.left \
    -side top -fill x -pady 0
  pack $logcanvas.tvers2 $logcanvas.twho2 \
      $logcanvas.tdate2 $logcanvas.tcomment2 \
    -in $logcanvas.up2.right \
    -side top -fill x -pady 0

  canvas $logcanvas.canvas -relief sunken -border 2 \
    -yscrollcommand "$logcanvas.yscroll set" \
    -xscrollcommand "$logcanvas.xscroll set"
  scrollbar $logcanvas.xscroll -relief sunken -orient horizontal \
    -command "$logcanvas.canvas xview"
  scrollbar $logcanvas.yscroll -relief sunken \
    -command "$logcanvas.canvas yview"

  #
  # Create buttons
  #
  button $logcanvas.help -text "Help" \
    -command log_browser
  button $logcanvas.view -text "View" \
    -command "logcanvas_view $logcanvas"
  button $logcanvas.diff -text "Diff" \
    -command "logcanvas_diff $logcanvas"
  pack $logcanvas.help $logcanvas.view $logcanvas.diff \
    -in $logcanvas.down -side left \
    -ipadx 2 -ipady 2 -padx 4 -pady 4 -fill both -expand 1
  if {$localfile != "no file"} {
    button $logcanvas.join -text "Merge Branch to Head" \
      -command "logcanvas_join $localfile $logcanvas"
    button $logcanvas.delta -text "Merge Changes to Head" \
      -command "logcanvas_delta $localfile $logcanvas"
    pack $logcanvas.join $logcanvas.delta \
      -in $logcanvas.down -side left \
      -ipadx 2 -ipady 2 -padx 4 -pady 4 -fill both -expand 1
  }
  button $logcanvas.viewtags -text "View Tags" \
    -command "nop"
  button $logcanvas.quit -text "Quit" \
    -command "destroy $logcanvas"
  pack $logcanvas.viewtags $logcanvas.quit \
    -in $logcanvas.down -side left \
    -ipadx 2 -ipady 2 -padx 4 -pady 4 -fill both -expand 1

  #
  # Put the canvas on to the display.
  #
  pack $logcanvas.xscroll -side bottom -fill x -padx 2 -pady 2
  pack $logcanvas.yscroll -side right -fill y -padx 2 -pady 2
  pack $logcanvas.canvas -fill both -expand 1

  logcanvas_clear $logcanvas

  #
  # Window manager stuff.
  #
  wm title $logcanvas "CVS Log Browser"
  wm minsize $logcanvas 1 1

  # Collect the history from the RCS file
  parse_vfile $logcanvas $filelog

  # Sort it into order - this makes drawing the tree much easier
  set revlist [lsort -command sortrevs [array names revdate]]

  # Now draw the revision tree
  set n 0
  foreach rev $revlist {
    set revprev [lindex $revlist [expr $n-1]]
    logcanvas_draw_box $logcanvas \
        $rev $revwho($rev) $revdate($rev) $revcomment($rev) $revprev
    incr n
  }
}

proc sortrevs {a b} {
  #
  # Proc for lsort -command, to sort revision numbers
  #
  set alist [split $a "."]
  set blist [split $b "."]
  set alength [llength $alist]
  set blength [llength $blist]
  set shorter [expr {($alength < $blength) ? $alength : $blength}]

  # As soon as a field is greater than its counterpart, return
  for {set i 0} {$i <= $shorter} {incr i} {
    set A($i) [lrange $alist $i $i]
    set B($i) [lrange $blist $i $i]
      if {$A($i) < $B($i)} { return -1}
      if {$A($i) > $B($i)} { return 1}
  }
  # Tie breaker needed
  if {$alength < $blength} {
    return -1
  } elseif {$alength > $blength} {
    return 1
  } else {
    return 0
  }
}

proc logcanvas_clear {logcanvas} {
  #
  # Clears the canvas and resets globals ready to re-draw.
  #
  global cvscanv

  set cvscanv(branches) {}
  set cvscanv(xhigh) [expr $cvscanv(boxx) + $cvscanv(indx)]
  set cvscanv(yhigh) [expr $cvscanv(boxy) + $cvscanv(indy)]
  set cvscanv(maxy) $cvscanv(indy)
  $logcanvas.canvas delete all
}

proc logcanvas_colours {revnum} {
  #
  # Determines what colour the box should be.
  #
  global cvscfg
  global tags

  set colours {black}

  if [info exists tags($revnum)] {
    foreach tag $tags($revnum) {
      if [info exists cvscfg(boxcolour,$tag)] {
        set colours $cvscfg(boxcolour,$tag)
      }
      if [info exists cvscfg(dotcolour,$tag)] {
        lappend colours $cvscfg(dotcolour,$tag)
      }
    }
  }
  return $colours
}

proc logcanvas_draw_box {logcanvas \
        revnum revwho revdate revcomment revprev} {
  #
  # Draws a box containing a revision of a file.
  #
  global tags
  global cvscanv
  global revlist

  set versions [split $revnum "."]
  set branchdepth [llength $versions]
  set prevdepth [llength [split $revprev "."]]
  #puts -nonewline "$revnum   "
  #puts "branchdepth $branchdepth  prevdepth $prevdepth"

  # Default - if theres no previous revision, put it at the beginning index
  set parent ""
  set y $cvscanv(indy)
  set x $cvscanv(indx)

  # Else, find the parent and place this one after it
  if [info exists cvscanv(posy$revprev)] {
    set leaf [lindex $versions [expr $branchdepth - 1]]
    set branch [join [lrange $versions 0 [expr $branchdepth - 2]] "."]
    set branchroot [join [lrange $versions 0 [expr $branchdepth - 3]] "."]

    set prevversions [split $revprev "."]
    set prevbranch \
        [join [lrange $prevversions 0 [expr $prevdepth - 2]] "."]
    set prevbranchroot \
        [join [lrange $prevversions 0 [expr $prevdepth - 3]] "."]
    # Set a default unless certain conditions are met
    set parent $revprev

    if {$branchdepth < $prevdepth} {
    # Here we have come to the end of a branch and have to find the
    # trunk again.
        if {[string compare $branch $prevbranch] != 0} {
        set shortind [expr [lsearch -exact $revlist $revnum] - 1]
        set shortlist  [lrange $revlist 0 $shortind]
        set shortlist [lsort -dictionary -decreasing $shortlist]

        # Search for the forerunner on the trunk.  Remember that
        # numbers may be missing from the sequence.
        set match 0
        incr leaf -1
        #puts "trying to match $branch.$leaf"
        #puts "  from $shortlist"
        while {$match == 0} {
          foreach item $shortlist {
              if {[string compare $item "$branch.$leaf"] == 0} {
              set parent $item
              set match 1
              continue
            }
          }
          incr leaf -1
        }
      } else {
        set parent $prevbranchroot
      }
      set x $cvscanv(posx$parent)
    } elseif { $branchdepth == $prevdepth } {
      # If it's on the main trunk, it doesn't need any special processing
      if {$branchdepth == 2} {
          set parent $revprev
      } else {
        set x $cvscanv(posx$parent)
        set prevversions [split $revprev "."]
        set prevparent \
            [join [lrange $prevversions 0 [expr $prevdepth - 3]] "."]
        # If theres another branch from the same root,
        # make sure the parent is correct
        if {$branch != $prevbranch} {
          # Move it over with respect to the (incorrect) parent before
          # setting the new parent
          set x [expr $x + $cvscanv(gapx)]
          set parent $prevparent
        }
      }
    } else {
      # $branchdepth > $prevdepth is the only remaining possiblity
      set parent [join [lrange $versions 0 [expr $branchdepth - 3]] "."]
      set x [expr $cvscanv(posx$parent) + $cvscanv(gapx)]
    }
    set y [expr $cvscanv(posy$parent) - $cvscanv(gapy)]
  }
  #puts " parent $parent"

  # If the parent has more than two tags, it needs more vertical space
  set extratags 0
  if [info exists tags($parent)] {
    set ntags [llength $tags($parent)]
    if {$ntags > 2} {
      set extratags [expr $ntags - 2]
    }
  }
  set y [expr $y - (12 * $extratags)]

  # Maybe theres a tall branch already occupying this position - better check
  if {$branchdepth > 2} {
    if {$y < $cvscanv(maxy)} { set cvscanv(maxy) $y }
    if {$branchdepth > $prevdepth} {
      if {$y > $cvscanv(maxy)} {
        set x [expr $x + $cvscanv(gapx)] 
      }
    }
  }

  # draw the box and remember its position
  logcanvas_rectangle $logcanvas \
      $revnum $revwho $revdate $revcomment $x $y $parent
  set cvscanv(posx$revnum) $x
  set cvscanv(posy$revnum) $y
}

proc logcanvas_rectangle {logcanvas \
        revnum revwho revdate revcomment x y revprev} {
  #
  # Breaks out some of the code from the logcanvas_draw_box procedure.
  #
  global cvscanv
  global tags

  set versions [split $revnum "."]
  set prevversions [split $revprev "."]

  # draw the box
  set colours [logcanvas_colours $revnum]
  set boxcolour [lindex $colours 0]
  set dotcolours [lrange $colours 1 end]
  $logcanvas.canvas create rectangle \
    $x $y [expr $x + $cvscanv(boxx)] [expr $y + $cvscanv(boxy)] \
    -width 3 \
    -outline $boxcolour \
    -tags v$revnum

  # draw connecting line
  if [info exists cvscanv(posx$revprev)] {
    if {[llength $versions] > [llength $prevversions]} {
      set xbegin [expr $cvscanv(posx$revprev) + $cvscanv(boxx)]
      set ybegin [expr $cvscanv(posy$revprev) + ($cvscanv(boxy)/2)]
    } else {
      set xbegin [expr $cvscanv(posx$revprev) + ($cvscanv(boxx)/2)]
      set ybegin $cvscanv(posy$revprev)
    }
    set xend [expr $x + ($cvscanv(boxx)/2)]
    set yend [expr $y + $cvscanv(boxy)]
    $logcanvas.canvas create line $xbegin $ybegin $xend $yend
  }

  # Make sure the scrolling region is big enough
  if {$cvscanv(xlow) > $x} {
    set cvscanv(xlow) $x
  }
  if {$cvscanv(ylow) > $y} {
    set cvscanv(ylow) $y
  }
  if {$cvscanv(xhigh) < [expr $x + $cvscanv(boxx)]} {
    set cvscanv(xhigh) [expr $x + $cvscanv(boxx) + 10]
  }
  if {$cvscanv(yhigh) < [expr $y + $cvscanv(boxy)]} {
    set cvscanv(yhigh) [expr $y + $cvscanv(boxy) + 10]
  }
  $logcanvas.canvas configure \
    -scrollregion "$cvscanv(xlow) $cvscanv(ylow) \
    $cvscanv(xhigh) $cvscanv(yhigh)"

  # Put the version number in the box
  $logcanvas.canvas create text \
    [expr $x + 4] [expr $y + 2] \
    -anchor nw -text $revnum  \
    -fill $boxcolour \
    -tags v$revnum
  $logcanvas.canvas create text \
    [expr $x + 4] [expr $y + 14] \
    -anchor nw -text $revwho \
    -fill $boxcolour \
    -tags v$revnum

  if [info exists tags($revnum)] {
    set n 0
    foreach item $tags($revnum) {
      $logcanvas.canvas create text \
        [expr $x + $cvscanv(boxx) + 2] [expr $y + $cvscanv(boxy) - $n] \
        -anchor sw -text $item \
        -fill $boxcolour \
        -tags v$revnum
      set n [expr $n + 12]
    }
  }

  # Put some dots in the box if necessary.
  set dotnum 0
  foreach dotcolour $dotcolours {
    set dotx [expr $x + $cvscanv(boxx) - 8]
    set doty [expr $dotnum * 6 + $y + 3]
    $logcanvas.canvas create oval \
      $dotx $doty [expr $dotx + 5] [expr $doty + 5] \
      -fill $dotcolour \
      -outline ""
    incr dotnum
  }

  # Bind to the tag.
  $logcanvas.canvas bind v$revnum <ButtonPress-1> \
    "$logcanvas.tvers1 delete 0 end
    $logcanvas.tvers1 insert end $revnum
    $logcanvas.twho1 configure -text $revwho
    $logcanvas.tdate1 configure -text $revdate
    $logcanvas.tcomment1 delete 1.0 end
    $logcanvas.tcomment1 insert end {$revcomment}"
  $logcanvas.canvas bind v$revnum <ButtonPress-3> \
    "$logcanvas.tvers2 delete 0 end
    $logcanvas.tvers2 insert end $revnum
    $logcanvas.twho2 configure -text $revwho
    $logcanvas.tdate2 configure -text $revdate
    $logcanvas.tcomment2 delete 1.0 end
    $logcanvas.tcomment2 insert end {$revcomment}"
}

proc logcanvas_view {logcanvas} {
  #
  # Views the selected version.
  #
  set ver1 [$logcanvas.tvers1 get]
  set localfile [$logcanvas.tlocalfile get]

  if {$localfile != "no file"} {
    cvs_view_r $ver1 $localfile
  } else {
    set fname [$logcanvas.tfname get]
    rcs_fileview $fname $ver1
  }
}

proc logcanvas_diff {logcanvas} {
  #
  # Diffs two versions.
  #
  set ver1 [$logcanvas.tvers1 get]
  set ver2 [$logcanvas.tvers2 get]
  set localfile [$logcanvas.tlocalfile get]

  if {$localfile != "no file"} {
    cvs_diff_r $ver1 $ver2 $localfile
  } else {
    set fname [$logcanvas.tfname get]
    rcs_filediff $fname $ver1 $ver2
  }
}

proc logcanvas_join {localfile logcanvas} {
  #
  # Joins a branch version to the head version.
  #
  set ver1 [$logcanvas.tvers1 get]
  set versions [split $ver1 "."]
  set branchdepth [llength $versions]
  if {$branchdepth < 4} {
    cvserror "Please select a branch version for this function!"
    return 1
  }

  cvs_join $localfile $ver1
}

proc logcanvas_delta {localfile logcanvas} {
  #
  # Merges changes in the delta between two versions to the head
  # version.
  #
  set ver1 [$logcanvas.tvers1 get]
  set ver2 [$logcanvas.tvers2 get]

  cvs_delta $localfile $ver1 $ver2
}

proc parse_vfile {logcanvas filelog} {
  #
  # Splits the rcs file up and parses it using a simple state machine.
  #
  global revdate
  global revwho
  global revcomment
  global tags

  set loglist [split $filelog "\n"]
  set logstate "searching"
  set logstate "rcsfile"
  foreach logline $loglist {
    switch -exact -- $logstate {
      "rcsfile" {
        # Look for the first text line which should give the file name.
        set fileline [split $logline]
        if {[lindex $fileline 0] == "RCS"} {
          $logcanvas.tfname delete 0 end
          $logcanvas.tfname insert end [lindex $fileline 2]
          set logstate "tags"
          set taglist ""
          continue
        }
      }
      "tags" {
        # Any line with a tab leader is a tag
        if { [string index $logline 0] == "\t" } {
          set taglist "$taglist$logline\n"
          set tagitems [split $logline ":"]
          set tagstring [string trim [lindex $tagitems 0]]
          set tagrevision [string trim [lindex $tagitems 1]]
          lappend tags($tagrevision) $tagstring
        } else {
          if {$logline == "description:"} {
            # No more tags after this point
            $logcanvas.viewtags configure \
              -command "view_output Tags \"$taglist\""
            set logstate "searching"
            continue
          }
          if {$logline == "----------------------------"} {
            # Oops, missed something.
            $logcanvas.viewtags configure \
              -command "view_output Tags \"$taglist\""
            set logstate "revision"
            continue
          }
        }
      }
      "searching" {
        # Look for the line that starts a revision message.
        if {$logline == "----------------------------"} {
          set logstate "revision"
          continue
        }
      }
      "revision" {
        # Look for a revision number line
        set revline [split $logline]
        set revnum [lindex $revline 1]
        set logstate "date"
      }
      "date" {
        # Look for a date line.  This also has the name of the author.
        set dateline [split $logline]
        set revdate($revnum) [lindex $dateline 1]

        set who  [lindex $dateline 5]
        set revwho($revnum) [string range $who 0 \
                     [expr [string length $who] - 2]]

        set revcomment($revnum) ""
        set logstate "logmessage"
      }
      "logmessage" {
        # Read the log message which follows the date line.
        if {$logline == "----------------------------"} {
          set logstate "revision"
          continue
        } elseif {$logline == "============================================================================="} {
          set logstate "terminated"
          continue
        }
        # Process a revision log line
        regsub -all "\"" $logline "'" newline
        set revcomment($revnum) "$revcomment($revnum)$newline\n"
      }
      "terminated" {
        # ignore any further lines
        continue
      }
    }
  }
}
