/*
  gsumi version 0.5

  Copyright 1997 Owen Taylor <owt1@cornell.edu>

  gsumi is heavily based upon:

  xink version 0.02

  Copyright 1997 Raph Levien <raph@acm.org>

  This code is free for commercial and non-commercial use or
  redistribution, as long as the source code release, startup screen,
  or product packaging includes this copyright notice.
*/

#include <stdlib.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>

#include "gsumi.h"

#undef DEBUG_SELECT

AreaInfo info;

ToolInfo *tools[MAX_TOOLS];
int num_tools = 0;
guint32 current_deviceid;

GdkColorContext *color_context;

/* Initialize the gray_pixels[] and red_pixel values, allocating colormap
   entries if need be. */
static void
color_init(int ngrays)
{
  int i;
  GdkVisual *system_visual = gdk_visual_get_system();

  gushort vals[ngrays];
  gulong reduced[ngrays];
  gushort red, green, blue;

  gint nallocated = 0;
  
  color_context = gdk_color_context_new (system_visual,
					 gdk_colormap_get_system());

  switch (system_visual->depth) {
  case 8:
  case 16:
  case 24:
  case 32:
    for (i = 0; i < ngrays; i++) {
      vals[i] = (i * 0xffff) / (ngrays - 1);
      reduced[i] = 0;
    }
    
    gdk_color_context_get_pixels (color_context, vals, vals, vals, ngrays,
				  reduced, &nallocated);

    if (nallocated != ngrays)
      g_warning("Could only allocate %d grays, colors may be incorrect",
		nallocated);

    for (i = 0; i < 256; i++)
      gray_pixels[i] = reduced[(i * (ngrays - 1)) / 255];

    break;

  default:
    g_error("Can't handle %d bpp pixel displays.\n", system_visual->depth);
    break;
  }

  red = 0xffff; green = blue = 0;
  nallocated = 0;
  gdk_color_context_get_pixels (color_context, &red, &green, &blue, 1,
				&red_pixel, &nallocated);

  if (!nallocated)
    g_warning("Could not allocate red color");

  red = 0x7fff; green = blue = 0;
  nallocated = 0;
  gdk_color_context_get_pixels (color_context, &red, &green, &blue, 1,
				&maroon_pixel, &nallocated);

  if (!nallocated)
    g_warning("Could not allocate maroon color");
}

/*
 * Event callbacks for drawing area
 */

static gdouble
translate_pressure (gdouble p)
{
  int pi;
  double frac;
  
  p *= 31;
  pi = p;
  frac = pi - p;

  if (pi == 31)
    return info.tool->profile[pi];
  else
    return (frac)*info.tool->profile[pi+1] + (1-frac)*info.tool->profile[pi];
}

static int
area_expose_event (GtkWidget      *widget,
		   GdkEventExpose *event)
{
  rect rec;

  create_ximage(widget, widget->allocation.width, widget->allocation.height);

  rec.x0 = event->area.x;
  rec.y0 = event->area.y;
  rec.x1 = event->area.x + event->area.width;
  rec.y1 = event->area.y + event->area.height;
  if (rec.x1 > image_width)
    rec.x1 = image_width;
  if (rec.y1 > image_width)
    rec.y1 = image_width;
  expose_rect (& rec);

  return FALSE;
}

static void
area_configure_event (GtkWidget *w,
		      GdkEventConfigure event,
		      gpointer data)
{
  /*  rect rec;

  rec.x0 = 0;
  rec.y0 = 0;
  rec.x1 = w->allocation.width;
  rec.y1 = w->allocation.height;
  create_ximage(w,w->allocation.width,w->allocation.height);

  add_to_render_region (& rec);
  scroll(0,0);*/
}

static void
area_device_changed (guint32        devid)
{
  int i;

  current_deviceid = devid;

  for (i=0;i<num_tools;i++)
    {
      if (tools[i]->id == devid)
	info.tool = tools[i];
    }

  if (info.tool->has_cursor)
    update_cursor (info.tool->state.x, info.tool->state.y, 0);

  tool_box_set_current_device(devid);
}

static GdkDeviceInfo *get_device_info(guint32 deviceid)
{
  GList *tmp_list = gdk_input_list_devices();
  while (tmp_list)
    {
      if (((GdkDeviceInfo *)tmp_list->data)->deviceid == deviceid)
	return (GdkDeviceInfo *)tmp_list->data;
      tmp_list = tmp_list->next;
    }
  
  return NULL;
}

void
area_enable_device (GtkWidget *w, guint32 devid)
{
  int i, j;
  GdkDeviceInfo *devinfo;

  devinfo = get_device_info(devid);

  for (i=0;i<num_tools;i++)
    if (tools[i]->id == devid)
      {
	tool_box_add_device(tools[i]);
	tools[i]->has_cursor = devinfo->has_cursor;
	
	return;
      }

  num_tools++;
  if (num_tools > MAX_TOOLS)
    g_error("Too many devices");

  tools[i] = (ToolInfo *)malloc(sizeof(ToolInfo));
  tools[i]->id = devid;
  tools[i]->name = devinfo->name;
  tools[i]->has_cursor = devinfo->has_cursor;

  tools[i]->tool = PEN_TOOL;
  tools[i]->size = 8.;
  tools[i]->sens = 1.;
  tools[i]->aspect = 1.;
  tools[i]->angle = 0.;

  tools[i]->use_threshhold = 0;
  tools[i]->threshhold = 0.0;
  for (j=0; j<32; j++)
    {
      tools[i]->profile[j] = j/31.;
    }
  
  tools[i]->state.x = 0;
  tools[i]->state.y = 0;
  tools[i]->state.pressure = 1.0;
  tools[i]->state.xtilt = 0.0;
  tools[i]->state.ytilt = 0.0;
  tools[i]->state.buttons = 0;
  tools[i]->state.op = 0;		/* unused */

  tool_box_add_device(tools[i]);
  if (!info.tool)
    info.tool = tools[i];
}

void
area_disable_device (GtkWidget *w, guint32 devid)
{
  tool_box_remove_device(devid);
  /* just leave the ToolInfo structure in tools */
}

static int
area_button_press_event (GtkWidget      *widget,
			 GdkEventButton *event)
{
  if (event->deviceid != current_deviceid)
    area_device_changed(event->deviceid);

  info.tool->state.x = event->x;
  info.tool->state.y = event->y;
  info.tool->state.pressure = translate_pressure (event->pressure);
  info.tool->state.xtilt = event->xtilt;
  info.tool->state.ytilt = event->ytilt;

  if ((event->button != 1) || !info.tool->use_threshhold)
    {
      info.tool->state.buttons |= 1 << event->button;

      if (event->button == 1) 
	{
	  draw_begin (&info.tool->state, info.tool);
	  info.is_changed = 1;
	}
    }

  return FALSE;
}

static gint
area_button_release_event (GtkWidget      *widget,
			   GdkEventButton *event)
{
  if (event->deviceid != current_deviceid)
    area_device_changed(event->deviceid);

  if ((event->button != 1) || !info.tool->use_threshhold)
    {
      info.tool->state.buttons &= ~(1 << event->button);

      if (event->button == 1)
	draw_end (&info.tool->state, info.tool);
    }
  
  return FALSE;
}

static int
area_motion_notify_event (GtkWidget      *widget,
			  GdkEventMotion *event)
{
  if (event->deviceid != current_deviceid)
    area_device_changed(event->deviceid);

  if (info.tool->use_threshhold)
    {
      /* Work around bug in old XFree86 drivers, where a pen lifted
       * from the tablet reports a pressure of 0.5
       */
      if ((event->pressure > 0.25) && !(event->state & GDK_BUTTON1_MASK))
	{
	  event->pressure = 0.0;
	}

      if ((event->pressure >= info.tool->threshhold) && 
	  !(info.tool->state.buttons & (1 << 1)))
	{
	  info.tool->state.x = event->x;
	  info.tool->state.y = event->y;
	  info.tool->state.pressure = translate_pressure (event->pressure);
	  info.tool->state.xtilt = event->xtilt;
	  info.tool->state.ytilt = event->ytilt;
	  
	  draw_begin (&info.tool->state, info.tool);
	  info.tool->state.buttons |= (1 << 1);
	  info.is_changed = 1;
	  return FALSE;
	}
      else if ((event->pressure < info.tool->threshhold) && 
	       (info.tool->state.buttons & (1 << 1)))
	{
	  draw_end (&info.tool->state, info.tool);
	  info.tool->state.buttons &= ~(1 << 1);
	  return FALSE;
	}
    }

  if (info.tool->state.buttons & 4)
    {
      /* we're scrolling with button 2 */
      screen_scroll(info.tool->state.x - event->x,
		    info.tool->state.y - event->y);
      info.tool->state.x = event->x;
      info.tool->state.y = event->y;
    }
  else if (info.tool->state.buttons & 2) 
    {
      input_state new_state = info.tool->state;
      new_state.x = event->x;
      new_state.y = event->y;
      new_state.pressure = translate_pressure (event->pressure);
      new_state.xtilt = event->xtilt;
      new_state.ytilt = event->ytilt;
      
      draw_update (&info.tool->state, &new_state,
		   info.tool);
      info.tool->state = new_state;
    }
  
  if (!info.tool->has_cursor)
    update_cursor (event->x, event->y, 1);

  return FALSE;
}

static int
area_proximity_in_event(GtkWidget         *widget,
			GdkEventProximity *event)
{
  if (event->deviceid != current_deviceid)
    area_device_changed(event->deviceid);

  if (!info.tool->has_cursor)
    update_cursor (info.tool->state.x, info.tool->state.y, 1);

  return FALSE;
}

static int
area_proximity_out_event(GtkWidget         *widget,
			 GdkEventProximity *event)
{
  if (event->deviceid != current_deviceid)
    area_device_changed(event->deviceid);

  if ((info.tool->state.buttons & (1 << 1)))
    {
      draw_end (&info.tool->state, info.tool);
      info.tool->state.buttons &= ~(1 << 1);
    }

  if (!info.tool->has_cursor)
    update_cursor (info.tool->state.x, info.tool->state.y, 0);

  return FALSE;
}

static int
area_key_press_event (GtkWidget      *widget,
		      GdkEventKey *event)
{
  switch (event->keyval) 
    {
    case GDK_space:
      tool_box_toggle_current_tool();
      break;
    case GDK_Up:
      scroll (0, -100);
      break;
    case GDK_Left:
      scroll (-100, 0);
      break;
    case GDK_Right:
      scroll (100, 0);
      break;
    case GDK_Down:
      scroll (0, 100);
      break;
    }

  return FALSE;
}

static void 
scroll_update (GtkAdjustment *adjustment,
		gpointer data)
{
  scroll_to(info.hscroll_data->value,info.vscroll_data->value);
}


/* all the rendering and drawing actually goes on here */
static int
idle_func (GtkWidget *area) 
{
  /* we have to be careful here not to short circuit */
  int updating = render_chunk(area);
  updating = draw(area) || updating; 
  if (!updating)
    {
      info.is_updating = 0;
      return FALSE;		/* don't continue idle */
    }
  return TRUE;
}

void
start_idle_update()
{
  if (!info.is_updating)
    {
      info.is_updating = 1;
      gtk_idle_add ((GtkFunction)idle_func, info.area);
    }
}

static GtkWidget *inputd = NULL;

void
input_dialog_destroy (GtkWidget *w, gpointer data)
{
  inputd = NULL;
}

void
do_input_dialog (GtkWidget *w, gpointer data)
{
  if (!inputd)
    {
      inputd = gtk_input_dialog_new();
  
      gtk_signal_connect (GTK_OBJECT(inputd), "destroy",
			  (GtkSignalFunc)input_dialog_destroy, NULL);
      gtk_signal_connect_object (GTK_OBJECT(GTK_INPUT_DIALOG(inputd)->close_button),
			  "clicked",
			  (GtkSignalFunc)gtk_widget_hide,
			  GTK_OBJECT(inputd));
      gtk_signal_connect (GTK_OBJECT(GTK_INPUT_DIALOG(inputd)->save_button),
			  "clicked",
			  (GtkSignalFunc)save_gsumirc, NULL);

      gtk_signal_connect (GTK_OBJECT(inputd), "enable_device",
			  (GtkSignalFunc)area_enable_device, NULL);
      gtk_signal_connect (GTK_OBJECT(inputd), "disable_device",
			  (GtkSignalFunc)area_disable_device, NULL);
      gtk_widget_show (inputd);
    }
  else
    {
      if (!GTK_WIDGET_MAPPED(inputd))
	gtk_widget_show(inputd);
      else
	gdk_window_raise(inputd->window);
    }
}

void
do_tool_box (GtkWidget *w, gpointer data)
{
  int i;
  tool_box_create ();
  

  gsumirc_update_devices();

  for (i=0;i<num_tools;i++)
    tool_box_add_device(tools[i]);

  tool_box_set_current_device (current_deviceid);

  tool_box_show ();
}

/* called when the display xform is changed */
/* FIXME: does this get called the user moves a scrollbar? */
void 
update_scrollbars(xform *xf)
{
  info.hscroll_data->value = xf->off_x;
  info.hscroll_data->lower = 0;
  info.hscroll_data->upper = bitmap->width;
  if (info.hscroll_data->upper < 0)
    info.hscroll_data->upper = 0;
  info.hscroll_data->step_increment = xf->scale;
  info.hscroll_data->page_increment = xf->scale * image_width / 2;
  info.hscroll_data->page_size = xf->scale * image_width;
  gtk_signal_emit_by_name (GTK_OBJECT (info.hscroll_data), "changed");

  info.vscroll_data->value = xf->off_y;
  info.vscroll_data->lower = 0;
  info.vscroll_data->upper = bitmap->height;
  if (info.vscroll_data->upper < 0)
    info.vscroll_data->upper = 0;
  info.vscroll_data->step_increment = xf->scale;
  info.vscroll_data->page_increment = xf->scale * image_height / 2;
  info.vscroll_data->page_size = xf->scale * image_height;
  gtk_signal_emit_by_name (GTK_OBJECT (info.vscroll_data), "changed");
}

/* create the drawing area and scroll bars */
GtkWidget *
create_canvas_area ()
{
  GtkWidget *table;
  GtkWidget *hscrollbar;
  GtkWidget *vscrollbar;
  GtkWidget *area;

  table = gtk_table_new (2, 2, FALSE);
  gtk_table_set_col_spacing (GTK_TABLE (table), 0, 2);
  gtk_table_set_row_spacing (GTK_TABLE (table), 0, 2);
  gtk_container_border_width (GTK_CONTAINER (table), 1);
  gtk_widget_show (table);

  info.hscroll_data = GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 1, 0, 1, 1));
  hscrollbar = gtk_hscrollbar_new (info.hscroll_data);
  gtk_table_attach (GTK_TABLE (table), hscrollbar, 0, 1, 1, 2,
		    GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_FILL,
		    0, 0);
  gtk_widget_show (hscrollbar);
  gtk_signal_connect (GTK_OBJECT (info.hscroll_data), "value_changed",
		      (GtkSignalFunc) scroll_update, NULL);

  info.vscroll_data = GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, 1, 0, 1, 1));
  vscrollbar = gtk_vscrollbar_new (info.vscroll_data);
  gtk_table_attach (GTK_TABLE (table), vscrollbar, 1, 2, 0, 1,
		    GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		    0, 0);
  gtk_widget_show (vscrollbar);
  gtk_signal_connect (GTK_OBJECT (info.vscroll_data), "value_changed",
		      (GtkSignalFunc) scroll_update, NULL);

  /* minor point: this function doesn't need to refer to info elsewhere */
  info.area = area = gtk_drawing_area_new();
  gtk_drawing_area_size( GTK_DRAWING_AREA(area), 500, 500);

  gtk_table_attach (GTK_TABLE (table), area, 0, 1, 0, 1,
		    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
		    0, 0);

  gtk_signal_connect_object (GTK_OBJECT (area), "expose_event",
			     (GtkSignalFunc) area_expose_event, GTK_OBJECT(area));
  gtk_signal_connect(GTK_OBJECT(area),"configure_event",
		     (GtkSignalFunc) area_configure_event, NULL);

  gtk_signal_connect (GTK_OBJECT (area), "button_press_event",
		      (GtkSignalFunc) area_button_press_event, GTK_OBJECT(area));

  gtk_signal_connect (GTK_OBJECT (area), "button_release_event",
		      (GtkSignalFunc) area_button_release_event, GTK_OBJECT(area));

  gtk_signal_connect (GTK_OBJECT (area), "motion_notify_event",
		      (GtkSignalFunc) area_motion_notify_event, GTK_OBJECT(area));

  gtk_signal_connect (GTK_OBJECT (area), "proximity_in_event",
		      (GtkSignalFunc) area_proximity_in_event, GTK_OBJECT(area));
  gtk_signal_connect (GTK_OBJECT (area), "proximity_out_event",
		      (GtkSignalFunc) area_proximity_out_event, GTK_OBJECT(area));
  gtk_signal_connect_object (GTK_OBJECT (area), "key_press_event",
			     (GtkSignalFunc) area_key_press_event, NULL);


  gtk_widget_set_events (area, GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK |
			 GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK |
			 GDK_KEY_PRESS_MASK | GDK_PROXIMITY_IN_MASK |
			 GDK_PROXIMITY_OUT_MASK);
  gtk_widget_set_extension_events (area, GDK_EXTENSION_EVENTS_ALL);

  gtk_widget_show (area);

  return table;
}

/* Create the applications main window */
void
create_main_window ()
{
  GtkWidget *window;
  GtkWidget *box1;
  GtkWidget *menubar;
  GtkAcceleratorTable *accelerator_table;
  GtkWidget *area;

  info.mainwin = window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  /*  gtk_signal_connect(GTK_OBJECT(window),
      "destroy", (GtkSignalFunc)do_exit, &window); */
  gtk_window_set_title (GTK_WINDOW (window), "gsumi");

  box1 = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (window), box1);
  gtk_widget_show (box1);
 
  /* create a menubar */
  menus_get_main_menubar (&menubar, &accelerator_table);
  gtk_box_pack_start (GTK_BOX (box1), menubar, FALSE, TRUE, 0);
  gtk_widget_show (menubar);

  /*  Install the accelerator table in the main window  */
  gtk_window_add_accelerator_table (GTK_WINDOW (window), accelerator_table);

  area = create_canvas_area();
  gtk_box_pack_start (GTK_BOX (box1), area, TRUE, TRUE, 0);

  gtk_widget_show (window);
} 

static void
usage(char *program_name)
{
  g_print("Usage: %s [ options ]\n",program_name);
  g_print("Valid options are:\n");
  g_print("  -h --help            Print this message\n");
  g_print("  -v --version         Print version number\n");
  g_print("  -n --num-grays NUM   Number of gray levels\n");
}

int
main (int argc, char *argv[])
{
  int i;
  int ngrays = 32;

  /* gdk_set_show_events (TRUE); */
  gtk_init (&argc, &argv);
  /*   gtk_rc_parse ("gsumirc"); */

  /* Now parse the command line arguments */
  
  for (i=1;i<argc;i++)
    {
      if (!strcmp(argv[i],"--help") ||
	  !strcmp(argv[i],"-h"))
	{
	  usage(argv[0]);
	  exit(0);
	}
      else if (!strcmp(argv[i],"--version") ||
	  !strcmp(argv[i],"-v"))
	{
	  g_print("gsumi v0.5\n");
	  exit(0);
	}
      else if (!strcmp(argv[i],"--num-grays") ||
	  !strcmp(argv[i],"-n"))
	{
	  i++;
	  if (i>=argc)
	    {
	      usage(argv[0]);
	      exit(1);
	    }
	  ngrays = atoi(argv[i]);
	  if (ngrays <= 0 || ngrays > 255)
	    {
	      fprintf(stderr,"%s: invalid number of gray levels\n", argv[0]);
	      exit(1);
	    }
	}
      else
	{
	  usage(argv[0]);
	  exit(1);
	}
    }

  color_init (ngrays);	/* allocate colors */
  load_gsumirc();
  gsumirc_update_menus();

  create_main_window ();

  /* start things up */

  bitmap_init (4000, 4000);
  info.is_changed = 0;

  render_init ();

  init_scale ();		/* initialize transforms */

  info.name = NULL;
  info.file_type = FILE_BY_EXTENSION;
  set_filename();

  info.tool = NULL;
  tool_box_create ();
  area_enable_device (NULL, GDK_CORE_POINTER);
  tool_box_set_current_device (GDK_CORE_POINTER);

  tool_box_show ();

  gsumirc_update_devices();

  info.is_updating = 0;
  start_idle_update();

  /*  gtk_timeout_add (500, draw_data, NULL);*/
  
  gtk_main ();

  gtk_widget_destroy (info.mainwin);
  tool_box_destroy ();
  if (inputd)
    gtk_widget_destroy (inputd);

  gdk_color_context_free (color_context);


  return 0;
}
