/*
 * Copyright 1995,96 Thierry Bousch
 * Licensed under the Gnu Public License, Version 2
 *
 * $Id: memoize.c,v 2.6 1996/09/18 10:02:56 bousch Exp $
 *
 * Memoization of worthwhile objects. This requires the gdbm(3) library.
 */

#include <assert.h>
#include <errno.h>
#include <gdbm.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "saml.h"
#include "saml-parse.h"
#include "induce.h"

typedef struct _memo_entry {
	struct _memo_entry *next;
	int arity;
	const char* rootname;
	int slot;
} PRECIOUS_ENTRY;

typedef struct {
	const char *filename;
	GDBM_FILE handle;
} DB_ENTRY;

#define MAX_DB_ENTRIES	30
static DB_ENTRY db_entries[MAX_DB_ENTRIES];
static int nb_db_entries = 0, current_db_entry = -1;

static char temporary_memo[] = _PATH_TMPMEMO;
static PRECIOUS_ENTRY *precious_list[256];

int register_memo (const char *filename)
{
	GDBM_FILE gf;
	int i;

	filename = tilde_expand(filename);
	/* Already registered ? */
	for (i = 0; i < nb_db_entries; i++)
		if (strcmp(db_entries[i].filename,filename) == 0)
			return (current_db_entry = i);

	if (nb_db_entries >= MAX_DB_ENTRIES-1) {
		fprintf(stderr,
		    "induce: too many memos (max %d), skipping `%s'\n",
		    MAX_DB_ENTRIES, filename);
		return (current_db_entry = -1);
	}
	if (strcmp(filename,"-") == 0) {
		mktemp(temporary_memo);
		gf = gdbm_open(temporary_memo, 0, GDBM_WRCREAT|GDBM_FAST,
			0666, NULL);
		if (gf == NULL)
			perror("induce: gdbm_open");
		unlink(temporary_memo);
	} else {
		/* The prototype of gdbm_open() ought to be fixed, really */
		gf = gdbm_open(filename, 0, GDBM_WRCREAT, 0666, NULL);
		if (gf == NULL)
			perror("induce: gdbm_open");
	}
	current_db_entry = -1;
	if (gf) {
		current_db_entry = nb_db_entries++;
		db_entries[current_db_entry].filename = filename;
		db_entries[current_db_entry].handle = gf;
	}
	return current_db_entry;
}
	
static inline int hash (const char *rootname, int arity)
{
	/*
	 * Takes two bits from the arity, six from rootname[0], and
	 * thus returns a number in [0..255].
	 */
	return (*rootname & 63) | ((arity & 3) << 6);
}

static int get_memo2 (const char* rootname, int arity)
{
	PRECIOUS_ENTRY *p;
	int h = hash(rootname, arity);

	for (p = precious_list[h]; p; p = p->next)
		if (p->arity == arity && !strcmp(p->rootname,rootname))
			return p->slot;
	return -1;
}
	
static GDBM_FILE get_memo (const char *name)
{
	char *rootname, c, *p;
	int arity, slot;

	rootname = alloca(strlen(name) + 1);
	strcpy(rootname, name);
	arity = 0;
	p = strchr(rootname, '[');
	if (p != NULL) {
		arity = 1;
		*p++ = '\0';	/* Isolate the rootname */
		while ((c = *p++) != '\0')
			if (c == ',')
				++arity;
	}
	slot = get_memo2(rootname, arity);
	if (slot < 0)
		return NULL;
	return db_entries[slot].handle;
}

void declare_precious (const char *rootname, int arity)
{
	int h;
	PRECIOUS_ENTRY *p;

	if (floating_precision || current_db_entry < 0) {
		/* Memoizing is disabled. */
		return;
	}
	if (get_memo2(rootname, arity) >= 0) {
		/* Already declared as precious */
		return;
	}
	h = hash(rootname, arity);
	p = malloc(sizeof(PRECIOUS_ENTRY)); assert(p);
	p->next = precious_list[h];
	precious_list[h] = p;
	p->arity = arity;
	p->rootname = rootname;
	p->slot = current_db_entry;
}

int is_saved (const char* name)
{
	datum key;
	GDBM_FILE memo = get_memo(name);

	if (!memo)
		return 0;
	key.dptr = (char *) name;
	key.dsize = strlen(key.dptr);
	return gdbm_exists(memo, key);
}

int save_precious (vertex *v)
{
	datum key, content;
	GDBM_FILE memo = get_memo(v->name);
	int ret;

	if (!memo)
		return -1;  /* not precious */
	key.dptr = v->name;
	key.dsize = strlen(key.dptr);
	content.dptr = mref_string(v->mr);
	content.dsize = strlen(content.dptr);
	ret = gdbm_store(memo, key, content, GDBM_REPLACE);
	if (ret != 0)
		fprintf(stderr, "save_precious: store failed\n");
	return ret;
}

int retrieve_precious (vertex *v)
{
	datum key, content;
	GDBM_FILE memo = get_memo(v->name);
	mref_t model;
	int ret;

	if (!memo)
		return -1;  /* not precious */
	key.dptr = v->name;
	key.dsize = strlen(key.dptr);
	content = gdbm_fetch(memo, key);
	if (content.dptr == NULL) {
		/* Record not found */
		return -1;
	}
#if 0
	ret = parse_poly(v->mr, ST_RATIONAL, content.dptr, content.dsize);
#else
	model = mref_new();
	mref_build(model, ST_RATIONAL, "0");
	mref_cast(model, parsed_poly_type);
	saml_init_lexer_mem(content.dptr, content.dsize);
	ret = saml_parse(v->mr, model);
	mref_free(model);
#endif
	/* Gdbm does not free this space itself */
	free(content.dptr);
	if (ret != 0) {
		fprintf(stderr, "retrieve_precious(%s): syntax error\n",
			v->name);
		return -1;
	}
	return 0;
}
