
/***

ifstats.c	- the interface statistics module
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.
	
***/

#include <curses.h>
#include <panel.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if_arp.h>
#include <linux/if_packet.h>
#include <net/if.h>
#include <netinet/ip.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>
#include <linux/if_ether.h>
#include "ifstats.h"
#include "ifaces.h"
#include "packet.h"
#include "log.h"
#include "dirs.h"
#include "stdwinset.h"
#include "deskman.h"
#include "ipcsum.h"
#include "attrs.h"
#include "error.h"
#include "serv.h"
#include "timer.h"
#include "options.h"
#include "instances.h"

#define SCROLLUP 0
#define SCROLLDOWN 1

/* from log.c, applicable only to this module */

void writegstatlog(struct iftab *table, unsigned long nsecs, FILE * logfile);
void writedstatlog(char *ifname, float activity, float pps,
		   float peakactivity, float peakpps,
		   struct iftotals *ts, unsigned long nsecs,
		   FILE * logfile);

char *ltrim(char *buf)
{
    char *tmp = buf;

    while ((*tmp == ' ') || (*tmp == '\t'))
	tmp++;

    strcpy(buf, tmp);
    return buf;
}

/*
 * Function to check if an interface is already in the interface list.
 * This eliminates duplicate interface entries due to aliases
 */

int ifinlist(struct iflist *list, char *ifname)
{
    struct iflist *ptmp = list;
    int result = 0;

    while ((ptmp != NULL) && (result == 0)) {
	result = (strcmp(ifname, ptmp->ifname) == 0);
	ptmp = ptmp->next_entry;
    }

    return result;
}

void initiflist(struct iflist **list)
{
    FILE *fd;
    char buf[161];
    char ifname[8];
    struct iflist *itmp = NULL;
    struct iflist *tail = NULL;
    unsigned int index = 0;
    int resp;

    *list = NULL;

    fd = fopen("/proc/net/dev", "r");
    if (fd == NULL) {
	errbox("Unable to obtain interface list", ANYKEY_MSG, &resp);
	return;
    }
    fgets(buf, 160, fd);	/* dummy header lines */
    fgets(buf, 160, fd);

    while (!(feof(fd))) {
	strcpy(buf, "");
	fgets(buf, 160, fd);
	if (strcmp(buf, "") != 0) {
	    strcpy(ifname, ltrim(strtok(buf, ":")));

	    if (!(iface_supported(ifname)))
		continue;

	    if (ifinlist(*list, ifname))	/* ignore entry if already in */
		continue;	/* interface list */

	    itmp = malloc(sizeof(struct iflist));
	    bzero(itmp, sizeof(struct iflist));
	    strcpy(itmp->ifname, ifname);
	    index++;
	    itmp->index = index;

	    if (*list == NULL) {
		*list = itmp;
		itmp->prev_entry = NULL;
	    } else {
		tail->next_entry = itmp;
		itmp->prev_entry = tail;
	    }

	    tail = itmp;
	    itmp->next_entry = NULL;
	}
    }
    fclose(fd);
}

void positionptr(struct iftab *table, struct iflist **ptmp, char *ifname)
{
    struct iflist *plast = NULL;
    int ok = 0;

    *ptmp = table->head;

    while ((*ptmp != NULL) && (!ok)) {
	ok = (strcmp((*ptmp)->ifname, ifname) == 0);

	if (!ok) {
	    if ((*ptmp)->next_entry == NULL)
		plast = *ptmp;

	    *ptmp = (*ptmp)->next_entry;
	}
    }

    if (*ptmp == NULL) {
	*ptmp = malloc(sizeof(struct iflist));
	bzero(*ptmp, sizeof(struct iflist));
	(*ptmp)->index = plast->index + 1;
	plast->next_entry = *ptmp;
	(*ptmp)->prev_entry = plast;
	(*ptmp)->next_entry = NULL;
	strcpy((*ptmp)->ifname, ifname);
	
	if ((*ptmp)->index <= LINES - 4)
	   table->lastvisible = *ptmp;
    }
}

void destroyiflist(struct iflist *list)
{
    struct iflist *ctmp;
    struct iflist *ptmp;

    if (list != NULL) {
	ptmp = list;
	ctmp = ptmp->next_entry;

	do {
	    free(ptmp);
	    ptmp = ctmp;
	    if (ctmp != NULL)
		ctmp = ctmp->next_entry;
	} while (ptmp != NULL);
    }
}

void updaterates(struct iftab *table, time_t starttime, time_t now,
		 unsigned int idx)
{
    struct iflist *ptmp = table->firstvisible;

    wattrset(table->statwin, HIGHATTR);
    do {
	ptmp->rate = ((float) (ptmp->spanbr * 8 / 1000)) / ((float) (now - starttime));

	if (ptmp->rate > ptmp->peakrate)
	    ptmp->peakrate = ptmp->rate;

	wmove(table->statwin, ptmp->index - idx, 52);
	wprintw(table->statwin, "%8.2f kbits/sec", ptmp->rate);
	ptmp->spanbr = 0;
	ptmp = ptmp->next_entry;
    } while (ptmp != table->lastvisible->next_entry);
}

void printifentry(struct iflist *ptmp, WINDOW * win, unsigned int idx)
{
    unsigned int target_row;

    if ((ptmp->index < idx) || (ptmp->index > idx + (LINES - 5)))
	return;

    target_row = ptmp->index - idx;

    wattrset(win, STDATTR);
    wmove(win, target_row, 1);
    wprintw(win, "%s", ptmp->ifname);
    wattrset(win, HIGHATTR);
    wmove(win, target_row, 12);
    printlargenum(ptmp->total, win);
    wmove(win, target_row, 22);
    printlargenum(ptmp->iptotal, win);
    wmove(win, target_row, 32);
    printlargenum(ptmp->noniptotal, win);
    wmove(win, target_row, 42);
    wprintw(win, "%8lu", ptmp->badtotal);
}

void preparescreen(struct iftab *table)
{
    struct iflist *ptmp = table->head;
    unsigned int i = 1;

    unsigned int winht = LINES - 4;

    table->firstvisible = table->head;

    do {
	printifentry(ptmp, table->statwin, 1);

	if (i <= winht)
	    table->lastvisible = ptmp;

	ptmp = ptmp->next_entry;
	i++;
    } while ((ptmp != NULL) && (i <= winht));
}

void labelstats(WINDOW * win)
{
    wmove(win, 0, 1);
    wprintw(win, " Iface ");
    wmove(win, 0, 16);
    wprintw(win, " Total ");
    wmove(win, 0, 29);
    wprintw(win, " IP ");
    wmove(win, 0, 36);
    wprintw(win, " NonIP ");
    wmove(win, 0, 45);
    wprintw(win, " BadIP ");
    wmove(win, 0, 55);
    wprintw(win, " Activity ");
}

void initiftab(struct iftab *table)
{
    initiflist(&(table->head));

    table->borderwin = newwin(LINES - 2, 80, 1, 0);
    table->borderpanel = new_panel(table->borderwin);

    move(LINES - 1, 1);
    scrollkeyhelp();
    stdexitkeyhelp();
    wattrset(table->borderwin, BOXATTR);
    box(table->borderwin, ACS_VLINE, ACS_HLINE);
    labelstats(table->borderwin);
    table->statwin = newwin(LINES - 4, 78, 2, 1);
    table->statpanel = new_panel(table->statwin);
    stdwinset(table->statwin);
    wtimeout(table->statwin, -1);
    wattrset(table->statwin, STDATTR);
    colorwin(table->statwin);
    wattrset(table->statwin, BOXATTR);
    wmove(table->borderwin, LINES - 3, 32);
    wprintw(table->borderwin, " Total, IP, NonIP, and BadIP are packet counts ");
}

/*
 * Scrolling routines for the general interface statistics window
 */

void scrollgstatwin(struct iftab *table, int direction, unsigned int *idx)
{
    wattrset(table->statwin, STDATTR);
    if (direction == SCROLLUP) {
	if (table->lastvisible->next_entry != NULL) {
	    wscrl(table->statwin, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;
	    (*idx)++;
	    wmove(table->statwin, LINES - 5, 0);
	    scrollok(table->statwin, 0);
	    wprintw(table->statwin, "%78c", ' ');
	    scrollok(table->statwin, 1);
	    printifentry(table->lastvisible, table->statwin, *idx);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->statwin, -1);
	    table->firstvisible = table->firstvisible->prev_entry;
	    table->lastvisible = table->lastvisible->prev_entry;
	    (*idx)--;
	    wmove(table->statwin, 0, 0);
	    wprintw(table->statwin, "%78c", ' ');
	    printifentry(table->firstvisible, table->statwin, *idx);
	}
    }
}

void pagegstatwin(struct iftab *table, int direction, int *idx)
{
    int i = 1;

    if (direction == SCROLLUP) {
	while ((i <= LINES - 5) && (table->lastvisible->next_entry != NULL)) {
	    i++;
	    scrollgstatwin(table, direction, idx);
	}
    } else {
	while ((i <= LINES - 5) && (table->firstvisible != table->head)) {
	    i++;
	    scrollgstatwin(table, direction, idx);
	}
    }
}


/*
 * The general interface statistics function
 */

void ifstats(const struct OPTIONS *options, int facilitytime)
{
    int logging = options->logging;
    struct iftab table;

    char buf[8192];
    char *packet;
    struct iphdr *ipacket;
    char *tpacket;

    struct sockaddr_pkt fromaddr;

    struct iflist *ptmp = NULL;

    unsigned int idx = 1;

    int fd;
    FILE *logfile = NULL;

    int br;
    char ifname[10];

    int iphlen;
    int ipck, ipcsum;

    int ch;

    int endloop = 0;
    struct timeval tv;
    unsigned long starttime = 0;
    unsigned long statbegin = 0;
    unsigned long now = 0;
    unsigned long unow = 0;
    unsigned long startlog = 0;
    unsigned long updtime = 0;
    unsigned long updtime_usec = 0;

    if (!facility_active(GSTATIDFILE))
        mark_facility(GSTATIDFILE, "general interface statistics");
    else {
        errbox("General interface stats already active in another process",
               ANYKEY_MSG, &ch);
        return;
    }
        
    initiftab(&table);

    if (logging) {
	opentlog(&logfile, GSTATLOG);

	if (logfile == NULL)
	    logging = 0;
    }
    writelog(logging, logfile, "******** General interface statistics started");

    if (table.head != NULL) {
	open_socket(&fd, "");

	if (fd < 0)
	    return;

	preparescreen(&table);

	update_panels();
	doupdate();

	gettimeofday(&tv, NULL);
	starttime = startlog = statbegin = tv.tv_sec;

	while (!endloop) {
	    getpacket(fd, buf, &fromaddr, &ch, &br, 5, table.statwin);

	    if (ch != ERR) {
		switch (ch) {
		case KEY_UP:
		    scrollgstatwin(&table, SCROLLDOWN, &idx);
		    break;
		case KEY_DOWN:
		    scrollgstatwin(&table, SCROLLUP, &idx);
		    break;
		case KEY_PPAGE:
		case '-':
		    pagegstatwin(&table, SCROLLDOWN, &idx);
		    break;
		case KEY_NPAGE:
		case ' ':
		    pagegstatwin(&table, SCROLLUP, &idx);
		    break;
		case 'Q':
		case 'q':
		case 'X':
		case 'x':
		case 27:
		case 24:
		    endloop = 1;
		    break;
		}
	    }
	    if (br > 0) {
		strcpy(ifname, fromaddr.spkt_device);
		positionptr(&table, &ptmp, ifname);

		ptmp->total++;

		ptmp->spanbr += br;
		ptmp->br += br;

		if (fromaddr.spkt_protocol == ntohs(ETH_P_IP)) {
		    (ptmp->iptotal)++;
		    adjustpacket(buf, fromaddr.spkt_family,
				 &packet, &br);

		    if (packet == NULL)
			continue;

		    ipacket = (struct iphdr *) packet;
		    iphlen = ipacket->ihl * 4;
		    tpacket = packet + iphlen;

		    /* recalculate checksum */

		    ipcsum = ipacket->check;
		    ipacket->check = 0;
		    ipck = in_cksum((u_short *) ipacket, iphlen);
		    if ((ipcsum != ipck) || (!ipck)) {
			(ptmp->badtotal)++;
			continue;
		    }
		} else {
		    (ptmp->noniptotal)++;
		}
		printifentry(ptmp, table.statwin, idx);
	    }
	    
	    gettimeofday(&tv, NULL);
	    now = tv.tv_sec;
	    unow = tv.tv_usec;
	    
	    if ((now - starttime) >= 5) {
		updaterates(&table, starttime, now, idx);
		printelapsedtime(statbegin, now, LINES - 3, 1, table.borderwin);
		starttime = now;
	    }
	    if (((now - startlog) >= options->logspan) && (logging)) {
		writegstatlog(&table, time((time_t *) NULL) - statbegin, logfile);
		startlog = now;
	    }
	    if (((options->updrate != 0) && (now - updtime >= options->updrate)) ||
	       ((options->updrate == 0) && (unow - updtime_usec >= DEFAULT_UPDATE_DELAY))) {
		update_panels();
		doupdate();
		updtime = now;
		updtime_usec = unow;
	    }
	    if ((facilitytime != 0) && (((now - statbegin) / 60) >= facilitytime))
	        endloop = 1;
	}

	close(fd);
    }
    del_panel(table.statpanel);
    delwin(table.statwin);
    del_panel(table.borderpanel);
    delwin(table.borderwin);
    update_panels();
    doupdate();

    if (logging) {
	writegstatlog(&table, time((time_t *) NULL) - statbegin, logfile);
	writelog(logging, logfile, "******** General interface statistics stopped");
	fclose(logfile);
    }
    destroyiflist(table.head);
    unlink(GSTATIDFILE);
}

int initialize_brackets(char *ifname, struct iftotals *ttmp)
{
    struct ifreq ifr;
    int fd;
    int resp;
    int istat;
    unsigned int interval;
    int i;
    
    strcpy(ifr.ifr_name, ifname);
    
    fd = socket(PF_INET, SOCK_PACKET, htons(ETH_P_ALL));
    
    if (fd < 0) {
        errbox("Unable to open socket for MTU determination", ANYKEY_MSG, &resp);
        return 1;
    }
    
    istat = ioctl(fd, SIOCGIFMTU, &ifr);
    
    close(fd);
    if (istat < 0) {
        errbox("Unable to obtain interface MTU", ANYKEY_MSG, &resp);
	return 1;
    }
            
    ttmp->interval = interval = ifr.ifr_mtu / 16;    /* There are 16 packet size brackets */
    
    for (i = 0; i <= 15; i++) {
        ttmp->brackets[i].floor = interval * i + 1;
        ttmp->brackets[i].ceil = interval * (i + 1);
    }
    
    ttmp->brackets[15].ceil = ifr.ifr_mtu;
    return 0;
}

void printdetlabels(WINDOW * win, struct iftotals *totals)
{
    char *sizelabels = "Pkt Size (bytes)        Count";
    int i;

    wattrset(win, BOXATTR);
    wmove(win, 2, 14);
    wprintw(win, "Packets       Bytes");
    wmove(win, 12, 2);
    wprintw(win, sizelabels);
    wmove(win, 12, 36);
    wprintw(win, sizelabels);
    wattrset(win, STDATTR);
    wmove(win, 3, 2);
    wprintw(win, "Total:");
    wmove(win, 4, 2);
    wprintw(win, "IP:");
    wmove(win, 5, 2);
    wprintw(win, "TCP:");
    wmove(win, 6, 2);
    wprintw(win, "UDP:");
    wmove(win, 7, 2);
    wprintw(win, "ICMP:");
    wmove(win, 8, 2);
    wprintw(win, "Other IP:");
    wmove(win, 9, 2);
    wprintw(win, "Non-IP:");
    wmove(win, 5, 40);
    wprintw(win, "Activity:");

    wmove(win, 8, 40);
    wprintw(win, "IP Checksum Errors:");


    for (i = 0; i <= 7; i++) {
        wattrset(win, STDATTR);
        wmove(win, i + 13, 2);
        wprintw(win, "%4u to %4u:", totals->brackets[i].floor, totals->brackets[i].ceil);
        wmove(win, i+ 13, 23);
        wattrset(win, HIGHATTR);
        wprintw(win, "%8lu", 0);
    }

    for (i = 8; i <= 15; i++) {
        wattrset(win, STDATTR);
        wmove(win, i + 5, 36);
        wprintw(win, "%4u to %4u:", totals->brackets[i].floor, totals->brackets[i].ceil);
        wmove(win, i + 5, 57);
        wattrset(win, HIGHATTR);
        wprintw(win, "%8lu", 0);
    }

    update_panels();
    doupdate();
}

void printdetails(struct iftotals *totals, WINDOW * win)
{
    wattrset(win, HIGHATTR);

    /* Print totals on the IP protocols */

    wmove(win, 3, 12);
    printlargenum(totals->total, win);
    wmove(win, 3, 24);
    printlargenum(totals->bytestotal, win);
    wmove(win, 4, 12);
    printlargenum(totals->iptotal, win);
    wmove(win, 4, 24);
    printlargenum(totals->ipbtotal, win);

    wmove(win, 5, 12);
    printlargenum(totals->tcptotal, win);
    wmove(win, 5, 24);
    printlargenum(totals->tcpbtotal, win);

    wmove(win, 6, 12);
    printlargenum(totals->udptotal, win);
    wmove(win, 6, 24);
    printlargenum(totals->udpbtotal, win);

    wmove(win, 7, 12);
    printlargenum(totals->icmptotal, win);
    wmove(win, 7, 24);
    printlargenum(totals->icmpbtotal, win);

    wmove(win, 8, 12);
    printlargenum(totals->othtotal, win);
    wmove(win, 8, 24);
    printlargenum(totals->othbtotal, win);

    /* Print non-IP totals */

    wmove(win, 9, 12);
    printlargenum(totals->noniptotal, win);
    wmove(win, 9, 24);
    printlargenum(totals->nonipbtotal, win);

    /* Bad packet count */

    wmove(win, 8, 60);
    wprintw(win, "%8lu", totals->badtotal);
}

void update_size_distrib(unsigned int length, struct iftotals *totals, WINDOW *win)
{
    unsigned int i;
    
    i = (length - 1) / totals->interval;   /* minus 1 to keep interval
    					      boundary lengths within the
    					      proper brackets */
    
    if (i > 15)				   /* This is for extras for MTU's not */
        i = 15;				   /* divisible by 16 */
        
    totals->brackets[i].count++;
    
    if (i < 8)
        wmove(win, i + 13, 23);
    else
        wmove(win, i + 5, 57);
        
    wprintw(win, "%8lu", totals->brackets[i].count);
}

/*
 * The detailed interface statistics function
 */
 
void detstats(char *iface, const struct OPTIONS *options, int facilitytime)
{
    int logging = options->logging;

    WINDOW *statwin;
    PANEL *statpanel;

    char buf[8192];
    char *packet;
    struct iphdr *ipacket = NULL;
    char *tpacket;

    struct sockaddr_pkt fromaddr;

    int fd;
    FILE *logfile = NULL;
    int br;
    int framelen = 0;

    int iphlen;
    int ipck, ipcsum;
    unsigned int iplen;

    struct iftotals totals;

    int ch;

    struct timeval tv;
    unsigned long updtime = 0;
    unsigned long updtime_usec = 0;
    int endloop = 0;
    unsigned long starttime, now;
    unsigned long statbegin, startlog;
    unsigned long unow;
    float spanbr = 0;
    float spanpkt = 0;
    float activity = 0;
    float peakactivity = 0;
    float pps = 0;
    float peakpps = 0;

    /*
     * Mark this facility
     */
     
    if (!facility_active(DSTATIDFILE))
        mark_facility(DSTATIDFILE, "detailed interface statistics");
    else {
        errbox("Detailed interface stats already active in another process",
               ANYKEY_MSG, &ch);
        return;
    }
    
    if (!iface_supported(iface)) {
        err_iface_unsupported();
        unlink(DSTATIDFILE);
        return;
    }
        
    move(LINES - 1, 1);
    stdexitkeyhelp();
    statwin = newwin(23, 80, 1, 0);
    statpanel = new_panel(statwin);
    stdwinset(statwin);
    wtimeout(statwin, -1);
    wattrset(statwin, BOXATTR);
    colorwin(statwin);
    box(statwin, ACS_VLINE, ACS_HLINE);
    wmove(statwin, 0, 1);
    wprintw(statwin, " Statistics for %s ", iface);
    wattrset(statwin, STDATTR);
    update_panels();
    doupdate();

    bzero(&totals, sizeof(struct iftotals));
    
    if (logging) {
	opentlog(&logfile, DSTATLOG);

	if (logfile == NULL)
	    logging = 0;
    }
    writelog(logging, logfile, "******** Detailed interface statistics started");
    open_socket(&fd, iface);

    if (fd < 0)
	return;

    if (initialize_brackets(iface, &totals) != 0) {
        close(fd);
        del_panel(statpanel);
        delwin(statwin);
        update_panels();
        doupdate();
        unlink(DSTATIDFILE);
        return;
    }
   
    printdetlabels(statwin, &totals);
    printdetails(&totals, statwin);
    update_panels();
    doupdate();

    spanbr = 0;
    
    gettimeofday(&tv, NULL);
    starttime = startlog = statbegin = tv.tv_sec;
    
    leaveok(statwin, TRUE);

    /*
     * Data-gathering loop
     */
     
    while (!endloop) {
	getpacket(fd, buf, &fromaddr, &ch, &br, 5, statwin);

	if (ch != ERR) {
	    switch (ch) {
	    case 'Q':
	    case 'q':
	    case 'X':
	    case 'x':
	    case 24:
	    case 27:
		endloop = 1;
		break;
	    }
	}
	if (br > 0) {
	    if (strcmp(iface, fromaddr.spkt_device) != 0)
		continue;

	    framelen = br;
	    totals.total++;
	    totals.bytestotal += br;

	    spanbr += br;
	    spanpkt++;

	    if (fromaddr.spkt_protocol == ntohs(ETH_P_IP)) {
		totals.iptotal++;
		adjustpacket(buf, fromaddr.spkt_family,
			     &packet, &br);

		if (packet == NULL)
		    continue;

		ipacket = (struct iphdr *) packet;
		iphlen = ipacket->ihl * 4;
		tpacket = packet + iphlen;

		ipcsum = ipacket->check;
		ipacket->check = 0;
		ipck = in_cksum((u_short *) ipacket, iphlen);
		if ((ipcsum != ipck) || (!ipck)) {
		    totals.badtotal++;
		    continue;
		}
		iplen = ntohs(ipacket->tot_len);
		totals.ipbtotal += iplen;

		switch (ipacket->protocol) {
		case IPPROTO_TCP:
		    totals.tcptotal++;
		    totals.tcpbtotal += iplen;
		    break;
		case IPPROTO_UDP:
		    totals.udptotal++;
		    totals.udpbtotal += iplen;
		    break;
		case IPPROTO_ICMP:
		    totals.icmptotal++;
		    totals.icmpbtotal += iplen;
		    break;
		default:
		    totals.othtotal++;
		    totals.othbtotal += iplen;
		    break;
		}
	    } else {
		totals.noniptotal++;
		totals.nonipbtotal += br;
	    }
            update_size_distrib(framelen, &totals, statwin);
	}
	
	gettimeofday(&tv, NULL);
	now = tv.tv_sec;
	unow = tv.tv_usec;
	
	if (now - starttime >= 5) {
	    wattrset(statwin, BOXATTR);
	    printelapsedtime(statbegin, now, 22, 1, statwin);
	    activity = (float) (spanbr * 8 / 1000) / (float) (now - starttime);
	    pps = (float) (spanpkt) / (float) (now - starttime);
	    spanbr = 0;
	    spanpkt = 0;
	    starttime = now;

	    wattrset(statwin, HIGHATTR);
	    wmove(statwin, 5, 50);
	    wprintw(statwin, "%8.1f kbits/sec", activity);
	    wmove(statwin, 6, 50);
	    wprintw(statwin, "%8.1f packets/sec", pps);

	    if (activity > peakactivity)
		peakactivity = activity;

	    if (pps > peakpps)
		peakpps = pps;
	}
	if (((now - startlog) >= options->logspan) && (logging)) {
	    writedstatlog(iface, activity, pps, peakactivity, peakpps,
		    &totals, time((time_t *) NULL) - statbegin, logfile);
	    startlog = now;
	}
	if (((options->updrate == 0) && (unow - updtime_usec >= DEFAULT_UPDATE_DELAY)) ||
	   ((options->updrate != 0) && (now - updtime >= options->updrate))) {
  	    printdetails(&totals, statwin);
	    update_panels();
	    doupdate();
	    updtime_usec = unow;
	    updtime = now;
	}
	
	if ((facilitytime != 0) && (((now - statbegin) / 60) >= facilitytime))
	    endloop = 1;
    }

    close(fd);

    if (logging) {
	writedstatlog(iface, activity, pps, peakactivity, peakpps,
		    &totals, time((time_t *) NULL) - statbegin, logfile);
	writelog(logging, logfile, "******** Detailed interface statistics stopped");
	fclose(logfile);
    }
    del_panel(statpanel);
    delwin(statwin);
    unlink(DSTATIDFILE);
    update_panels();
    doupdate();
}

void selectiface(char *ifname, int *aborted)
{
    WINDOW *win;
    PANEL *panel;
    WINDOW *bw;
    PANEL *bp;

    int row;
    int ch;
    int exitloop = 0;

    struct iflist *list;
    struct iflist *ptmp;

    initiflist(&list);

    listkeyhelp();
    bw = newwin(14, 24, 6, 19);
    bp = new_panel(bw);
    wattrset(bw, BOXATTR);
    box(bw, ACS_VLINE, ACS_HLINE);

    win = newwin(12, 22, 7, 20);
    panel = new_panel(win);
    stdwinset(win);
    wattrset(win, BOXATTR);
    colorwin(win);
    wmove(bw, 0, 1);
    wprintw(bw, " Select Interface ");

    row = 0;
    ptmp = list;

    wattrset(win, STDATTR);
    do {
	wmove(win, row, 2);
	wprintw(win, "%s", ptmp->ifname);
	ptmp = ptmp->next_entry;
	row++;
    } while ((ptmp != NULL) && (row <= 11));
    update_panels();
    doupdate();

    row = 0;
    ptmp = list;

    do {
	wattrset(win, PTRATTR);
	wmove(win, row, 1);
	waddch(win, ACS_RARROW);
	ch = wgetch(win);
	wmove(win, row, 1);
	wprintw(win, " ");

	wattrset(win, STDATTR);
	switch (ch) {
	case KEY_DOWN:
	    if (ptmp->next_entry != NULL) {
		if (row < 11)
		    row++;
		else {
		    wscrl(win, 1);
		    scrollok(win, 0);
		    wmove(win, 11, 0);
		    wprintw(win, "%22c", ' ');
		    wmove(win, 11, 2);
		    wprintw(win, ptmp->next_entry->ifname);
		    scrollok(win, 1);
		}
		ptmp = ptmp->next_entry;
	    }
	    break;
	case KEY_UP:
	    if (ptmp->prev_entry != NULL) {
		if (row > 0)
		    row--;
		else {
		    wscrl(win, -1);
		    wmove(win, 0, 0);
		    wprintw(win, "%22c", ' ');
		    wmove(win, 0, 2);
		    wprintw(win, ptmp->prev_entry->ifname);
		}
		ptmp = ptmp->prev_entry;
	    }
	    break;
	case 13:
	    exitloop = 1;
	    *aborted = 0;
	    break;
	case 27:
	case 24:
	case 'X':
	case 'x':
	case 'Q':
	case 'q':
	    exitloop = 1;
	    *aborted = 1;
	    break;
	}
	update_panels();
	doupdate();
    } while (!exitloop);

    if (!(*aborted))
	strcpy(ifname, ptmp->ifname);

    destroyiflist(list);
    del_panel(panel);
    delwin(win);
    del_panel(bp);
    delwin(bw);
    update_panels();
    doupdate();
}
