/***

othptab.c - non-TCP protocol display module
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#ifndef _I386_TYPES_H
#include <asm/types.h>
#endif

#include <linux/if_fddi.h>
#include "tcptable.h"
#include "othptab.h"
#include "stdwinset.h"
#include "deskman.h"
#include "attrs.h"
#include "log.h"
#include "revname.h"
#include "rvnamed.h"

void convmacaddr(char *addr, char *result);	/* external; from hostmon.c */

void init_othp_table(struct othptable *table)
{
    unsigned int winht;
    unsigned int wintop;
    unsigned int obmaxx;

    winht = LINES - (LINES * 0.6) - 2;
    wintop = (LINES * 0.6) + 1;

    table->count = 0;
    table->lastpos = 0;
    table->htstat = NOHTIND;
    table->head = table->tail = NULL;
    table->firstvisible = table->lastvisible = NULL;
    table->borderwin = newwin(winht, 80, wintop, 0);
    table->borderpanel = new_panel(table->borderwin);
    wattrset(table->borderwin, BOXATTR);
    box(table->borderwin, ACS_VLINE, ACS_HLINE);

    table->head = table->tail = NULL;
    table->othpwin = newwin(winht - 2, 78, wintop + 1, 1);
    table->othppanel = new_panel(table->othpwin);
    wattrset(table->othpwin, STDATTR);
    colorwin(table->othpwin);
    update_panels();
    doupdate();

    stdwinset(table->othpwin);
    getmaxyx(table->borderwin, table->obmaxy, obmaxx);
    table->oimaxy = table->obmaxy - 2;
}

void process_dest_unreach(struct tcptable *table, char *packet, char *ifname,
                          int *nomem)
{
    struct iphdr *ip;
    struct tcphdr *tcp;
    struct tcptableent *tcpentry;
    unsigned int isclosed;

    ip = (struct iphdr *) (packet + 8);

    if (ip->protocol != IPPROTO_TCP)
	return;

    tcp = (struct tcphdr *) (packet + 8 + (ip->ihl * 4));

    /* 
     * We really won't be making use of nomem here
     */

    tcpentry = in_table(table, ip->saddr, ip->daddr,
			ntohs(tcp->source), ntohs(tcp->dest), ifname,
			&isclosed, 0, 0, NULL, nomem);	/* we'll not do any timeout */
    /* checking here */

    if (tcpentry != NULL) {
	tcpentry->stat = tcpentry->oth_connection->stat = FLAG_RST;
	addtoclosedlist(table, tcpentry, nomem);
    }
}

struct othptabent *add_othp_entry(struct othptable *table,
				  struct tcptable *tcptab,
				  unsigned long saddr,
				  unsigned long daddr, int is_ip,
				  int protocol, unsigned short linkproto,
				  char *packet, char *ifname,
				  int rev_lookup, int rvnfd,
				  unsigned int tm, int logging,
				  FILE * logfile, int servnames,
				  int *nomem)
{
    struct othptabent *new_entry;
    struct othptabent *temp;
    struct in_addr isaddr, idaddr;

    new_entry = malloc(sizeof(struct othptabent));

    if (new_entry == NULL) {
	printnomem();
	*nomem = 1;
	return NULL;
    }
    bzero(new_entry, sizeof(struct othptabent));

    new_entry->is_ip = is_ip;

    if (is_ip) {
	new_entry->saddr = isaddr.s_addr = saddr;
	new_entry->daddr = idaddr.s_addr = daddr;

	revname(rev_lookup, &isaddr, new_entry->s_fqdn, rvnfd);
	revname(rev_lookup, &idaddr, new_entry->d_fqdn, rvnfd);

	if (protocol == IPPROTO_ICMP) {
	    new_entry->un.icmp.type = ((struct icmphdr *) packet)->type;
	    new_entry->un.icmp.code = ((struct icmphdr *) packet)->code;
	} else if (protocol == IPPROTO_UDP) {
	    servlook(servnames, ((struct udphdr *) packet)->source,
		     IPPROTO_UDP, new_entry->un.udp.s_sname, 10);
	    servlook(servnames, ((struct udphdr *) packet)->dest,
		     IPPROTO_UDP, new_entry->un.udp.d_sname, 10);
	} else if (protocol == IPPROTO_OSPFIGP) {
	    new_entry->un.ospf.type = ((struct ospfhdr *) packet)->ospf_type;
	    new_entry->un.ospf.area = ntohl(((struct ospfhdr *) packet)->ospf_areaid.s_addr);
	    strcpy(new_entry->un.ospf.routerid, inet_ntoa(((struct ospfhdr *) packet)->ospf_routerid));
	}
    } else {
	new_entry->linkproto = linkproto;

	if (linkproto == ARPHRD_ETHER) {
	    convmacaddr(((struct ethhdr *) packet)->h_source, new_entry->smacaddr);
	    convmacaddr(((struct ethhdr *) packet)->h_dest, new_entry->dmacaddr);
	} else if (linkproto == ARPHRD_FDDI) {
	    convmacaddr(((struct fddihdr *) packet)->saddr, new_entry->smacaddr);
	    convmacaddr(((struct fddihdr *) packet)->daddr, new_entry->dmacaddr);
	}
    }

    new_entry->protocol = protocol;
    strcpy(new_entry->iface, ifname);

    if (table->head == NULL) {
	new_entry->prev_entry = NULL;
	table->head = new_entry;
	table->firstvisible = new_entry;
    }
    /*
     * Max number of entries in the lower window is 512.  Upon reaching
     * this figure, oldest entries are thrown out.
     */

    if (table->count == 512) {
	if (table->firstvisible == table->head) {
	    wscrl(table->othpwin, 1);
	    printothpentry(table, table->lastvisible->next_entry, table->oimaxy - 1, logging, logfile);
	    table->firstvisible = table->firstvisible->next_entry;
	    table->lastvisible = table->lastvisible->next_entry;
	}
	temp = table->head;
	table->head = table->head->next_entry;
	table->head->prev_entry = NULL;
	free(temp);
    } else
	table->count++;

    if (table->tail != NULL) {
	new_entry->prev_entry = table->tail;
	table->tail->next_entry = new_entry;
    }
    table->tail = new_entry;
    new_entry->next_entry = NULL;

    table->lastpos++;
    new_entry->index = table->lastpos;

    if (table->count <= table->oimaxy) {
	table->lastvisible = new_entry;
	printothpentry(table, new_entry, table->count - 1, logging, logfile);
    } else if (table->lastvisible == table->tail->prev_entry) {
	wscrl(table->othpwin, 1);
	table->firstvisible = table->firstvisible->next_entry;
	table->lastvisible = table->tail;
	printothpentry(table, new_entry, table->oimaxy - 1, logging, logfile);
    }
    return new_entry;
}

void printothpentry(struct othptable *table, struct othptabent *entry,
		    unsigned int target_row, int logging, FILE * logfile)
{
    char protname[29];
    char description[25];
    char additional[60];
    char msgstring[150];
    char scratchpad[80];

    unsigned int unknown = 0;

    wmove(table->borderwin, table->obmaxy - 1, 1);
    if ((table->lastvisible == table->tail) && (table->htstat != TIND) &&
	(table->count >= table->oimaxy)) {
	wprintw(table->borderwin, " Bottom ");
	table->htstat = TIND;
    } else if ((table->firstvisible == table->head) && (table->htstat != HIND)) {
	wprintw(table->borderwin, " Top ");
	table->htstat = HIND;
    }
    if (!(entry->is_ip)) {
	wattrset(table->othpwin, UNKNATTR);
	wmove(table->othpwin, target_row, 0);
	scrollok(table->othpwin, 0);
	wprintw(table->othpwin, "%78c", 32);
	scrollok(table->othpwin, 1);
	wmove(table->othpwin, target_row, 1);

	switch (entry->protocol) {
	case ETH_P_ARP:
	    sprintf(msgstring, "ARP");
	    break;
	case ETH_P_RARP:
	    sprintf(msgstring, "Reverse ARP");
	    break;
	default:
	    sprintf(msgstring, "Non-IP packet (0x%x)",
		    entry->protocol);
	}

	if ((entry->linkproto == ARPHRD_ETHER) ||
	    (entry->linkproto == ARPHRD_FDDI))  {
	    sprintf(scratchpad, " from %s to %s on %s",
		    entry->smacaddr, entry->dmacaddr, entry->iface);

	    strcat(msgstring, scratchpad);
	}
	wprintw(table->othpwin, "%s", msgstring);
	writelog(logging, logfile, msgstring);

	return;
    }
    strcpy(additional, "");
    strcpy(description, "");

    switch (entry->protocol) {
    case IPPROTO_UDP:
	wattrset(table->othpwin, UDPATTR);
	strcpy(protname, "UDP");
	break;
    case IPPROTO_ICMP:
	wattrset(table->othpwin, STDATTR);
	strcpy(protname, "ICMP");
	break;
    case IPPROTO_OSPFIGP:
	wattrset(table->othpwin, OSPFATTR);
	strcpy(protname, "OSPF");
	break;
    case IPPROTO_IGP:
	wattrset(table->othpwin, IGPATTR);
	strcpy(protname, "IGP");
	break;
    case IPPROTO_IGMP:
	wattrset(table->othpwin, IGMPATTR);
	strcpy(protname, "IGMP");
	break;
    case IPPROTO_IGRP:
	wattrset(table->othpwin, IGRPATTR);
	strcpy(protname, "IGRP");
	break;
    default:
	wattrset(table->othpwin, UNKNATTR);
	sprintf(protname, "Protocol %u", entry->protocol);
	unknown = 1;
    }

    if (entry->protocol == IPPROTO_ICMP) {
	switch (entry->un.icmp.type) {
	case ICMP_ECHOREPLY:
	    strcpy(description, "echo reply");
	    break;
	case ICMP_ECHO:
	    strcpy(description, "echo request");
	    break;
	case ICMP_DEST_UNREACH:
	    strcpy(description, "dest unreach");
	    switch (entry->un.icmp.code) {
	    case ICMP_NET_UNREACH:
		strcpy(additional, "network");
		break;
	    case ICMP_HOST_UNREACH:
		strcpy(additional, "host");
		break;
	    case ICMP_PROT_UNREACH:
		strcpy(additional, "protocol");
		break;
	    case ICMP_PORT_UNREACH:
		strcpy(additional, "port");
		break;
	    case ICMP_FRAG_NEEDED:
		strcpy(additional, "DF set");
		break;
	    case ICMP_SR_FAILED:
		strcpy(additional, "src route failed");
		break;
	    case ICMP_NET_UNKNOWN:
		strcpy(additional, "net unkn");
		break;
	    case ICMP_HOST_UNKNOWN:
		strcpy(additional, "host unkn");
		break;
	    case ICMP_HOST_ISOLATED:	/* obsolete */
		strcpy(additional, "src isoltd");
		break;
	    case ICMP_NET_ANO:
		strcpy(additional, "net comm denied");
		break;
	    case ICMP_HOST_ANO:
		strcpy(additional, "host comm denied");
		break;
	    case ICMP_NET_UNR_TOS:
		strcpy(additional, "net unreach for TOS");
		break;
	    case ICMP_HOST_UNR_TOS:
		strcpy(additional, "host unreach for TOS");
		break;
	    case ICMP_PKT_FILTERED:
		strcpy(additional, "pkt filtrd");
		break;
	    case ICMP_PREC_VIOLATION:
		strcpy(additional, "prec violtn");
		break;
	    case ICMP_PREC_CUTOFF:
		strcpy(additional, "prec cutoff");
		break;
	    }

	    break;
	case ICMP_SOURCE_QUENCH:
	    strcpy(description, "source quench");
	    break;
	case ICMP_REDIRECT:
	    strcpy(description, "redirect");
	    break;
	case ICMP_TIME_EXCEEDED:
	    strcpy(description, "time exceeded");
	    break;
	case ICMP_PARAMETERPROB:
	    strcpy(description, "parameter problem");
	    break;
	case ICMP_TIMESTAMP:
	    strcpy(description, "timestamp request");
	    break;
	case ICMP_INFO_REQUEST:
	    strcpy(description, "information request");
	    break;
	case ICMP_INFO_REPLY:
	    strcpy(description, "information reply");
	    break;
	case ICMP_ADDRESS:
	    strcpy(description, "address mask request");
	    break;
	case ICMP_ADDRESSREPLY:
	    strcpy(description, "address mask reply");
	    break;
	default:
	    strcpy(description, "bad/unknown");
	    break;
	}

    } else if (entry->protocol == IPPROTO_OSPFIGP) {
	switch (entry->un.ospf.type) {
	case OSPF_TYPE_HELLO:
	    strcpy(description, "hlo");
	    break;
	case OSPF_TYPE_DB:
	    strcpy(description, "DB desc");
	    break;
	case OSPF_TYPE_LSR:
	    strcpy(description, "LSR");
	    break;
	case OSPF_TYPE_LSU:
	    strcpy(description, "LSU");
	    break;
	case OSPF_TYPE_LSA:
	    strcpy(description, "LSA");
	    break;
	}
	sprintf(additional, "a=%lu r=%s", entry->un.ospf.area,
		entry->un.ospf.routerid);
    }
    strcpy(msgstring, protname);
    strcat(msgstring, " ");

    strcat(msgstring, description);
    strcat(msgstring, " ");

    if (strcmp(additional, "") != 0) {
	sprintf(scratchpad, "(%s) ", additional);
	strcat(msgstring, scratchpad);
    }
    if (unknown) {
	strcat(msgstring, "protocol code: ");
	sprintf(scratchpad, "%u", entry->protocol);
	strcat(msgstring, scratchpad);
    }
    if (entry->protocol == IPPROTO_UDP) {
	sprintf(scratchpad, "from %s:%s to %s:%s",
		entry->s_fqdn, entry->un.udp.s_sname,
		entry->d_fqdn, entry->un.udp.d_sname);
    } else {
	sprintf(scratchpad, "from %s to %s", entry->s_fqdn, entry->d_fqdn);
    }

    strcat(msgstring, scratchpad);
    strcat(msgstring, " on ");
    strcat(msgstring, entry->iface);

    wmove(table->othpwin, target_row, 0);
    scrollok(table->othpwin, 0);
    wprintw(table->othpwin, "%78c", ' ');
    scrollok(table->othpwin, 1);
    wmove(table->othpwin, target_row, 1);
    waddnstr(table->othpwin, msgstring, 76);

    writelog(logging, logfile, msgstring);
}

void destroyothptable(struct othptable *table)
{
    struct othptabent *ctemp;
    struct othptabent *ctemp_next;

    if (table->head != NULL) {
	ctemp = table->head;
	ctemp_next = table->head->next_entry;

	while (ctemp != NULL) {
	    free(ctemp);
	    ctemp = ctemp_next;

	    if (ctemp_next != NULL)
		ctemp_next = ctemp_next->next_entry;
	}
    }
}
