/*  General Configuration module written in C 
    Copyright (C) 1998  Charles P. Wright

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    You may link this program with the Qt Free Edition.
*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <syslog.h>

#include "mserver.h"

#ifdef USE_HASH
extern struct htab *hashtab[HASHELEMENTS];
#else
struct param_dbase_t dbase;
#endif

FILE *cfile;
char filename[1024];
char **saveval;
int savenum;

bool config_init (const char *name)
{
	char s[1024];
	char *retval;
	char key[1024];
	char data[1024];
	int i, n;
	
#ifdef USE_HASH
	inittab();
#else
	init_param_dbase(&dbase);
#endif

	strncpy (filename, name, 1024);
	cfile = fopen (filename, "r+");

	if (cfile != NULL)
	{
		retval = NULL;	// Make sure that retval doesn't start as NULL
		retval++;

		while (retval != NULL)
		{
			retval = fgets(s, 1023, cfile);

			if (retval != NULL)	
			{
				for (i = 0; i < strlen(s); i++)
				{
					if (s[i] == '#')
					{
						s[i] = '\0';
					}
				}
			
				if (strlen(s) != 0)
				{
		
					for (i = 0, n = 0; i < strlen(s) && s[i] != '='; i++)
					{
						if (!isspace(s[i]))
						{
							key[n++] = s[i];
						}
					}
					key[n] = '\0';
	
					while (isspace(s[++i]));

					if (s[i] == '"')
					{
						for (i++, n = 0; i < strlen(s) && s[i] != '"'; i++)
						{
							data[n++] = s[i];
						}
						data[n] = '\0';
					}
					else
					{
						for (n = 0; i < strlen(s); i++)
						{
							if (!isspace(s[i]))
							{
								data[n++] = s[i];
							}
						}
						data[n] = '\0';
					}

					config_setvalue(key, data);
				}
			}
		}
		fclose (cfile);

		config_setvalue ("version", "C Masqdialer Server 0.0.9 (Charles P. Wright)");
		config_setvalue ("author", "Charles P. Wright");

		return true;
	}
	else
	{
		return false;
	}

}

void config_refresh (int sig)
{
	syslog (LOG_NOTICE, "Recieved signal: %d", sig);

	config_clear();
	config_init(filename);

	syslog (LOG_INFO, "Reloaded configuration file: %s", filename);
	syslog (LOG_INFO, "Already existing children or siblings will not reread configuration file, unless you specifically send them the HUP signal");
	syslog (LOG_INFO, "Note not all values with take effect until you restart the masqdialer server. (port, slimit, ...)");
}

void config_close_save (void)
{
	char *key;
	char data[1024];
	int i;
#ifdef USE_HASH
	struct htab *curhash;
#endif

	cfile = fopen (filename, "w+");

#ifdef USE_HASH
	for (i = 0; i < HASHELEMENTS; i++)
	{
		if (hashtab[i] != NULL)
		{
			fprintf (cfile, "%s = %s\n", hashtab[i]->key, hashtab[i]->data);
			curhash = hashtab[i];
			while (curhash->child != NULL)
			{
				curhash = curhash->child;
				fprintf (cfile, "%s = %s\n", hashtab[i]->key, hashtab[i]->data);
			}
		}
	}
#else
#endif
	
	fclose(cfile);
}

void config_clear (void)
{
#ifdef USE_HASH
	char *key;
	char data[1024];
	int i;
	struct htab *curhash;
	struct htab * oldhash;

	for (i = 0; i < HASHELEMENTS; i++)
	{
		if (hashtab[i] != NULL)
		{
			curhash = hashtab[i];
			hashtab[i] = NULL;
			while (curhash->child != NULL)
			{
				oldhash = curhash;
				curhash = curhash->child;
				oldhash = NULL;
			}
			curhash = NULL;
		}
	}
#else
	clear_param_dbase(&dbase);	
#endif
}

void config_getvalue(const char *key, char *value, int maxchar)
{
#ifdef USE_HASH
	struct htab *rethash;

	rethash = findhash((char *) key);
	if (rethash != NULL)
	{
		strncpy(value, rethash->data , maxchar);
	}
	else
	{
		value[0] = '\0';
	}
#else
	strncpy(value, (char *) param_dbase_getval(&dbase, key), maxchar);
#endif
}

int config_getvalue_int (const char *key)
{
#ifdef USE_HASH
	struct htab *rethash;

	rethash = findhash((char *) key);
#endif

#ifdef USE_HASH
	if (rethash != NULL)
	{
		return (atoi(rethash->data));
	}
	else
	{
		return 0;
	}
#else
		return (atoi((char *) param_dbase_getval(&dbase, key)));
#endif
}

void config_getvalue_default (const char *key, char *value, char *defval, int maxchar)
{
#ifdef USE_HASH
	struct htab *rethash;

	rethash = findhash((char *) key);
	if (rethash != NULL)
	{
		strncpy(value, rethash->data , maxchar);
	}
	else
	{
		strncpy(value, defval, maxchar);
	}
#else
	char *retstr;

	retstr = (char *) param_dbase_getval(&dbase, key);

	if (retstr != NULL)
	{
		strncpy(value, retstr , maxchar);
	}
	else
	{
		strncpy(value, defval, maxchar);
	}
#endif
}

bool config_getvalue_bool (const char *key, bool defval)
{
	char value[1024];
	int maxchar = 1024;
#ifdef USE_HASH
	struct htab *rethash;

	rethash = findhash((char *) key);
	if (rethash != NULL)
	{
		strncpy(value, rethash->data , maxchar);
	}
	else
	{
		value[0] = '\0';
	}
#else
	char *retstr;
	
	retstr = (char *) param_dbase_getval(&dbase, key);

	if (retstr != NULL)
	{
		strncpy(value, retstr, maxchar);
	}
	else
	{
		value[0] = '\0';
	}
#endif

	if (!strncmp(value, "true", maxchar))
	{
		return true;
	}
	else if (!strncmp(value, "false", maxchar))
	{
		return false;
	}
	else if (!strncmp(value, "yes", maxchar))
	{
		return true;
	}
	else if (!strncmp(value, "no", maxchar))
	{
		return false;
	}

	return defval;
}
	
int config_setvalue (const char *key, const char *value)
{
#ifdef USE_HASH
	struct htab *rethash;
	rethash = addhash((char *) key, value);
	if (rethash == NULL)
	{
		return 0;
	}	
	else
	{
		return 1;
	}
#else
	param_dbase_setval(&dbase, key, value);
#endif
}
