 /*
  * sock_host() determines the type of socket (datagram, stream), the name
  * and address of the host at the other end of a socket, the local socket
  * address and port, and the remote username if username lookups are done
  * irrespective of client. All results are in static memory and will be
  * overwritten upon the next call.
  * 
  * The return status is (-1) if the remote host pretends to have someone elses
  * name, or if the remote host name is available but could not be verified;
  * in either case the hostname will be ignored. The return status is zero in
  * all other cases (the hostname is unavailable, or the host name double
  * check succeeds).
  * 
  * Diagnostics are reported through syslog(3).
  * 
  * Author: Wietse Venema, Eindhoven University of Technology, The Netherlands.
  *
  * Adapted for mserver by Charles p. Wright
  */

/* System libraries. */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <syslog.h>

#include "mserver.h"

char *sock_name(struct sockaddr_in *rmt_sin);

/* sock_host - determine endpoint info */

int sock_host(int fd, char addr_string[], int limit)
{
	static struct sockaddr rmt_sa;

	struct sockaddr_in *rmt_sin;
	char *addr;
	
	int len;
	char buf[BUFSIZ];
	

	/*
	 * Look up the remote host address. Hal R. Brand <BRAND@addvax.llnl.gov>
	 * suggested how to get the remote host info in case of UDP connections:
	 * peek at the first message without actually looking at its contents.
	 */

    	len = sizeof(rmt_sa);
	if (getpeername(fd, &rmt_sa, &len) < 0) {
		len = sizeof(rmt_sa);
		return (1); /* address and name unknown */
	}

	rmt_sin = (struct sockaddr_in *) & rmt_sa;

	strncpy (addr_string, sock_name(rmt_sin), limit);

#ifdef DEBUG
	syslog (LOG_DEBUG, "remote addr: %s", addr_string);
#endif
	
	return 0;
}

/* sock_names - map IP address info to readable address and name */

char *sock_name(struct sockaddr_in *rmt_sin)
{
	static char addr_buf[MAXHOSTNAMELEN];
	static char name_buf[MAXHOSTNAMELEN];
	struct hostent *hp;
	struct in_addr addr;
	
	/*
	 * Some stupid compilers can do struct assignment but cannot do structure
 	 * initialization.
	 */

	addr = rmt_sin->sin_addr;

	/*
 	 * Map the address to human-readable form.
	 */

	#define STRNCP(d,s,l) { strncpy((d),(s),(l)); ((d)[(l)-1] = 0); }

	STRNCP(addr_buf, inet_ntoa(addr), sizeof(addr_buf));

	return addr_buf;
}
