!**************************************************************************
!*
!* Boot-ROM-Code to load an operating system across a TCP/IP network.
!*
!* Module:  lddisk.S
!* Purpose: Load a boot sector from a disk
!* Entries: _loaddisk
!*
!**************************************************************************
!*
!* Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
!*
!*  This program is free software; you can redistribute it and/or modify
!*  it under the terms of the GNU General Public License as published by
!*  the Free Software Foundation; either version 2 of the License, or
!*  any later version.
!*
!*  This program is distributed in the hope that it will be useful,
!*  but WITHOUT ANY WARRANTY; without even the implied warranty of
!*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!*  GNU General Public License for more details.
!*
!*  You should have received a copy of the GNU General Public License
!*  along with this program; if not, write to the Free Software
!*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
!*


!
!**************************************************************************
!
! Include assembler macros:
!
#include <macros.inc>
#include <memory.inc>


!
!**************************************************************************
!
! Miscellaneous definitions
!
HD_BASE		equ	$80		! base number for all harddisks
FD_BASE		equ	$00		! base number for all floppy disks

BOOT_BLOCK	equ	$7C00		! offset to boot block after loading

BOOT_SIG	equ	$AA55		! boot block signature
BOOT_SIG_OFS	equ	$01FE		! offset to boot block signature

PART_BUF_OFS	equ	$0200		! offset to partition table buffer
PART_OFS	equ	$03BE		! offset to partition table
PART_HEAD_OFS	equ	$0001		! offset to head number in part entry
PART_SECT_OFS	equ	$0002		! offset to sector number in part entry
PART_TYPE_OFS	equ	$0004		! offset to partition type indicator

SECTSIZE	equ	512		! size of one sector

ENTRY_LEN	equ	$0010		! size of one partition table entry
ENTRY_NUM	equ	4		! maximum number of entries in part tab

EXT_PART_TYPE	equ	5		! type ID for extended partition


!
!**************************************************************************
!
! BSS segment
!
	.bss

	.lcomm	partnum,1		! partition number


!
!**************************************************************************
!
! Start code segment.
!
	.text

	public	_loaddisk		! define entry points


!
!**************************************************************************
!
! Load a disk boot sector into memory
! Input:  1. arg  -  pointer to name of device to load from
! Output: returns if not successful, otherwise doesnt return
!
_loaddisk:

	penter
	push	es
	push	si
	getcarg	(bx,0)
	call	decodestr		! decode device string
	jc	load9
	mov	partnum,ah
	mov	dl,al
	xor	ah,ah
	int	$13			! reset disk system

! First load the master boot record.

	mov	ax,#LOWMEM
	mov	es,ax
	mov	bx,#PART_BUF_OFS
	mov	cx,#$0001		! load first sector on track 0
	xor	dh,dh			! head 0
	call	reads1			! load the sector
	jc	load9

! Next load the first sector of the partition. If we are to load from a
! partition in an extended partition, we have to read the extended partition
! table first.

	mov	bl,partnum
	or	bl,bl			! check if thats all
	jz	load3
	cmp	bl,#5			! check if we have to load from an
	jb	load6			! extended partition

	mov	bx,#PART_OFS		! search for an extended partition
	mov	cx,#ENTRY_NUM
load1:	seg	es
	cmp	byte ptr PART_TYPE_OFS[bx],#EXT_PART_TYPE
	je	load2
	add	bx,#ENTRY_LEN		! continue with next entry
	loop	load1
	jmp	load9

load2:	mov	si,bx
	mov	bx,#PART_BUF_OFS
	call	readsect		! read extended partition table
	jc	load9
	mov	bl,partnum		! compute partition number in extended
	sub	bl,#4			! table
	cmp	bl,#5
	jae	load9

! Now really load the first sector of the boot partition

load6:	dec	bl			! partition numbers start with 1
	xor	bh,bh
	shift	(shl,bx,4)		! compute offset into partition table
	lea	si,PART_OFS[bx]
	xor	bx,bx
	call	readsect		! read first sector of boot partition
	jnc	load3

load9:	pop	si			! return in case of error
	pop	es
	pleave
	ret

! After the boot sector has been read, we can disable all bootrom services,
! restore the interrupt vectors etc., then copy the boot sector to where it
! belongs, and finally call it.

load3:	push	dx			! save disk ID for later
	push	si			! save offset to partition table entry
	mov	ah,#1			! terminate bootrom services
	int	SERVICES		! handled in kernel.S module
	pop	bx
	pop	dx

	cli
	xor	ax,ax
	mov	ss,ax
	mov	sp,#BOOT_BLOCK - 2	! set stack to beginning of boot block
	sti
	xor	ah,ah			! reset disk system
	int	$13

	cld
	mov	ax,#LOWMEM
	mov	ds,ax
	xor	si,si
	mov	es,si			! copy boot sector and partition table
	mov	di,#BOOT_BLOCK		! to their final destination
	mov	cx,#SECTSIZE
	rep
	movsw
	xor	ax,ax
	mov	ds,ax			! restore pointer to partition entry
	lea	si,BOOT_BLOCK[bx]	! in DS:SI
	jmpi	BOOT_BLOCK,0		! call the boot sector


!
!**************************************************************************
!
! Decode device string
! Input:  BX  -  pointer to string
! Output: AL  -  disk drive ID
!         AH  -  partition number
!         Carry flag set if error
! Registers changed: AX, BX
!
decodestr:

! First decode the first two letters. The string has to start either
! with 'fd' or 'hd'. We dont any other devices yet.

	mov	ah,[bx+2]
	mov	al,#HD_BASE
	cmp	word ptr [bx],#$6468	! check for hard disk 'hd'
	je	decod1
	mov	al,#FD_BASE
	cmp	word ptr [bx],#$6466	! check for floppy disk 'fd'
	jne	decod8

! OK, we have to look for a floppy device. After the initial 'fd' a number
! has to follow. The maximum is 4 disk drives, so check that its in the
! range 0 to 3.

	sub	ah,#$30			! check for correct number
	jb	decod8
	cmp	ah,#3
	ja	decod8
	add	al,ah			! compute final device number
	xor	ah,ah			! partition number is always zero
	add	bx,#3			! let BX point after last character
	jmp	decod7

! Now decode the harddisk description. The initial 'hd' has to be followed
! by a letter from 'a' to 'd' indicating the drive number, followed by a
! number indicating the partition number. If the number is missing, 0 is
! assumed, which means the whole disk, e.g. the master boot block.

decod1:	sub	ah,#$61			! check for correct letter
	jb	decod8
	cmp	ah,#3
	ja	decod8
	add	al,ah			! compute final device number
	add	bx,#3			! let BX point after last character
	mov	ah,[bx]
	or	ah,ah			! end of string means "zero"
	jz	decod7
	sub	ah,#$30			! check for correct partition
	jb	decod8
	cmp	ah,#8			! maximum number of 8 partitions
	ja	decod8
	inc	bx
decod7:	cmp	byte ptr [bx],#0	! string has to be followed by a null
	je	decod9			! return without error
decod8:	stc				! return with error
decod9:	ret


!
!**************************************************************************
!
! Read first sector of partition. reads1 is an additional subroutine entry
! point!
! Input:  ES:SI  -  pointer to partition table entry
!         ES:BX  -  pointer to sector buffer
!         DL     -  disk drive ID
! Output: Carry flag set if error
! Registers changed: AX, CX, DH

readsect:

	seg	es
	mov	al,PART_TYPE_OFS[si]
	or	al,al				! check if partition table entry
	jz	reads8				! is valid

	seg	es
	mov	cx,PART_SECT_OFS[si]		! get sector, track and head
	seg	es				! number from partition table
	mov	dh,PART_HEAD_OFS[si]
reads1:	mov	ax,#$0201			! actually read the sector
	int	$13
	jc	reads8
	seg	es
	cmp	word ptr BOOT_SIG_OFS[bx],#BOOT_SIG
	je	reads9
reads8:	stc
reads9:	ret


!
!**************************************************************************
!
	end

