! flash.S  -  FlashCard flash EPROM programmer
!
! Copyright (C) 1997,1998 Gero Kuhlmann   <gero@gkminix.han.de>
!
!  This program is free software; you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation; either version 2 of the License, or
!  any later version.
!
!  This program is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with this program; if not, write to the Free Software
!  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


#ifndef ASM_DEBUG
#undef ASM_DEBUG
#endif
#include "flash.inc"
#include "version.h"

	.text

	.org	0

	mov	dx,ds
	mov	ax,cs			! set DS
	mov	ds,ax
	mov	oldES,es
	mov	oldDS,dx		! save old register values in case
	mov	oldBP,bp		! we have to return to the boot rom
	mov	oldSI,si
	mov	oldDI,di
	mov	bp,sp
	mov	ax,[bp+4]
	mov	[header+0],ax		! load the address of the boot header
	mov	ax,[bp+6]
	mov	[header+2],ax
	mov	ax,[bp+8]
	mov	[bootp+0],ax		! load the address of the bootp block
	mov	ax,[bp+10]
	mov	[bootp+2],ax

! Tell the user who we are and that we started running

	mov	si,#sigmsg
	call	prnstr

! Check if the boot image header is correct.

	les	bx,header
	mov	si,#bmerr		! prepare for correct error message
	seg	es
	mov	ax,BOOT_HD_MAGIC+0[bx]
	cmp	ax,[bmagic+0]		! compare boot rom magic numbers
	jne	doerr1
	seg	es
	mov	ax,BOOT_HD_MAGIC+2[bx]
	cmp	ax,[bmagic+2]
	jne	doerr1

	mov	si,#vmerr		! prepare for correct error message
	seg	es
	mov	al,BOOT_HD_LENGTH[bx]
	mov	cl,#4
	shr	al,cl
	and	al,#0x0F
	cmp	al,#VENDOR_SIZE		! check vendor ID size
	jne	doerr1
	xor	di,di
dovmag:	mov	al,vmagic[di]		! check vendor ID
	or	al,al
	jz	getrom			! vendor ID ok, continue
	seg	es
	cmp	al,BOOT_HD_VENDOR[bx+di]
	jne	doerr1
	inc	di
	jmp	dovmag

doerr1:	call	prnstr			! in case of error return to the
	mov	si,oldSI		! boot rom with registers set
	mov	di,oldDI		! correctly
	mov	bp,oldBP
	mov	es,oldES
	mov	ds,oldDS
	retf

! Next get the address of the boot rom image

getrom:	mov	si,#recerr
	mov	al,#VENDOR_ROMIMAGE
	call	fndldr				! find load record for rom image
	mov	ax,es
	or	ax,di
	jz	doerr1
	seg	es
	mov	al,BOOT_LD_FLAGS[di]		! get load record flags
	test	al,#BOOT_FLAG_B0 + BOOT_FLAG_B1	! check that it has a
	jnz	doerr1				! correct flag

	seg	es
	mov	ax,BOOT_LD_ADDR+0[di]		! get base adress of rom image
	seg	es
	mov	bx,BOOT_LD_ADDR+2[di]
	mov	[romadr+0],ax
	mov	[romadr+2],bx

	seg	es
	mov	ax,BOOT_LD_MLENGTH+0[di]	! get rom image size
	seg	es
	mov	bx,BOOT_LD_MLENGTH+2[di]
	add	ax,#0x03ff			! round to nearest kb
	adc	bx,#0
	and	ax,#0xfc00
	or	bx,bx				! cant be larger than 64 kB
	jnz	doerr1
	mov	romsiz,ax

! Now try to find a flash EPROM in the upper memory block

	mov	bx,#SEGLOW
fndepr:	mov	es,bx
	call	readid				! read flash EPROM ID
	jnc	fndit
	cmp	bx,#SEGHIGH			! dont go into BIOS ROM area
	jae	notfnd
	add	bx,#SEGSTEP			! continue with next 8kB block
	jmp	fndepr

notfnd:	mov	si,#eprerr			! no flash EPROM found
	jmp	doerr2

fndit:	mov	eprseg,es			! found a flash EPROM
	mov	si,#adrmsg			! let the user know about the
	call	prnstr				! address
	mov	ax,eprseg
	call	prnwrd
	mov	si,#crlf
	call	prnstr

! Erase the flash EPROM

	call	erase				! erase the chip
	jnc	doprog
	mov	si,#eraerr			! error
doerr2:	jmp	near doerr1

! Program the flash EPROM

doprog:	mov	si,#prgmsg
	call	prnstr				! print programming message

	cld
	xor	di,di				! set programming address
	mov	ax,word ptr [romadr+0]
	mov	bx,word ptr [romadr+2]		! get address of rom image
	mov	si,ax
	and	si,#0x000F			! convert linear address into
	mov	cl,#12				! seg:ofs
	shl	bx,cl
	and	ax,#0xFFF0
	add	ax,bx
	mov	cx,romsiz			! get rom image size
	push	ds
	mov	ds,ax

prog1:	mov	al,#PROG_CMD
	call	sendop				! send programming command
	lodsb					! get next byte from rom image
	seg	es
	mov	byte ptr [di],al		! write next byte into EPROM
	call	waitop				! wait until programming
	jc	prog2				! operation is completed
	inc	di
	loop	prog1				! continue with next byte
	jmp	prog3

prog2:	pop	ds
	mov	si,#prgerr			! got a programming timeout
	jmp	doerr2

prog3:	pop	ds
	mov	si,#endmsg
	call	prnstr				! print end message
	xor	ax,ax
	int	0x16				! wait for a key press
	xor	ax,ax
	mov	ds,ax
	mov	word ptr [0x0472],#0x1234	! indicate warm boot
	jmpi	0,0xFFFF			! reboot the system



!====================================================================
!
! Find a load record in the boot header. The ID number of the load
! record is in AL, and ES:DI points to requested load record, or is
! the NULL pointer if load record not found.
!
! Changed registers: AX, DI, ES

fndldr:	push	cx
	mov	ch,al
	les	di,header		! examine boot image header
	seg	es
	mov	al,BOOT_HD_LENGTH[di]	! get length of image header
	call	getlen
	add	di,ax			! get the pointer to first load record
fndl1:	seg	es
	cmp	ch,BOOT_LD_TAG1[di]	! is it the desired one ?
	je	fndl3
	seg	es
	mov	al,BOOT_LD_FLAGS[di]	! no, so check if its the last record
	test	al,#BOOT_FLAG_EOF
	jnz	fndl2
	seg	es
	mov 	al,BOOT_LD_LENGTH[di]	! no, get the address of the next one
	call	getlen
	add	di,ax
	jmp	fndl1

fndl2:	xor	ax,ax			! couldnt find the desired record
	mov	es,ax
	mov	di,ax
fndl3:	pop	cx
	ret



!====================================================================
!
! Compute the length of a load record address from a length byte
! in AL. Return the offset in AX.
!
! Changed registers: AX

getlen:	push	cx
	mov 	ah,al
	mov 	cl,#4
	shr	ah,cl
	and	ax,#0x0f0f		! compute the total length in
	add	al,ah			! bytes from the length of the
	xor	ah,ah			! record and that of the vendor
	shl	ax,1			! information.
	shl	ax,1
	pop	cx
	ret



!====================================================================
!
! Read manufacturer ID from Flash-EPROM. This routine is used to scan
! through the upper data area to look for a Flash-EPROM. Since it is
! possible to run into some RAM area (for example dual-ported ram of
! a network card), we have to preserve all bytes which are changed,
! and the whole process has to be atomic.
!
! Input:  ES  -  segment of Flash EPROM
! Output: Carry flag set, if not a Flash EPROM at current segment
! Changed registers: AX

readid:	push	bx
	push	cx
	push	dx
	seg	es			! if we can read the AMD ID already,
	mov	ax,[0x0000]		! this cannot be a Flash EPROM: the
	cmp	ax,#AMD_ID		! bootrom doesnt start with this ID
	je	rdid8			! and an empty Flash EPROM has all 0xFF

	mov	cx,#3			! set number of retries
rdid1:	cli				! disable all interrupts
	seg	es
	mov	dl,byte ptr [CMDOFS1]	! save command bytes
	seg	es
	mov	dh,byte ptr [CMDOFS2]
	mov	al,#READID_CMD
	call	sendop			! send READID command
	seg	es
	mov	bx,[0x0000]		! read manufacturer ID
	mov	al,#RESET_CMD
	call	sendop			! reset chip
	seg	es
	mov	byte ptr [CMDOFS1],dl	! restore command bytes
	seg	es
	mov	byte ptr [CMDOFS2],dh
	sti				! re-enable all interrupts again
	cmp	bx,#AMD_ID		! check for correct ID
	je	rdid9
	loop	rdid1			! retry
rdid8:	stc				! not found, return error
rdid9:	pop	dx
	pop	cx
	pop	bx
	ret



!====================================================================
!
! Erase whole chip
!
! Input:  ES  -  segment of Flash EPROM
! Output: carry flag set if error
! Registers changed: AX, BX

erase:	mov	al,#ERASE1_CMD
	call	sendop			! send ERASE1 command
	mov	al,#ERASE2_CMD
	call	sendop			! send ERASE2 command
	xor	bx,bx
	mov	al,#0xFF
	call	waitop			! wait until operation finished
	jnc	erase9
	mov	al,#RESET_CMD
	call	sendop			! reset chip
	stc
erase9:	ret



!====================================================================
!
! Send OP code to Flash-EPROM. This involves writing three bytes into the
! flash EPROM at exactly specified locations. See AMD data sheets for
! further information.
! Input:  AL  -  command byte
!         ES  -  segment of Flash-EPROM
! Output: none
! Registers changed: none

sendop:	seg	es
	mov	byte ptr [CMDOFS2],#CMDOP2
	jcxz	sndop1
sndop1:	jcxz	sndop2
sndop2:	seg	es
	mov	byte ptr [CMDOFS1],#CMDOP1
	jcxz	sndop3
sndop3:	jcxz	sndop4
sndop4:	seg	es
	mov	byte ptr [CMDOFS2],al
	ret



!====================================================================
!
! Wait for the chip to process programming/erase. When programming is in
! progress, the flash EPROM toggles the highest bit, and converts it back
! to its original value when the programming is done. For a further ex-
! planation of this algorithm see the AMD data sheets.
!
! Input:  AL  -  programmed byte
!         DI  -  offset to programming location
!         ES  -  segment of Flash EPROM
! Output: carry flag set if error
! Registers changed: AX

waitop:	push	cx
	and	al,#%10000000
wait1:	seg	es
	mov	ah,byte ptr [di]
	mov	ch,ah
	and	ah,#%10000000
	cmp	al,ah
	je	wait8
	test	ch,#%00100000
	jz	wait1
	seg	es
	mov	ah,byte ptr [di]
	and	ah,#%10000000
	cmp	al,ah
	je	wait8
	stc
	jmp	wait9
wait8:	clc
wait9:	pop	cx
	ret



!====================================================================
! Print a string in DS:SI onto the console
!
! Changed registers: AL

prnstr:	push	si
	cld
prns1:	lodsb				! loop over all characters of
	or	al,al			! string
	jz	prns2
	push	bx
	mov	ah,#0x0E		! print it
	mov	bl,#0x07
	xor	bh,bh
	int	0x10
	pop	bx
	jmp	prns1
prns2:	pop	si
	ret



!====================================================================
!
! Print hexadecimal values (in AX or AL) or characters onto the console
!
! Changed registers: AX

prnwrd:	push	ax
	mov	al,ah
	call	prnbyt			! print the upper byte
	pop	ax
prnbyt:	push	ax
	shr	al,1			! prepare upper nibble
	shr	al,1
	shr	al,1
	shr	al,1
	call	prnnib			! print it
	pop	ax
prnnib:	and	al,#0x0F		! prepare lower nibble
	add	al,#0x30
	cmp	al,#0x39		! convert it into hex
	jle	prnchr
	add	al,#7
prnchr:	push	bx
	mov	ah,#0x0E		! print it
	mov	bl,#0x07
	xor	bh,bh
	int	0x10
	pop	bx
	ret



!====================================================================
!
! String and constants definitions


! Startup signature

sigmsg:	.byte	0x0D, 0x0A
	.ascii	"FlashCard Flash EPROM programmer "
	.ascii	VERSION
	.byte	0x0D, 0x0A
	.ascii	COPYRIGHT
	.byte	0x0D, 0x0A
crlf:	.byte	0x0D, 0x0A
	.byte	0


! Magic numbers for boot record

bmagic:	.long	BOOT_MAGIC		! boot image magic number
vmagic:	.ascii	VENDOR_MAGIC		! vendor magic ID
	.byte	0			! end of vendor magic ID


! Error messages

recerr:	.ascii	"Error in load record data"
	.byte	0x0D, 0x0A
	.byte	0

bmerr:	.ascii	"Invalid boot header magic number"
	.byte	0x0D, 0x0A
	.byte	0

vmerr:	.ascii	"Invalid vendor magic ID"
	.byte	0x0D, 0x0A
	.byte	0

eprerr:	.ascii	"Could not find a Flash EPROM"
	.byte	0x0D, 0x0A
	.byte	0

eraerr:	.ascii	"Unable to erase Flash EPROM"
	.byte	0x0D, 0x0A
	.byte	0

prgerr:	.byte	0x0D, 0x0A
	.ascii	"Timeout while programming Flash EPROM"
	.byte	0x0D, 0x0A
	.byte	0


! Other messages to be printed for user information

adrmsg:	.asciz	"Found Flash EPROM at address "
prgmsg:	.asciz	"Programming... "
endmsg:	.ascii	"done"
	.byte	0x0D, 0x0A
	.asciz	"Press any key to reboot and activate the new rom image..."



!====================================================================
!
! Variable definitions

header:	.long	0			! pointer to boot header from boot rom
bootp:	.long	0			! pointer to bootp block from boot rom

oldDS:	.word	0			! old DS from boot rom
oldES:	.word	0			! old ES from boot rom
oldBP:	.word	0			! old BP from boot rom
oldSI:	.word	0			! old SI from boot rom
oldDI:	.word	0			! old DI from boot rom

romadr:	.long	0			! adress of rom image
romsiz:	.word	0			! size of rom image
eprseg:	.word	0			! flash EPROM segment

