/* filename: rlpr-rlpr.c
 * project: rlpr
 * author: meem  --  meem@sherilyn.wustl.edu
 * version: $Id: rlpr-rlpr.c,v 1.1 1997/02/22 08:58:13 meem Exp meem $
 * contents: contains routines for the `rlpr' client of the rlpr package
 *
 * Time-stamp: <1997/04/27 04:12:07 cdt -- meem@sherilyn.wustl.edu>
 */

/* copyright (c) 1996, 1997 meem, meem@gnu.ai.mit.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include "config.h"

#include <sys/types.h>  
#include <stdlib.h>             /* for malloc(), getenv(), ..  */
#include <stdio.h>              /* ...                         */
#include <unistd.h>             /* for getuid(), geteuid(), .. */
#include <string.h>             /* for strchr(), ..            */
#include <fcntl.h>              /* O_* constants               */
#include <pwd.h>                /* for getpwuid()              */
#include <assert.h>

#include "getopt.h"

#include "rlpr-rlpr.h"
#include "rlpr-msg.h"                 /* rlpr messaging utility */
#include "rlpr-util.h"                /* utility routines */
#include "rlpr-client.h"              /* common to parts of the rlpr client */
#include "rlpr-rfc1179.h"

static rlpr_rlpr_info rlpr_rlpr_props = { 0, /* all args */ };
extern struct option  rlpr_long_opts[]; 
extern const char *   rlpr_opts;

char * program_name;		      /* our name */

int main(int argc, char *argv[])
{
  int    cfd;                         /* control file descriptor */
  char * cfname;                      /* control file name */
  char * cffullname;                  /* full pathname to control file */
  char * dfname;                      /* data file name */
  int    nf;                          /* number of files spooled */
  char * localhost;                   /* our local hostname */
  int    sockfd    = -1;              /* our socket connection */
  int    orig_argc = argc;            /* original # args */

  program_name = argv[0];

  toggle_euid();                      /* lose euid of root */

  /* this sets the rlpr_rlpr_props and net_ structures */

  argc -= rlpr_rlpr_args(orig_argc, argv);
  argv += orig_argc - argc;

  /* create name templates for temp files */

  localhost = rlpr_client_get(RLPR_CLIENT_LOCALHOST);
  
  cfname     = rlpr_malloc(strlen(localhost) + sizeof "cfAxxx");
  sprintf(cfname, "cfA%03d%s", getpid() % 1000, localhost);
  dfname     = rlpr_strdup(cfname);
  dfname[0]  = 'd';
  cffullname = rlpr_malloc(strlen(cfname) + strlen(rlpr_rlpr_props.tmpdir) + 2);
  sprintf(cffullname, "%s/%s", rlpr_rlpr_props.tmpdir, cfname);

  if (!rlpr_rlpr_props.wflag) {       /* if not connecting to a winNT lpd */
    sockfd = rlpr_client_open();      /* server now connected */
    rlpr_send_job_req(sockfd);
  }

  nf = argc ? argc : 1;               /* find out number of files to print */

  /*
   * open the cf once.  we're going to keep reusing the same one
   * on the client side because we only need one cf at a time
   */
  
  if ((cfd = open(cffullname, O_RDWR|O_TRUNC|O_CREAT, 0600)) < 0)
    rlpr_msg(FATAL, ERRNO, "%s - cannot create", cffullname);
  if (unlink(cffullname) < 0)
    rlpr_msg(FATAL, ERRNO, "%s - cannot unlink", cffullname);
  
  do {
    size_t copies;

    if (rlpr_rlpr_props.wflag) {             /* if connecting to a winNT lpd */
      sockfd = rlpr_client_open();
      rlpr_send_job_req(sockfd);
    }

    lseek(cfd, 0, SEEK_SET);                 /* reposition at start of file */

    rlpr_cf_init(cfd, argc ? *argv : NULL);  /* initialize the control file */

    for (copies = 0; copies < rlpr_rlpr_props.copies; copies++)
      rlpr_cf_add(rlpr_rlpr_props.filetype, dfname, cfd);
    
    rlpr_cf_add('U', dfname, cfd);

    /* from how i read the rfc, this next 'N' line should not be necessary.
     * however, berkeley lpd's seem to require it here in order for correct
     * processing of the burst page. berkeley nonstandard? :-) never
     */

    {
      static char file[MAX_SOURCE_LEN + 1];  /* just to be safe  */
      rlpr_cf_add('N', strncpy(file, argc ? *argv : "", MAX_SOURCE_LEN), cfd);
    }

    rlpr_cf_send(sockfd, cfd, cfname);

    rlpr_df_send(sockfd, argc ? *argv : NULL, dfname);
    if (rlpr_rlpr_props.rflag && argc)       /* unlink the file if wanted */
      if (unlink(*argv) < 0)                 /* and if it's possible...   */
        rlpr_msg(WARNING, ERRNO, "%s - cannot unlink", *argv);
    
    argv++;
    dfname[2]++;                             /* letter++ (dfA --> dfB), etc */
    cfname[2]++;        

    if (rlpr_rlpr_props.wflag) rlpr_client_close(sockfd); /* close under NT */
  } while (--argc > 0);
  
  if (!rlpr_rlpr_props.wflag) rlpr_client_close(sockfd);  /* close under unix */
  
  rlpr_msg(INFO, NO_ERRNO, "%i file%s spooled to %s:%s (proxy: %s)",
           nf, nf == 1 ? "" : "s", rlpr_client_get(RLPR_CLIENT_PRINTHOST),
           rlpr_client_get(RLPR_CLIENT_PRINTER), 
           rlpr_client_get(RLPR_CLIENT_PROXYHOST) ?
           rlpr_client_get(RLPR_CLIENT_PROXYHOST) : "none");
  
  /* free() stuff.. totally unnecessary but morally satisfying */

  free(cffullname);
  free(cfname);
  free(dfname);
  
  exit(EXIT_SUCCESS);
}

static void rlpr_rlpr_init(void)
{
  rlpr_rlpr_props.fflag    = 0;                 /* don't auto-form feed by default */
  rlpr_rlpr_props.bflag    = 1;                 /* burst page on by default */
  rlpr_rlpr_props.mflag    = 0;                 /* don't mail after printing */
  rlpr_rlpr_props.rflag    = 0;                 /* don't remove after printing */
  rlpr_rlpr_props.wflag    = 0;                 /* windows braindead mode (off) */
  rlpr_rlpr_props.copies   = 1;                 /* number of copies to make */
  rlpr_rlpr_props.job      = NULL;
  rlpr_rlpr_props.class    = NULL;
  rlpr_rlpr_props.title    = NULL;
  rlpr_rlpr_props.filetype = 'f';               /* default to a regular file */
  rlpr_rlpr_props.width    = 0;
  rlpr_rlpr_props.indent_cols = NULL;           /* number of columns to indent output */

  { /* find out what user we are -- user can always provide an alternate
     * choice with the -u option
     */

    struct passwd *pw;
    if ((pw = getpwuid(getuid())) != NULL)
      rlpr_rlpr_props.user = rlpr_strdup(pw->pw_name);
    else rlpr_msg(FATAL, ERRNO, "who are you?");
  }
  {
    char *tmp = getenv("TMPDIR");
    rlpr_rlpr_props.tmpdir   = tmp ? tmp : DEFAULT_TMP_DIR;
  }
}

static void rlpr_rlpr_fini(void)
{
  /* initialize rlpr_msg to be quiet if we were invoked as "lpr" */

  int islpr = !strcmp(program_name, "lpr");
  rlpr_msg_set(RLPR_MSG_QUIET, &islpr);

  /* now we need to go through all the options and make sure they're
   * not too large.. if they are, we may overflow the limits of some
   * poorly-coded lpd's
   */

  rlpr_msg(DEBUG, NO_ERRNO, "checking user-supplied params in main module...");

  if (rlpr_rlpr_props.title)
    assert(strlen(rlpr_rlpr_props.title)       < MAX_TITLE_LEN);
  if (rlpr_rlpr_props.job)   
    assert(strlen(rlpr_rlpr_props.job)         < MAX_JOB_LEN);
  if (rlpr_rlpr_props.user)   
    assert(strlen(rlpr_rlpr_props.user)        < MAX_USER_LEN);
  if (rlpr_rlpr_props.class)   
    assert(strlen(rlpr_rlpr_props.class)       < MAX_CLASS_LEN);
  if (rlpr_rlpr_props.width)   
    assert(strlen(rlpr_rlpr_props.width)       < MAX_STR_LEN);
  if (rlpr_rlpr_props.indent_cols) 
    assert(strlen(rlpr_rlpr_props.indent_cols) < MAX_STR_LEN);

  rlpr_msg(DEBUG, NO_ERRNO, "params passed in main module");
}

int rlpr_rlpr_args(int argc, char *argv[])
{
  int c, num_args = 1;               /* the number of arguments read */
  extern int optind;
  
  rlpr_rlpr_init();

  optind = 0;                         /* reset to start of argv       */

  while ((c = getopt_long(argc, argv, rlpr_opts, rlpr_long_opts, NULL)) != EOF) {
    num_args++;
    switch(c) {

    case '1':                         /* troff type R */
    case '2':                         /* troff type I */
    case '3':                         /* troff type B */
    case '4':                         /* troff type S */
    case 'c':                         /* cifplot */
    case 'd':                         /* TeX/DVI */
    case 'f':                         /* fortran */
    case 'g':                         /* graph */
    case 'l':                         /* leave control characters */
    case 'n':                         /* ditroff */
    case 'o':                         /* postscript (not in lpr) */
    case 'p':                         /* pr */
    case 't':                         /* troff */
    case 'v':                         /* raster input */
      rlpr_rlpr_props.filetype =  c == 'f' ? 'r' : c;
      break;
    case '#':                         /* number of copies to make */
      rlpr_rlpr_props.copies   = atoi(optarg);
      break;
    case 'h':                         /* suppress printing of burst page */
      rlpr_rlpr_props.bflag    = 0;
      break;
    case 'i':                         /* indentation */
      rlpr_rlpr_props.indent_cols  = optarg ? optarg : DEFAULT_INDENT_NO_PARAM;
      break;
    case 'm':                         /* send mail after printing */
      rlpr_rlpr_props.mflag    = 1;
      break;
    case 'r':                         /* remove file after printing */
      rlpr_rlpr_props.rflag    = 1;
      break;
    case 's':                         /* for compatibility with BSD lpr */
      rlpr_msg(WARNING, NO_ERRNO, "symlink option not applicable (ignored)");
      break;
    case 'w':                         /* width for pr */
      rlpr_rlpr_props.width    = optarg ? optarg : DEFAULT_WIDTH_NO_PARAM;
      break;
    case 'C':                         /* class */
      rlpr_rlpr_props.class    = optarg;
      break;
    case 'F':                         /* form feed after printing */
      rlpr_rlpr_props.fflag    = 1;
      break;
    case 'J':                         /* job name on burst page */
      rlpr_rlpr_props.job      = optarg;
      break;
    case 'T':                         /* title for pr (def: filename) */
      rlpr_rlpr_props.title    = optarg;
      break;
    case 'U':                         /* username to print on title... */
      rlpr_rlpr_props.user     = optarg;
      break;
    case 'V':                         /* print version info */
      fprintf(stdout, "%s: version "VERSION" from "__DATE__" "__TIME__ 
              " -- meem@gnu.ai.mit.edu\n", program_name);
      exit(EXIT_SUCCESS);
    case 'W':
      rlpr_rlpr_props.wflag    = 1;
      break;
    case -2:
      printf("usage: %s [-Hprinthost]  [-Pprinter]  [-Xproxyhost] [OPTIONS] "
             "[file ...]\n\nplease see the manpage for detailed help.\n",
	     program_name);
      exit(EXIT_SUCCESS);
    case -3:                         /* tmpdir to use */
      rlpr_rlpr_props.tmpdir   = optarg; 
      break;
    }
  }

  rlpr_msg_args(argc, argv);    /* initialize the messaging subsystem */
  rlpr_client_args(argc, argv); /* initialize the client subsystem    */

  rlpr_rlpr_fini();
  return num_args;
}



/* add an entry to the control file */

void rlpr_cf_add(char op, const char *str, int cfd)
{
  /* we could do an sprintf(), but we'd waste a lot of our time just
     malloc()'ing and free()'ing, so instead just employ brute force */

  safe_writen(cfd, &op, 1);
  safe_writen(cfd, str, strlen(str));
  safe_writen(cfd, "\n", 1);
}

/* uses options struct to put standard header on control file */

void rlpr_cf_init(int cfd, const char *filename)
{
  /* rlpr_rlpr_props gets really annoying and bloats the code here
   * -- let's change to something shorter
   */

  rlpr_rlpr_info * rrp = &rlpr_rlpr_props;

  rlpr_cf_add('H', rlpr_client_get(RLPR_CLIENT_LOCALHOST), cfd);
  rlpr_cf_add('P', rrp->user, cfd);    /* responsible user */

  if (rrp->indent_cols)                /* indent if appropriate */
    rlpr_cf_add('I', rrp->indent_cols, cfd);

  if (rrp->mflag)                      /* mail user if appropriate */
    rlpr_cf_add('M', rrp->user, cfd);         

  if (rrp->bflag) {                    /* print burst page */

    /* the defaults on the 'J' and 'C' options are meant to appease
     * berkeley lpd, which is apparently more finicky than the RFC
     * calls for.. also the ordering of these commands mirrors
     * berkeley lpr, in case anyone is relying on the behavior
     */
    
    rlpr_cf_add('J', rrp->job ? rrp->job : filename ? filename : "stdin", cfd);
    rlpr_cf_add('C', rrp->class ? rrp->class
                                : rlpr_client_get(RLPR_CLIENT_LOCALHOST), cfd);
    rlpr_cf_add('L', rrp->user,   cfd);
  }

  /* another berkeleyism that disagrees with the RFC */
  if (rrp->width && strchr("flp", rrp->filetype))
    rlpr_cf_add('W',  rrp->width,  cfd);  /* width for f, l, and p types */

  if (rrp->title)                    /* title for pr(1) */
    rlpr_cf_add('T', rrp->title,  cfd);
}


void rlpr_send_job_req(int sockfd)
{
  char         c       = RECVJ;
  const char * printer = rlpr_client_get(RLPR_CLIENT_PRINTER);

  rlpr_msg(DEBUG, NO_ERRNO, "sending job req to printqueue `%s'", printer);

  safe_writen(sockfd, &c, 1);
  safe_writen(sockfd, printer, strlen(printer));
  safe_writen(sockfd, "\n", 1);
  get_and_verify_ack(sockfd, __FUNCTION__);

  rlpr_msg(DEBUG, NO_ERRNO, "job request sent successfully");
}


void rlpr_cf_send(int sockfd, int cfd, const char *cfname)
{
  char  buf[BUFSIZ];            /* this is okay: user cannot overflow it */
  off_t sz;                     /* cfd size */

  rlpr_msg(DEBUG, NO_ERRNO, "sending control file `%s'", cfname);

  /* send header for control file */
  if ((sz = filesz(cfd)) == (off_t)-1)
    rlpr_msg(FATAL, ERRNO, "fstat on control file");
  
  sprintf(buf, "%c"OFF_T_S" %.*s\n", RECVCF, sz, MAXHOSTNAMELEN, cfname);
  rlpr_msg(DEBUG, NO_ERRNO, "sending control file request `%x"OFF_T_S" %s'",
	   RECVCF, sz, cfname);
  
  safe_writen(sockfd, buf, strlen(buf));
  get_and_verify_ack(sockfd, __FUNCTION__);
  rlpr_msg(DEBUG, NO_ERRNO, "control file request acknolwedged");
   
  if (*(char *)rlpr_msg_get(RLPR_MSG_DEBUG)) { /* is debugging on? */
    char   *eol;
    FILE   *cf = fdopen(dup(cfd), "r");

    if (lseek(cfd, 0, SEEK_SET) < 0)
      rlpr_msg(FATAL, ERRNO, "lseek on control file");

    while (fgets(buf, BUFSIZ, cf) != 0) {
      if ((eol = strrchr(buf, '\n'))) *eol = '\0';
      rlpr_msg(DEBUG, NO_ERRNO, "  %s", buf);
    }
    fclose(cf);
  }

  /* send control file */

  if (lseek(cfd, 0, SEEK_SET) < 0)
    rlpr_msg(FATAL, ERRNO, "lseek on control file");

  if (read_fd_to_fd(cfd, sockfd) < 0)
    rlpr_msg(FATAL, ERRNO, "sending control file to lpd");

  safe_writen(sockfd, "\0", 1);       /* just being explicit */
  get_and_verify_ack(sockfd, __FUNCTION__);
  rlpr_msg(DEBUG, NO_ERRNO, "control file sent successfully");
}

/* if filename == NULL, then we're reading from STDIN */

void rlpr_df_send(int sockfd, char *filename, const char *dfname)
{
  char  buf[BUFSIZ];                  /* temporary buffer */
  int   dfd;                          /* descriptor for datafile */
  off_t sz;                           /* dfd size */
  
  if (filename == NULL) /* STDIN */ {

    /* we need to make a dummy file to send, so that we know the size
     * of stdin for machines that are not local. this is annoying but
     * apparently required for most lpd's (even though RFC 1179 seems
     * to say otherwise)
     */

    filename = rlpr_malloc(strlen(dfname) + strlen(rlpr_rlpr_props.tmpdir) + 2);
    sprintf(filename, "%s/%s", rlpr_rlpr_props.tmpdir, dfname);

    if ((dfd = open(filename, O_RDWR|O_TRUNC|O_CREAT, 0600)) < 0)
      rlpr_msg(FATAL, ERRNO, "%s - cannot create", filename);
    if (unlink(filename) < 0)
      rlpr_msg(WARNING, ERRNO, "%s - cannot unlink", filename);

    if (read_fd_to_fd(STDIN_FILENO, dfd) < 0)
      rlpr_msg(FATAL, ERRNO, "copying `stdin' to temporary file");

    if (lseek(dfd, 0, SEEK_SET) < 0)
      rlpr_msg(WARNING, ERRNO, "%s - cannot rewind", filename);

    free(filename);
    filename = "stdin";
    
    /* else have a filename from the commandline */

  } else {
    if ((dfd = open(filename, O_RDONLY)) < 0)
      rlpr_msg(FATAL, ERRNO, "%s - cannot open", filename);
  }

  /* COMMON */

  rlpr_msg(DEBUG, NO_ERRNO, "sending data file `%s'", filename);

  if ((sz = filesz(dfd)) == (off_t)-1)
    rlpr_msg(FATAL, ERRNO, "fstat on data file");

  sprintf(buf, "%c"OFF_T_S" %.*s\n", RECVDF, sz, MAXHOSTNAMELEN, dfname);
  safe_writen(sockfd, buf, strlen(buf));
  get_and_verify_ack(sockfd, __FUNCTION__);
  
  if (read_fd_to_fd(dfd, sockfd) < 0)
    rlpr_msg(FATAL, ERRNO, "sending data from %s to server", filename);

  safe_writen(sockfd, "\0", 1);
  get_and_verify_ack(sockfd, __FUNCTION__);

  rlpr_msg(DEBUG, NO_ERRNO, "data file `%s' sent successfully", filename);
}

/* global structures too bulky to stick up top */

struct option rlpr_long_opts[] = {
    { "copies"  ,  1, NULL, '#'  },
    { "help"    ,  0, NULL, -2   },
    { "indent"  ,  1, NULL, 'i'  },
    { "job"     ,  1, NULL, 'J'  },
    { "mail"    ,  0, NULL, 'm'  },
    { "no-burst",  0, NULL, 'h'  },
    { "remove"  ,  0, NULL, 'r'  },
    { "tmpdir"  ,  1, NULL, -3   },
    { "title"   ,  1, NULL, 'T'  },
    { "user"    ,  1, NULL, 'U'  },
    { "width"   ,  1, NULL, 'w'  },
    { "windows" ,  1, NULL, 'W'  },
    { "version" ,  0, NULL, 'V'  },
    { "debug"   ,  0, NULL, -101 },
    { "quiet"   ,  0, NULL, 'q'  },
    { "silent"  ,  0, NULL, 'q'  },
    { "verbose" ,  0, NULL, -102 },
    { "no-bind" ,  0, NULL, 'N'  },
    { "port"    ,  1, NULL, -201 },
    { "printer" ,  1, NULL, 'P'  },
    { "queue"   ,  1, NULL, 'P'  },
    { "printhost", 1, NULL, 'H'  },
    { "proxy"   ,  1, NULL, 'X'  },
    { "proxyhost", 1, NULL, 'X'  },
    { 0, 0, 0, 0 }
  };
  
const char * rlpr_opts = "1234cdfglnoptv#:hi::mrsw::C:FJ:T:U:VWqP:Q:NH:X:";
