/*  cdrdao - write audio CD-Rs in disc-at-once mode
 *
 *  Copyright (C) 1998  Andreas Mueller <mueller@daneb.ping.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Log: CdrDriver.h,v $
 * Revision 1.11  1998/10/24 14:28:59  mueller
 * Changed prototype of 'readDiskToc()'. It now accepts the name of the
 * audio file that should be placed into the generated toc-file.
 * Added virtual function 'readDisk()' that for reading disk toc and
 * ripping audio data at the same time.
 *
 * Revision 1.10  1998/10/03 15:10:38  mueller
 * Added function 'writeZeros()'.
 * Updated function 'writeData()'.
 *
 * Revision 1.9  1998/09/27 19:19:18  mueller
 * Added retrieval of control nibbles for track with 'analyzeTrack()'.
 * Added multi session mode.
 *
 * Revision 1.8  1998/09/22 19:15:13  mueller
 * Removed memory allocations during write process.
 *
 * Revision 1.7  1998/09/08 11:54:22  mueller
 * Extended disk info structure because CDD2000 does not support the
 * 'READ DISK INFO' command.
 *
 * Revision 1.6  1998/09/07 15:20:20  mueller
 * Reorganized read-toc related code.
 *
 * Revision 1.5  1998/08/30 19:12:19  mueller
 * Added supressing of error messages for some commands.
 * Added structure 'DriveInfo'.
 *
 * Revision 1.4  1998/08/25 19:24:07  mueller
 * Moved basic index extraction algorithm for read-toc to this class.
 * Added vendor codes.
 *
 */

#ifndef __CDRDRIVER_H__
#define __CDRDRIVER_H__

#include "ScsiIf.h"
#include "Msf.h"

class Toc;
class Track;

struct DiskInfo {
  long capacity; // recordable capacity of medium
  Msf  manufacturerId;   // disk identification
  int  recSpeedLow;      // lowest recording speed
  int  recSpeedHigh;     // highest recording speed

  unsigned int empty : 1; // 1 for empty disk, else 0
  unsigned int cdrw  : 1; // 1 if disk is a CD-RW
  
  struct {
    unsigned int empty : 1;
    unsigned int cdrw : 1;
    unsigned int capacity : 1;
    unsigned int manufacturerId : 1;
    unsigned int recSpeed : 1;
  } valid;
};

struct DriveInfo {
  int maxReadSpeed;
  int currentReadSpeed;
  int maxWriteSpeed;
  int currentWriteSpeed;

  unsigned int accurateAudioStream : 1;
};

struct CdToc {
  int track;   // number
  long start;  // LBA of track start
  unsigned char adrCtl; // ADR/CTL field
};

class CdrDriver {
public:
  CdrDriver(ScsiIf *scsiIf, Toc *toc);
  virtual ~CdrDriver();

  // static constructor
  static CdrDriver *instance(ScsiIf *scsiIf, Toc *toc);

  // returns name of driver
  virtual const char *driverName() const { return driverName_; }

  // returns 1 if drive takes audio samples in big endian byte order or
  // 0 for littel endian byte order
  virtual int bigEndianSamples() const = 0;

  // returns current writing speed
  virtual int speed() { return speed_; }
  
  // sets writing speed, returns 0 for OK or 1 for illegal speed
  virtual int speed(int) = 0;

  // returns 1 if simulation mode, 0 for real writing
  virtual int simulate() const { return simulate_; }

  // sets simulation mode, returns 0 for OK, 1 if given mode is not supported
  virtual int simulate(int s) { simulate_ = s != 0 ? 1 : 0; return 0; }

  // Sets multi session mode (0: close session, 1: open next session).
  // Returns 1 if multi session is not supported by driver, else 0
  virtual int multiSession(int);

  // Returns mutli session mode.
  virtual int multiSession() const { return multiSession_; }

  // general commands
  virtual int testUnitReady(int) const;

  virtual int startStopUnit(int) const;

  virtual int preventMediumRemoval(int) const;

  virtual int rezeroUnit(int showMessage = 1) const;

  virtual int loadUnload(int) const = 0;

  virtual int flushCache() const;

  virtual int readCapacity(long *length, int showMessage = 1);

  // disk at once related commands

  // Used to make necessary initializations but without touching the CD-R.
  // It should be possible to abort the writing process after this function
  // has been called without destroying the CD-R.
  virtual int initDao() = 0;

  // Performs all steps that must be done before the first user data block
  // is written, e.g. sending cue sheet, writing lead-in.
  virtual int startDao() = 0;

  // Performs all steps for successfully finishing the writing process,
  // e.g. writing lead-out, flushing the cache.
  virtual int finishDao() = 0;

  // Sends given data to drive. First block should have lba 0.
  virtual int writeData(long &lba, const char *buf, long len);


  // disk read commands
  
  // analyzes the CD structure (Q sub-channels) of the inserted CD
  virtual Toc *readDiskToc(const char *);

  // analyzes the CD structure and rips audio data
  virtual Toc *readDisk(const char *);

  // returns information about inserted medium
  virtual DiskInfo *diskInfo() { return 0; }


  // static functions

  // returns vendor/type of CD-R medium
  static int cdrVendor(Msf &, const char **vendor, const char** mediumType);

protected:
  struct StringTable {
    int code;
    char *message;
  };
  
  struct CDRVendorTable {
    char m1, s1, f1; // 1st vendor code
    char m2, s2, f2; // 2nd vendor code
    const char *id;  // vendor ID
  };

  ScsiIf *scsiIf_;
  char *driverName_;

  int blockLength_; // length of data block for 'writeData' command
  long blocksPerWrite_; // number of blocks that can be written with a
                        // single SCSI WRITE command
  char *zeroBuffer_; // zeroed buffer for writing zeros
  
  int speed_;
  int simulate_;
  int multiSession_;

  Toc *toc_;

  static StringTable senseKeys[];
  static CDRVendorTable vendorTable[];
  
  static int speed2Mult(int);
  static int mult2Speed(int);

  virtual int sendCmd(const unsigned char *cmd, int cmdLen,
		      const unsigned char *dataOut, int dataOutLen,
		      unsigned char *dataIn, int dataInLen,
		      int showErrorMsg = 1) const;

  virtual const char *getFromStringTable(const StringTable *, int code) const;
  virtual int decodeSense(const unsigned char *buf, int len,
			  int ignoreUnitAttention,
			  int showErrorMsg = 1) const;

  virtual int getModePage(int pageCode, unsigned char *buf, long bufLen,
			  int showErrorMsg = 1);
  virtual int setModePage(unsigned char *buf, int showErrorMsg = 1);

  virtual int writeZeros(long &lba, long count);

  // readToc related functions:

  // returns TOC data of inserted CD, this function is implemented in
  // 'CdrDriver.cc' because it seems to be standard
  virtual CdToc *getToc(int *nofTracks);

  // Called by 'readDiskToc()' to retrieve following information about
  // the track 'trackNr' with given start/end lba addresses:
  // - all index increments, filled into 'index'/'indexCnt'
  // - ISRC Code, filled into provided buffer 'isrcCode' (13 bytes)
  // - length of pre-gap of next track, filled into 'pregap'
  // - control nibbles read from track, filled into bits 0-3 of 'ctrl',
  //   bit 7 must be set to indicate valid data
  // return: 0: OK, 1: error occured
  virtual int analyzeTrack(int trackNr, long startLba, long endLba, Msf *index,
			   int *indexCnt, long *pregap, char *isrcCode,
			   unsigned char *ctl);

  // finds position (lba) where index for given track number switches to
  // 'index' (binary search, base algorithm is implemented in 'CdrDriver').
  // It uses the method 'getTrackIndex()' which must be overloaded by
  // the actual driver.
  virtual long findIndex(int track, int index, long trackStart, long trackEnd);

  // Retrieves track, index and control nibbles at given lba address
  virtual int getTrackIndex(long lba, int *trackNr, int *indexNr, 
			    unsigned char *ctl);

  // Reads catalog number and stores it in given 'Toc' object
  virtual int readCatalog(Toc *toc) = 0;

  friend class CDD2600Base;
};

#endif
