;; $Id: dbtable.dsl 1.7 1998/08/28 16:57:46 nwalsh Exp $
;;
;; This file is part of the Modular DocBook Stylesheet distribution.
;; See ../README or http://nwalsh.com/docbook/dsssl/
;;
;; This file contains table functions common to both print and HTML
;; versions of the DocBook stylesheets.
;;

;; If **ANY** change is made to this file, you _MUST_ alter the
;; following definition:

(define %docbook-common-table-version%
  "Modular DocBook Stylesheet Common Table Functions version 1.02")

;; == Table Support =====================================================

;; ----------------------------------------------------------------------
;; Functions for finding/retrieving table attributes

(define (tgroup-align tgroup)
  (attribute-string "ALIGN" tgroup))

(define (tgroup-colsep tgroup)
  (attribute-string "COLSEP" tgroup))

(define (tgroup-rowsep tgroup)
  (attribute-string "ROWSEP" tgroup))

;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (find-colspec colname)
  (let* ((tgroup (ancestor "TGROUP"))
	 (colspecs (select-elements (descendants tgroup)
				    (normalize "colspec"))))
    (let loop ((nl colspecs))
	(if (node-list-empty? nl)
	    ;; we've run out of places to look, stop looking...
	    (error (string-append "Could not find COLSPEC named " colname))
	    (if (equal? (normalize colname)
			(attribute-string (normalize "COLNAME") 
					  (node-list-first nl)))
		(node-list-first nl)
		(loop (node-list-rest nl)))))))

(define (find-colspec-by-number colnum)
  (let* ((tgroup (ancestor "TGROUP"))
	 (colspecs (select-elements (children tgroup) "COLSPEC")))
    (let loop ((nl colspecs))
      (if (node-list-empty? nl)
	  ;; we've run out of places to look, stop looking...
	  (empty-node-list)
	  (if (equal? (colspec-colnum (node-list-first nl)) colnum)
	      (node-list-first nl)
	      (loop (node-list-rest nl)))))))

(define (colspec-align colspec)
  (attribute-string "ALIGN" colspec))

(define (colspec-char colspec)
  (attribute-string "CHAR" colspec))

(define (colspec-charoff colspec)
  (let ((charoff (attribute-string "CHAROFF" colspec)))
    (if charoff
	(string->number charoff)
	#f)))

(define (colspec-colnum colspec)
  ;; returns the column number of the associated colspec...which is 
  ;; either the value of COLNUM or obtained by counting
  (let* ((tgroup (ancestor "TGROUP" colspec))
	 (colspecs (select-elements (children tgroup) "COLSPEC")))
    (if (attribute-string "COLNUM" colspec)
	(string->number (attribute-string "COLNUM" colspec))
	(let loop ((nl colspecs) (curcol 1))
	  (let ((colnum (attribute-string "COLNUM" (node-list-first nl))))
	    (if (node-list=? (node-list-first nl) colspec)
		curcol
		(if colnum
		    (loop (node-list-rest nl) (+ (string->number colnum) 1))
		    (loop (node-list-rest nl) (+ curcol 1)))))))))

(define (colspec-colname colspec)
  (attribute-string "COLNAME" colspec))

(define (colspec-colsep colspec)
  (attribute-string "COLSEP" colspec))

(define (colspec-colwidth colspec)
  (attribute-string "COLWIDTH" colspec))

(define (colspec-rowsep colspec)
  (attribute-string "ROWSEP" colspec))

;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (find-spanspec spanname)
  (let* ((tgroup (ancestor "TGROUP"))
	 (spanspecs (select-elements (descendants tgroup) 
				     (normalize "SPANSPEC"))))
    (let loop ((nl spanspecs))
      (if (node-list-empty? nl)
	  (error (string-append "Could not find SPANSPEC named " spanname))
	  (if (equal? spanname 
		      (attribute-string (normalize "SPANNAME")
					(node-list-first nl)))
	      (node-list-first nl)
	      (loop (node-list-rest nl)))))))

(define (spanspec-align spanspec)
  (attribute-string "ALIGN" spanspec))

(define (spanspec-char spanspec)
  (attribute-string "CHAR" spanspec))

(define (spanspec-charoff spanspec)
  (let ((charoff (attribute-string "CHAROFF" spanspec)))
    (if charoff
	(string->number charoff)
	#f)))

(define (spanspec-colsep spanspec)
  (attribute-string "COLSEP" spanspec))

(define (spanspec-nameend spanspec)
  (attribute-string "NAMEEND" spanspec))

(define (spanspec-namest spanspec)
  (attribute-string "NAMEST" spanspec))

(define (spanspec-rowsep spanspec)
  (attribute-string "ROWSEP" spanspec))

(define (spanspec-spanname spanspec)
  (attribute-string "SPANNAME" spanspec))

;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;; Calculate spans

(define (hspan entry)
  ;; Returns the horizontal span of an entry
  (let* ((spanname (attribute-string "SPANNAME" entry))
	 (namest   (if spanname
		       (spanspec-namest (find-spanspec spanname))
		       (attribute-string "NAMEST" entry)))
	 (nameend  (if spanname
		       (spanspec-nameend (find-spanspec spanname))
		       (attribute-string "NAMEEND" entry)))
	 (colst    (if namest
		       (colspec-colnum (find-colspec namest))
		       #f))
	 (colend   (if nameend
		       (colspec-colnum (find-colspec nameend))
		       #f)))
    (if (and namest nameend)
	(+ (- colend colst) 1)
	1)))

(define (vspan entry)
  ;; Returns the vertical span of an entry.  Note that this is one more
  ;; than the specified MOREROWS attribute.
  (let* ((morerows (attribute-string "MOREROWS" entry)))
    (if morerows
	(+ (string->number morerows) 1)
	1)))

;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;; Update the "overhang" list

(define (adjust-overhang overhang oldoverhang entry)
  (let* ((colst    (cell-column-number entry oldoverhang))
	 (span     (hspan entry)))
    (if (> (vspan entry) 1)
	(list-put overhang colst (- (vspan entry) 1) span)
	overhang)))

(define (overhang-skip overhang startcol)
  (if (> startcol (length overhang))
      ;; this is a _broken_ table.  should I output a debug message!?
      startcol
      (let loop ((overtail (list-tail overhang (- startcol 1))) (col startcol))
	(if (null? overtail)
	    col
	    (if (equal? (car overtail) 0)
		col
		(loop (cdr overtail) (+ col 1)))))))

(define (update-overhang row oldoverhang)
  (let loop ((overhang (decrement-list-members oldoverhang))
	     (entries  (node-list-filter-out-pis (children row))))
    (if (node-list-empty? entries)
	overhang
	(loop (adjust-overhang overhang oldoverhang 
			       (node-list-first entries))
	      (node-list-rest entries)))))

;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;; Calculate information about cells

(define (cell-prev-cell entry)
  ;; Return the cell which precedes entry in the current row.
  (let loop ((nd (ipreced entry)))
    (if (node-list-empty? nd)
	nd
	(if (equal? (node-property 'class-name nd) 'element)
	    nd
	    (loop (ipreced nd))))))

(define (cell-column-number entry overhang)
  (let* ((entry     (ancestor-member entry '("ENTRY" "ENTRYTBL")))
	 (row       (ancestor "ROW" entry))
	 (preventry (cell-prev-cell entry))
	 (prevspan  (if (node-list-empty? preventry) 1 (hspan preventry)))
	 (colname   (attribute-string "COLNAME" entry))
	 (namest    (attribute-string "NAMEST" entry))
	 (nameend   (attribute-string "NAMEEND" entry))
	 (spanname  (attribute-string "SPANNAME" entry)))
    (if colname
	(colspec-colnum (find-colspec colname))
	(if spanname
	    (colspec-colnum (find-colspec 
			     (spanspec-namest (find-spanspec spanname))))
	    (if namest
		(colspec-colnum (find-colspec namest))
		(if (node-list-empty? preventry)
		    (overhang-skip overhang 1)
		    (overhang-skip overhang 
				   (+ (cell-column-number preventry overhang) 
				   prevspan))))))))

;; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

(define (cell-colsep cell colnum)
  (let* ((entry     (ancestor-member cell '("ENTRY" "ENTRYTBL")))
	 (spanname  (attribute-string "SPANNAME" entry))
	 (tgroup    (ancestor "TGROUP" entry))
	 (table     (parent tgroup))
	 (calscolsep 
	  (if (attribute-string "COLSEP" entry)
	      (attribute-string "COLSEP" entry)
	      (if (and spanname 
		       (spanspec-colsep (find-spanspec spanname)))
		  (spanspec-colsep (find-spanspec spanname))
		  (if (colspec-colsep (find-colspec-by-number colnum))
		      (colspec-colsep (find-colspec-by-number colnum))
		      (if (tgroup-colsep tgroup)
			  (tgroup-colsep tgroup)
			  (if (attribute-string "COLSEP" table)
			      (attribute-string "COLSEP" table)
			      (if ($cals-rule-default$ cell)
				  "1"
				  "0"))))))))
    (> (string->number calscolsep) 0)))

(define (cell-rowsep cell colnum)
  (let* ((entry     (ancestor-member cell '("ENTRY" "ENTRYTBL")))
	 (spanname  (attribute-string "SPANNAME" entry))
	 (row       (ancestor "ROW" entry))
	 (tgroup    (ancestor "TGROUP" entry))
	 (table     (parent tgroup))
	 (calsrowsep 
	  (if (attribute-string "ROWSEP" entry)
	      (attribute-string "ROWSEP" entry)
	      (if (and spanname (spanspec-rowsep (find-spanspec spanname)))
		  (spanspec-rowsep (find-spanspec spanname))
		  (if (colspec-rowsep (find-colspec-by-number colnum))
		      (colspec-rowsep (find-colspec-by-number colnum))
		      (if (attribute-string "ROWSEP" row)
			  (attribute-string "ROWSEP" row)
			  (if (tgroup-rowsep tgroup)
			      (tgroup-rowsep tgroup)
			      (if (attribute-string "ROWSEP" table)
				  (attribute-string "ROWSEP" table)
				  (if ($cals-rule-default$ cell)
				      "1"
				      "0")))))))))
    (> (string->number calsrowsep) 0)))

;; ======================================================================
