##
##  Text::BlockParser - parse a delimited block of text
##  Copyright (c) 1998,1999 Ralf S. Engelschall, All Rights Reserved. 
##

package Text::BlockParser;

require 5.003;

use strict;
use vars qw(@ISA @EXPORT @EXPORT_OK $VERSION);
use Exporter;
use Text::DelimMatch;

@ISA       = qw(Exporter);
@EXPORT    = qw();
@EXPORT_OK = qw(parseblock);

$VERSION   = '1.0';

sub parseblock {
    my ($text,
        $op_func, $op_inout,
        $del_begin, $del_end, $del_case, $del_strip, 
        $del_quote, $del_esc, $del_esc2) = @_;

    #   create a delimiter matching object
    my $dm = new Text::DelimMatch;
    $dm->delim($del_begin, $del_end);
    $dm->quote($del_quote);
    $dm->escape($del_esc);
    $dm->double_escape($del_esc2);
    $dm->case_sensitive($del_case);

    my $bufferin = $text;
    my $bufferout = '';
    while (1) {
        #   match against the delimiters
        my ($prolog, $match, $epilog) = $dm->match($bufferin);
        if ($match ne '') {
            if ($del_strip) {
                #   strip delimiters
                $match =~ s|^$del_begin(.*?)$del_end|$1|s if ($del_case);
                $match =~ s|^$del_begin(.*?)$del_end|$1|is if (not $del_case);
            }
            else {
                #   keep delimiters
                $match  =~ s|^($del_begin)(.*?)($del_end)$|
                             $prolog = $prolog.$1, $epilog = $3.$epilog, $1|se if ($del_case);
                $match  =~ s|^($del_begin)(.*?)($del_end)$|
                             $prolog = $prolog.$1, $epilog = $3.$epilog, $1|ise if (not $del_case);
            }

            #   process nesting out->in
            $match = &{$op_func}($match) if (not $op_inout);

            #   recursively process nested delimiters
            $match = &parseblock(
                $match,
                $op_func, $op_inout,
                $del_begin, $del_end, $del_case, $del_strip,
                $del_quote, $del_esc, $del_esc2
            );

            #   process nesting in->out
            $match = &{$op_func}($match) if ($op_inout);

            #   reconstruct the buffer
            $bufferout .= $prolog.$match;
            $bufferin = $epilog;
        }
        else {
            $bufferout .= $bufferin;
            last;
        }
    }

    return $bufferout;
}

1;
__END__

=head1 NAME

Text::BlockParser - parse a delimited block of text

=head1 SYNOPSIS

 use Test::BlockParser qw(parseblock);

 $text = &parseblock(
     $text,
     \&operate_func, $operate_inout,
     $del_begin, $del_end, $del_case, $del_strip,
     $del_quote, $del_esc, $del_esc2
 );
 
 sub parse_func {
     my ($block) = @_;
     [...]
     return $block;
 }

=head1 DESCRIPTION

This is just a recursive wrapper for the Text::DelimMatch(3) module from
Norman Walsh. The idea is to recusively parse a nested structure by the use of
a Text::DelimMatch(3) object and apply an operation function at each nesting
level.

=head1 AUTHOR

Ralf S. Engelschall

=head1 SEE ALSO

Text::DelimMatch(3).

=cut

