/*
 *	
 * Esetroot -- Set the root pixmap.  This program enables non-Enlightenment
 *             users to use Eterm's support for pseudotransparency.
 *
 * Written by Nat Friedman <ndf@mit.edu> with modifications by Gerald Britton
 * <gbritton@mit.edu> and Michael Jennings <mej@tcserv.com>
 *
 */

static const char *cvs_ident = "$Id: Esetroot.c,v 1.9 1998/10/14 19:56:49 mej Exp $";

#include "../config.h"

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/Xos.h>
#include <Imlib.h>

Display *Xdisplay;
Screen *scr;
Window Xroot;
int screen;
unsigned char debug = 0;

#define Xdepth	 (DefaultDepth(Xdisplay, screen))

void
set_pixmap_property(Pixmap p) {

  Atom prop;

  /* This will locate the property, creating it if it doesn't exist */
  prop = XInternAtom(Xdisplay, "_XROOTPMAP_ID", False);

  /* The call above should have created it.  If that failed, we can't continue. */
  if (prop == None) {
    fprintf(stderr, "Esetroot:  creation of pixmap property failed.\n");
    exit(1);
  }

  XChangeProperty(Xdisplay, Xroot, prop, XA_PIXMAP, 32, PropModeReplace,
		  (unsigned char *) &p, 1);
  XSetCloseDownMode(Xdisplay, RetainPermanent);
  XFlush(Xdisplay);
}

int
main(int argc, char *argv[]) {

  unsigned char scale = 0;
  char *displayname = NULL;
  char *fname = NULL;
  ImlibData *id;
  ImlibImage *im;
  Pixmap p, temp_pmap;
  register unsigned char i;
  GC gc;
  XGCValues gcv;
  int w, h;

  if (argc < 2) {
    fprintf(stderr, "%s [-display <display_name>] [-scale] pixmap\n", *argv);
    fprintf(stderr, "\t Short options are also recognized (-d and -s)\n");
    exit(0);
  }

  for (i=1; i < argc; i++) {
    if (*argv[i] != '-') {
      break;
    }
    if (argv[i][1] == 'd') {
      displayname = argv[++i];
    } else if (argv[i][1] == 's') {
      scale = 1;
    } else if (argv[i][1] == 'x') {
      fprintf(stderr, "Debugging activated.\n");
      debug = 1;
    } else {
      fprintf(stderr, "%s:  Unrecognized option \'%c\'\n\n", *argv, argv[i][1]);
      fprintf(stderr, "%s [-display display] [-scale] pixmap\n", *argv);
      fprintf(stderr, "\t Short options are also recognized (-d and -s)\n");
      exit(2);
    }
  }

  fname = argv[i];

  if (debug) {
    fprintf(stderr, "%s:%d:  Display name is \"%s\"\n", __FILE__, __LINE__, displayname ? displayname : "(nil)");
    fprintf(stderr, "%s:%d:  Image will be %s\n", __FILE__, __LINE__, scale ? "scaled" : "tiled");
    fprintf(stderr, "%s:%d:  Image file is %s\n", __FILE__, __LINE__, fname ? fname : "(nil)");
  }

  if (!displayname) {
    displayname = getenv("DISPLAY");
    if (debug) {
      fprintf(stderr, "%s:%d:  Display name set to %s via getenv(\"DISPLAY\")\n", __FILE__, __LINE__, displayname ? displayname : "(nil)");
    }
  }

  if (! displayname) {
    displayname = ":0.0";
    if (debug) {
      fprintf(stderr, "%s:%d:  Display name defaulted to %s\n", __FILE__, __LINE__, displayname ? displayname : "(nil)");
    }
  }

  if ((Xdisplay = XOpenDisplay(displayname)) == 0) {
    fprintf(stderr, "%s:  Unable to open display %s\n", *argv, displayname);
    exit(1);
  }

  screen = DefaultScreen(Xdisplay);
  Xroot = RootWindow(Xdisplay, screen);
  scr = ScreenOfDisplay(Xdisplay, screen);
  if (debug) {
    fprintf(stderr, "%s:%d:  Chose screen %d\n", __FILE__, __LINE__, screen);
    fprintf(stderr, "%s:%d:  Root window is 0x%08x\n", __FILE__, __LINE__, Xroot);
    fprintf(stderr, "%s:%d:  Found screen information at 0x%08x\n", __FILE__, __LINE__, scr);
  }

  id = Imlib_init(Xdisplay);
  im = Imlib_load_image(id, fname);
  if (debug) {
    fprintf(stderr, "%s:%d:  The Imlib Data is at 0x%08x\n", __FILE__, __LINE__, id);
    fprintf(stderr, "%s:%d:  The Imlib Image is at 0x%08x\n", __FILE__, __LINE__, im);
  }

  if (scale) {
    w = scr->width;
    h = scr->height;
    p = XCreatePixmap(Xdisplay, Xroot, scr->width, scr->height, Xdepth);
    gc = XCreateGC(Xdisplay, p, 0, &gcv);
    XFillRectangle(Xdisplay, p, gc, 0, 0, w, h);
  } else {
    w = im->rgb_width;
    h = im->rgb_height;
    p = XCreatePixmap(Xdisplay, Xroot, scr->width, scr->height, Xdepth);
    gc = XCreateGC(Xdisplay, p, 0, &gcv);
  }
  if (debug) {
    fprintf(stderr, "%s:%d:  Assigned width and height for rendering as %dx%d\n", __FILE__, __LINE__, w, h);
    fprintf(stderr, "%s:%d:  Created %dx%d pixmap 0x%08x\n", __FILE__, __LINE__, scr->width, scr->height, p);
    fprintf(stderr, "%s:%d:  Applied Graphics Context 0x%08x to pixmap.\n", __FILE__, __LINE__, gc);
  }

  Imlib_render(id, im, w, h);
  temp_pmap = Imlib_move_image(id, im);
  if (debug) {
    fprintf(stderr, "%s:%d:  Rendered at %dx%d onto pixmap 0x%08x\n", __FILE__, __LINE__, w, h, temp_pmap);
  }

  if (temp_pmap != None) {
    XSetTile(Xdisplay, gc, temp_pmap);
    XSetTSOrigin(Xdisplay, gc, 0, 0);
    XSetFillStyle(Xdisplay, gc, FillTiled);
    XFillRectangle(Xdisplay, p, gc, 0, 0, scr->width, scr->height);
    set_pixmap_property(p);
    XFlush(Xdisplay);
    XSetWindowBackgroundPixmap(Xdisplay, Xroot, p);
    XClearWindow(Xdisplay, Xroot);
    XFlush(Xdisplay);
  }

  return 0;
}

