/**********************************************************************
 *  wmapm-1.2 (C) 1998 Chris D. Faulhaber <jedgar@speck.ml.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * Portions of code derived from:
 *   apm/apmd/libapm : (c) 1996 Rickard E. Faith (r.faith@ieee.org)
 *   wmmon           : (c) 1998 Martijn Pieterse (pieterse@xs4all.nl) and
 *                              Antoine Nulle (warp@xs4all.nl) 
 ***********************************************************************/

/*

Changes:

1.2		- Fixed bug that showed 100% battery capacity
                  as 90% (I never noticed since my battery seems
                  to max out at 98%).
                  Released 980731

1.1             - Removed libapm dependency; tweaked some code.
                  Released 980729

1.0             - Initial release version.
                  Released 980725

*/

/*  Includes  */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include "wmapm.h"

#include <X11/xpm.h>

#include "../wmgeneral/wmgeneral.h"

#include "wmapm-master.xpm"
#include "wmapm-mask.xbm"

/*  Functions  */

int apm_exists();
int apm_read(apm_info *i);
void ParseCMDLine(int argc, char *argv[]);

/*  apm_exists()  */

int apm_exists() {

  apm_info i;

  if (access("/proc/apm", R_OK))
    return 1;
  return apm_read(&i);

}

/*  apm_read()  */

int apm_read(apm_info *i) {

  FILE *str;
  char units[10];
  char buffer[100];
  int  retcode = 0;

  if (!(str = fopen("/proc/apm", "r")))
    return 1;
  fgets(buffer, sizeof(buffer) - 1, str);
  buffer[sizeof(buffer) - 1] = '\0';
  sscanf(buffer, "%s %d.%d %x %x %x %x %d%% %d %s\n",
	 (char *)i->driver_version,
	 &i->apm_version_major,
	 &i->apm_version_minor,
	 &i->apm_flags,
	 &i->ac_line_status,
	 &i->battery_status,
	 &i->battery_flags,
	 &i->battery_percentage,
	 &i->battery_time,
	 units);
  i->using_minutes = !strncmp(units, "min", 3) ? 1 : 0;
  if (i->driver_version[0] == 'B') { /* old style.  argh. */
    strcpy((char *)i->driver_version, "pre-0.7");
    i->apm_version_major  = 0;
    i->apm_version_minor  = 0;
    i->apm_flags          = 0;
    i->ac_line_status     = 0xff;
    i->battery_status     = 0xff;
    i->battery_flags      = 0xff;
    i->battery_percentage = -1;
    i->battery_time       = -1;
    i->using_minutes      = 1;

    sscanf(buffer, "BIOS version: %d.%d",
	   &i->apm_version_major, &i->apm_version_minor);
    fgets(buffer, sizeof(buffer) - 1, str);
    sscanf(buffer, "Flags: 0x%02x", &i->apm_flags);
    if (i->apm_flags & APM_32_BIT_SUPPORT) {
      fgets(buffer, sizeof(buffer) - 1, str);
      fgets(buffer, sizeof(buffer) - 1, str);
      if (buffer[0] != 'P') {
	if (!strncmp(buffer+4, "off line", 8))
	  i->ac_line_status = 0;
	else if (!strncmp(buffer+4, "on line", 7))
	  i->ac_line_status = 1;
	else if (!strncmp(buffer+4, "on back", 7))
	  i->ac_line_status = 2;

	fgets(buffer, sizeof(buffer) - 1, str);
	if (!strncmp(buffer+16, "high", 4))
	  i->battery_status = 0;
	else if (!strncmp(buffer+16, "low", 3))
	  i->battery_status = 1;
	else if (!strncmp(buffer+16, "crit", 4))
	  i->battery_status = 2;
	else if (!strncmp(buffer+16, "charg", 5))
	  i->battery_status = 3;

	fgets(buffer, sizeof(buffer) - 1, str);
	if (strncmp(buffer+14, "unknown", 7))
	  i->battery_percentage = atoi(buffer + 14);
	if (i->apm_version_major >= 1 && i->apm_version_minor >= 1) {
	  fgets(buffer, sizeof(buffer) - 1, str);
	  sscanf(buffer, "Battery flag: 0x%02x", &i->battery_flags);

	  fgets(buffer, sizeof(buffer) - 1, str);
	  if (strncmp(buffer+14, "unknown", 7))
	    i->battery_time = atoi(buffer + 14);
	}
      }
    }
  }

                                /* Fix possible kernel bug -- percentage
                                   set to 0xff (==255) instead of -1. */
  if (i->battery_percentage > 100) i->battery_percentage = -1;

  fclose(str);
  return retcode;
}

/* ParseCMDLine()  */

void ParseCMDLine(int argc, char *argv[])
{
  char *cmdline;
  int  i;

  for (i = 1; i < argc; i++) {
    cmdline = argv[i];
    if (cmdline[0] == '-') {
      switch(cmdline[1]) {
      case 'd': break;
      default:  printf("\nwmapm v%s, by jedgar@speck.ml.org\n", WMAPM_VERSION);
	        printf("http://speck.ml.org/~wmapm/\n\n");
                printf("usage:\n");
                printf("\t-d <display>\tselects target display\n");
                printf("\t-h\tdisplay this screen\n\n");
                exit(0);
      }
    }
  }
}

/*  main  */

int main(int argc, char *argv[])
{
  apm_info cur_info;
  int      time_left, hour_left, min_left, digit;
  unsigned long sleep_time = DELAY;

  ParseCMDLine(argc, argv);

  /*  Check for APM support */
  switch (apm_exists()) {
  case 1: fprintf(stderr, "No APM support in kernel\n");
          exit(1);
  case 2: fprintf(stderr, "Old APM support in kernel\n");
          exit(2);
  }

  openXwindow(argc,argv,wmapm_master_xpm,wmapm_mask_bits,
              wmapm_mask_width,wmapm_mask_height);

  while (1)      /* Do it until we die :) */
    {
      if (apm_read(&cur_info)) {
      fprintf(stderr, "Cannot read APM information\n");
      exit(1);
      }

      /* Check AC status */
      switch (cur_info.ac_line_status) {
      case 1:  copyXPMArea(66, 5, 29, 10, 29, 5);  /* AC on-line */
               break;
      default: copyXPMArea(66, 19, 29, 10, 29, 5); /* AC off-line */
      }

      /* Check Battery Status */
      copyXPMArea(73, 65, 15, 9, 41, 47);          /* Default Battery Status */
      copyXPMArea(65, 45, 44, 9, 5, 34);           /* Default Charge Status */
      switch (cur_info.battery_status) {
      case 0:  copyXPMArea(100, 65, 5, 9, 51, 47); /* Battery Status high */
               break;
      case 1:  copyXPMArea(95, 65, 5, 9, 46, 47);  /* Battery Status low */
               break;
      case 2:  copyXPMArea(90, 65, 5, 9, 41, 47);  /* Battery Status crit */
               break;
      case 3:  copyXPMArea(65, 33, 44, 9, 5, 34);  /* Battery Charging */
               break;
      }

      /* Do Time Left */
      copyXPMArea(84, 92, 34, 11, 5, 46);          /* Default Time */  
      if (cur_info.battery_time >= 0) {
        if (cur_info.using_minutes)
          time_left = cur_info.battery_time;
        else
          time_left = cur_info.battery_time / 60; 
        hour_left = time_left / 60;
        min_left = time_left % 60;
        digit = hour_left / 10;
        copyXPMArea(digit * 7 + 4, 92, 7, 11, 5, 46);  /* Show 10's (hour) */
        digit = hour_left % 10;
        copyXPMArea(digit * 7 + 4, 92, 7, 11, 12, 46); /* Show 1's (hour) */
        digit = min_left / 10;
        copyXPMArea(digit * 7 + 4, 92, 7, 11, 24, 46); /* Show 10's (min) */
        digit = min_left % 10;
        copyXPMArea(digit * 7 + 4, 92, 7, 11, 31, 46); /* Show 1's (min) */
        copyXPMArea(75, 92, 4, 11, 20, 46);
      }

      /* Do Battery Percentage */
      if (cur_info.battery_percentage >= 0) {
        digit = cur_info.battery_percentage / 10;
        digit = digit == 10 ? 9 : digit;
        if (digit == 10) {
          copyXPMArea(58, 80, 6, 9, 5, 19);           /* If 100%, show 99% */
          copyXPMArea(58, 80, 6, 9, 11, 19);
	  copyXPMArea(37, 65, 30, 10, 29, 19);
        }
        else {
          copyXPMArea(digit * 6 + 4, 80, 6, 9, 5, 19);  /* Show 10's */
          digit = cur_info.battery_percentage % 10;
          copyXPMArea(digit * 6 + 4, 80, 6, 9, 11, 19); /* Show 1's */
          copyXPMArea(64, 80, 7, 9, 17, 19);            /* Show '%' */
          copyXPMArea(37, 65, cur_info.battery_percentage * 30 / 100, 10,
                      29, 19);                          /* Show Meter */
        }
      }
      else {
        copyXPMArea(75, 80, 19, 9, 5, 19);            /* Show Default % */
        copyXPMArea(5, 65, 30, 10, 29, 19);           /* Show Default Meter */
      }

      /* Redraw and wait for next update */
      RedrawWindow();
      usleep(sleep_time);
    }
}
