/* --------------------------------------------------------------------*/
/*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
/*                                                                     */
/*                                     ,--^,                           */
/*                               _ ___/ /|/                            */
/*                           ,;'( )__, ) '                             */
/*                          ;;  //   L__.                              */
/*                          '   \   /  '                               */
/*                               ^   ^                                 */
/*                                                                     */
/*                                                                     */
/*    This program is distributed in the hope that it will be useful.  */
/*    Use and copying of this software and preparation of derivative   */
/*    works based upon this software are permitted, so long as the     */
/*    following conditions are met:                                    */
/*           o credit to the authors is acknowledged following         */
/*             current academic behaviour                              */
/*           o no fees or compensation are charged for use, copies,    */
/*             or access to this software                              */
/*           o this copyright notice is included intact.               */
/*      This software is made available AS IS, and no warranty is made */
/*      about the software or its performance.                         */
/*                                                                     */
/*      Bug descriptions, use reports, comments or suggestions are     */
/*      welcome. Send them to                                          */
/*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
/*-------------------------------------------------------------------- */
/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime1.9b/Clib/csymbol.c           */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Wed Feb 12 14:51:41 1992                          */
/*    Last change :  Thu Jun 12 14:20:46 1997 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Symbol handling (creation and hash tabling).                     */
/*=====================================================================*/
#include <string.h>
#include <bigloo1.9c.h>

/*---------------------------------------------------------------------*/
/*    Quelques prototypes                                              */
/*---------------------------------------------------------------------*/
extern obj_t string_to_symbol();
static obj_t make_symbol();
extern obj_t make_vector();
extern long  get_hash_number();

/*---------------------------------------------------------------------*/
/*    Quelques variables statics                                       */
/*---------------------------------------------------------------------*/
static obj_t c_symtab;

/*---------------------------------------------------------------------*/
/*    init_symbol_table ...                                            */
/*---------------------------------------------------------------------*/
init_symbol_table()
{
   c_symtab = make_vector( SYMBOL_HASH_TABLE_SIZE, BNIL );
}
          
/*---------------------------------------------------------------------*/
/*    make_symbol ...                                                  */
/*    -------------------------------------------------------------    */
/*    Symbols are not allocated in the first generation because        */
/*    they can't be collected.                                         */
/*---------------------------------------------------------------------*/
static obj_t
make_symbol( name )
char *name;
{
   obj_t symbol;

   symbol = GC_MALLOC( SYMBOL_SIZE );
   
   symbol->symbol_t.header = MAKE_HEADER( SYMBOL_TYPE, SYMBOL_SIZE );
   symbol->symbol_t.name   = (char *)GC_MALLOC_ATOMIC( strlen( name ) + 1 );
   symbol->symbol_t.cval   = BNIL;
   
   strcpy( symbol->symbol_t.name, name );

   return BREF( symbol );
}
   
/*---------------------------------------------------------------------*/
/*    string_to_symbol ...                                             */
/*    char * --> obj_t                                                 */
/*---------------------------------------------------------------------*/
obj_t
string_to_symbol( name )
char *name;
{
   long hash_number;
   obj_t bucket;

   hash_number = get_hash_power_number( name, SYMBOL_HASH_TABLE_SIZE_SHIFT );
   bucket = VECTOR_REF( c_symtab, hash_number );
   
   if( NULLP( bucket ) )
   {
      obj_t symbol = make_symbol( name );
      obj_t pair   = MAKE_PAIR( symbol, BNIL );
      
      VECTOR_SET( c_symtab, hash_number, pair );
      
      return symbol;
   }
   else
   {
      obj_t run = bucket, back = bucket;
      
      while( !NULLP( run ) && strcmp( SYMBOL( CAR( run ) ).name, name ) )
         back = run, run = CDR( run );
      
      if( !NULLP( run ) )
         return CAR( run );
      else
      {
         obj_t symbol = make_symbol( name );
	 obj_t pair   = MAKE_PAIR( symbol, BNIL );
	 
         SET_CDR( back, pair );

         return symbol;
      }
   }
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    symbol_exists_p ...                                              */
/*    -------------------------------------------------------------    */
/*    Cette fonction retourne vrai s'il existe un symbol de nom        */
/*    `string' et faux sinon                                           */
/*---------------------------------------------------------------------*/
int
symbol_exists_p( name )
char *name;
{
   long hash_number;
   obj_t bucket;

   hash_number = get_hash_power_number( name, SYMBOL_HASH_TABLE_SIZE_SHIFT );
   bucket = VECTOR_REF( c_symtab, hash_number );
   
   if( NULLP( bucket ) )
   {
      return 0;
   }
   else
   {
      obj_t run = bucket;
      
      while( !NULLP( run ) && strcmp( SYMBOL( CAR( run ) ).name, name ) )
         run = CDR( run );

      return !NULLP( run );
   }
}

