;! @file     read.scm                                                     !;
;! @created  Tue Jun 17 14:47:54 1997                                     !;
;! @modified Thu Mar  5 15:25:43 1998                                     !;
;! @version 1.0                                                           !;
;! @copyright Dominique Boucher                                           !;
;! @description                                                           !;
;! Reading functions                                                      !;

(module read
	(export 
	 (scm-read port)
	 (read-text port lst)
	 (pack-doc-strings docs)))

;! @description 
;!   Reads a top-level expression (one that starts a the beginning of a line)
;!   or a documentation comment.
;! @param port The port from which to read
;! @return A two-element list, the first being the type of the data
;!   read ('expression or a doc-comment type), the second being
;!   the expression read or the documentation string.
(define (scm-read port)
  (let return ()
    (let ((c (peek-char port)))
      (cond
       ((eof-object? c)
	c)
       
       ((char=? c #\()
	(list 'expression (read port)))
       
       ((char=? c #\;)
	(read-char port)
	(let ((c (peek-char port)))
	  (if (char=? c #\!)
	      (begin
		(read-char port)
		(read-doc-comment port))
	      (begin
		(read-to-next-line port)
		(return)))))
       
       (else
	(read-to-next-line port)
	(return))))))

(define (read-text port lst)
  (let loop ((l lst) (e (scm-read port)))
    (if (eof-object? e)
        (reverse l)
        (cond
          ((eq? (car e) 'end)
           (reverse l))
          ((eq? (car e) 'doc)
           (loop (cons (cadr e) l) (scm-read port)))
          (else
	   (loop l (scm-read port)))))))
  
;! @description 
;!   Reads all the characters up to the end of the line and put
;!   them in a string.
;! @param port The port from which the function reads 
;! @return A string containing all the characters read, excluding
;!   the end-of-line character
(define (read-to-next-line port)
  (let loop ((l '()))
    (let ((c (read-char port)))
      (if (or (eof-object? c)
	      (char=? c #\newline))
	  (strip-ending-comment (list->string (reverse l)))
	  (loop (cons c l))))))

(define (strip-ending-comment str)
  (let ((len (string-length str)))
    (if (and (>= len 2)
	     (char=? (string-ref str (- len 2)) #\!)
	     (char=? (string-ref str (- len 1)) #\;))
	(substring str 0 (- len 2))
	str)))


;! @description 
;!   Reads a documentation comment, one that starts at the beginning
;!   of a line
;! @param port The port from which the function reads the comment
;! @return A two-element list, the first being the type of the comment
;!   and the other being the documentation string
(define (read-doc-comment port)

  (define (strip-spaces port)
    (let loop ()
      (let ((c (peek-char port)))
	(if (char=? c #\space)
	    (begin (read-char port) (loop))
	    #f))))
  
  (define (read-symbol port)
    (let loop ((l '()))
      (let ((c (peek-char port)))
	(if (or (eof-object? c) (memq c '(#\newline #\tab #\space)))
	    (string->symbol
	     (list->string
	      (map char-upcase (reverse l))))
	    (loop (cons (read-char port) l))))))
  
  (define (read-string port)
    (let loop ((l '()))
      (let ((c (peek-char port)))
	(if (or (eof-object? c) (memq c '(#\newline #\tab #\space)))
	    (list->string (reverse l))
	    (loop (cons (read-char port) l))))))
  
  (let ((c1 (read-char port)))
    (if (char=? c1 #\space)
        (let ((c2 (peek-char port)))
          (if (char=? c2 #\@)
              (begin
                (read-char port)
                (let* ((ident (read-symbol port)))
                  (case ident
                    ((param field)
                     (strip-spaces port)
                     (let* ((name (read-symbol port))
                            (data (read-to-next-line port)))
                       (list ident name data)))
		    ((end)
		     (read-to-next-line port)
		     (list ident))
                    ((include)
                     (strip-spaces port)
                     (let ((name (read-string port)))
                       (read-to-next-line port)
                       (list ident name)))
                    (else
                      (let ((data  (read-to-next-line port)))
                        (list ident data))))))
              
              (list 'doc (read-to-next-line port))))
        (list 'doc (read-to-next-line port)))))


;! @description
;!   Since documentation strings can span many lines, we must pack
;!   all the related doc strings.
;! @param lst A list of documentation string objects
;! @return A packed list of doc. string objects
(define (pack-doc-strings lst)
  
  (define (pack lst)
    (let ((type (caar lst)))
      (cond
       ((memq type '(param field))
	`(param ,(cadar lst) ,(caddar lst) ,@(cdr lst)))
;;        ((eq? type 'end)                                                ;;
;; 	`(,type))                                                         ;;
       (else
	`(,type ,(cadar lst) ,@(cdr lst))))))
  
  (let loop ((l lst)) ; we first strip all 'doc types at the beginning
    (if (and (pair? l) (eq? (caar l) 'doc))
	(loop (cdr l))
	(if (null? l)
	    l
	    (let loop ((l (cdr l)) (docstr (list (car l))) (res '()))
	      (if (null? l)
		  (reverse (cons (pack (reverse docstr)) res))
		  (let ((fst (car l)))
		    (if (eq? (car fst) 'doc)
			(loop (cdr l) (cons (cadr fst) docstr) res)
			(loop (cdr l) 
			      (list fst)
			      (cons (pack (reverse docstr)) res))))))))))
	


