//
// File:        GenerateIORClient.java
// Package:     gov.llnl.babel.backend.ior
// Revision:    @(#) $Id: GenerateIORServer.java 4434 2005-03-17 17:05:29Z epperly $
// Description: generate IOR code based on a set of symbol identifiers
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.ior;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeGenerator;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.ior.IORHeader;
import gov.llnl.babel.backend.ior.IORSource;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import java.io.PrintWriter;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;

/**
 * Class <code>GenerateIORClient</code> is the main entry point for the IOR
 * code generation routines.  This class calls the IOR header and source
 * file generation routines.  It is assumed that all symbols necessary
 * to generate IOR code are available in the symbol table, which can be
 * guaranteed by calling <code>resolveAllReferences</code> on the symbol
 * table.
 */
public class GenerateIORServer implements CodeGenerator {

   /**
    * The constructor does nothing interesting.  The entry point for
    * the <code>GenerateIORClient</code> class is <code>generateCode</code>.
    */
   public GenerateIORServer() { }

   /**
    * Generate IOR code for each symbol identifier in the set argument.
    * This routine assumes that all necessary symbol references are available
    * in the symbol table.
    */
   public void generateCode(Set symbols) throws CodeGenerationException {
     for (Iterator s = symbols.iterator(); s.hasNext(); ) {
       SymbolID id     = (SymbolID) s.next();
       if (   BabelConfiguration.isSIDLBaseClass(id) 
          && !BabelConfiguration.getInstance().generateStdlib() )
       {
         continue;
       }
       Symbol   symbol = SymbolTable.getInstance().lookupSymbol(id);
       if (symbol != null) {
         generateSource(symbol);
       }
     }
   }

   /**
    * Generate IOR header code for the specified symbol.
    */
   private void generateHeader(Symbol symbol) throws CodeGenerationException {
      final int type = symbol.getSymbolType();
      final SymbolID id = symbol.getSymbolID();
      PrintWriter pw = null;
      try {
         String f = IOR.getHeaderFile(id);
         pw = FileManager.getInstance().
           createFile(id, type, "IORHDRS", f);
         LanguageWriterForC writer = new LanguageWriterForC(pw);
         IORHeader.generateCode(symbol, writer);
      } finally {
         if (pw != null) {
            pw.close();
         }
      }
   }

   /**
    * Generate IOR source code for the specified symbol.  Source code is
    * generated only for classes.
    */
   private void generateSource(Symbol symbol) throws CodeGenerationException {
      final SymbolID id = symbol.getSymbolID();
      final int type = symbol.getSymbolType();
      if ( type == Symbol.CLASS ) {
         PrintWriter pw = null;
         try {
            String f = IOR.getSourceFile(id);
            pw = FileManager.getInstance().
              createFile(id, type, "IORSRCS", f);
            LanguageWriterForC writer = new LanguageWriterForC(pw);
            IORSource.generateCode(symbol, writer);
         } finally {
            if (pw != null) {
               pw.close();
            }
         }
      }
   }

  public String getType() {
    return "skel";
  }

  public boolean getUserSymbolsOnly() {
    return true;
  }

  public Set getLanguages() {
    Set result = new TreeSet();
    result.add("ior");
    return result;
  }
}
