//
// File:        RmiStubHeader.java
// Package:     gov.llnl.babel.backend.cxx
// Revision:    @(#) $Revision: 4454 $
// Date:        $Date: 2005-03-22 14:18:30 -0800 (Tue, 22 Mar 2005) $
// Description: Write Cxx extension header file for a BABEL extendable
// 
// This is typically directed by GenCxxClient.
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.rmi2;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.rmi2.Cxx;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.writers.LanguageWriterForCxx;
import gov.llnl.babel.backend.writers.LineRedirector;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import java.util.Iterator;
/**
 * Create and write a header for a Cxx C extension class to wrap a 
 * BABEL extendable in a Cxx object. The header has to expose a 
 * function to create a wrapped IOR, a function to check if a 
 * <code>PyObject</code> is an instance of this extension type, and
 * an import macro.
 */
public class RmiStubHeader {
  private Extendable d_ext = null;
  private LanguageWriterForCxx d_writer = null;
  private CodeSplicer d_splicer = null;
  
  /**
   * Create an object capable of generating the header file for a
   * BABEL extendable.
   *
   * @param ext   an interface or class symbol that needs a header
   *              file for a Cxx C extension class.
   */
  public RmiStubHeader(Extendable ext) {
    d_ext = ext;
  }
  
  /**
   * Generate the header file for the extendable with which this object was
   * created.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode() 
    throws CodeGenerationException 
  {
    final SymbolID id = d_ext.getSymbolID();
    final int type = d_ext.getSymbolType();
    String filename = Cxx.generateFilename( id, 
                                            Cxx.FILE_ROLE_IMPL,
                                            Cxx.FILE_TYPE_CXX_HEADER );
    //System.out.println("Create " + filename + "...");

    try { 
      d_splicer = FileManager.getInstance().
        getCodeSplicer(id, type, filename);
      d_writer = Cxx.createHeader( d_ext, Cxx.FILE_ROLE_IMPL, "IMPLHDRS");
      d_splicer.setLineRedirector( (LineRedirector) d_writer );
      d_writer.println();
      d_writer.openHeaderGuard( filename );

      Cxx.generateDependencyIncludes( d_writer, d_ext, false );

      spliceIncludes();

      Cxx.nestPackagesInNamespaces( d_writer, d_ext );
      
      writeClassBeginning();

      writeSIDLDefinedMethods();

      writeClassEnd();

      Cxx.unnestPackagesInNamespaces( d_writer, d_ext );

      writeMiscStuff();

      d_writer.closeHeaderGuard();

      if (d_splicer.hasUnusedSymbolEdits()) {
        d_writer.println();
        d_writer.printlnUnformatted("#error File has unused splicer blocks.");
        d_writer.beginBlockComment(true);
        d_writer.println(CodeConstants.C_BEGIN_UNREFERENCED_METHODS);
        d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT1);
        d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT2);
        d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT3);
        d_writer.println(CodeConstants.C_END_UNREFERENCED_METHODS);
        d_splicer.outputUnusedSymbolEdits( d_writer.getPrintWriter() );
        d_writer.writeCommentLine(CodeConstants.C_END_UNREFERENCED_METHODS);
      }

    } catch ( java.io.IOException ex) { 
      throw new CodeGenerationException("IOException : " + ex.getMessage() );
    } finally { 
      if (d_writer != null) {
        d_writer.close();
        d_writer = null;
      }
    }
  }

  /**
   * write all the include statements including a block for 
   * user's includes.
   */
  private void spliceIncludes() { 
    d_writer.println();

    String includes = d_ext.getSymbolID().getFullName() + "." + "_includes";
    d_splicer.splice( includes, d_writer, "additional includes or code");
    d_writer.println();
  }

  private void writeClassBeginning() { 
    String name = d_ext.getSymbolID().getShortName();
    SymbolID id = d_ext.getSymbolID();
    String splicer_symbol_impl = d_ext.getSymbolID().getFullName() + "._implementation";
    String splicer_symbol_inherit = d_ext.getSymbolID().getFullName() + "._inherits";
    /*    d_writer.println("class " + name + "_impl : public " + 
     *                name + "_skel {");
     */
    d_writer.writeComment( d_ext, true );

    d_writer.println("class " + name + "_impl");
    d_splicer.splice( splicer_symbol_inherit, d_writer, 
                      "additional inheritance");
    d_writer.println("{");
    d_writer.println();
    d_writer.println("private:");
    d_writer.tab();
    d_writer.writeCommentLine("Pointer back to IOR.");
    d_writer.writeCommentLine("Use this to dispatch back through IOR vtable.");
    d_writer.println( name + " self;");
    d_writer.println();

    d_splicer.splice( splicer_symbol_impl, d_writer, 
                      "implementation details");
    d_writer.println();
    
    d_writer.backTab();
    d_writer.println("private:");
    d_writer.tab();
    d_writer.writeCommentLine("private default constructor (required)");
    d_writer.println(name + "_impl() {} ");
    d_writer.println();


    d_writer.backTab();
    d_writer.println("public:");
    d_writer.tab();
    d_writer.writeCommentLine("sidl constructor (required)");
    d_writer.writeCommentLine("Note: alternate Skel constructor doesn't call addref()");
    d_writer.writeCommentLine("(fixes bug #275)");
    d_writer.println(name + "_impl( " + IOR.getObjectName(id) + 
                     " * s ) : self(s,true) { _ctor(); }");
    d_writer.println();
    d_writer.writeCommentLine("user defined construction");
    d_writer.println("void _ctor();");
    d_writer.println();

    d_writer.writeCommentLine("virtual destructor (required)");
    d_writer.println("virtual ~" + name + "_impl() { _dtor(); }");
    d_writer.println();
    d_writer.writeCommentLine("user defined destruction");
    d_writer.println("void _dtor();");
    d_writer.println();
  }

  private void writeClassEnd() {
    String name = d_ext.getSymbolID().getShortName();
    d_writer.backTab();
    d_writer.print("};  "); // end of class
    d_writer.writeCommentLine("end class " + name + "_impl" );
    d_writer.println();
  }
  
  private void writeMiscStuff() { 
    String splicer_symbol = d_ext.getSymbolID().getFullName() + "._misc";
    d_splicer.splice( splicer_symbol, d_writer, 
                      "miscellaneous");
    d_writer.println();
  }

  private void writeSIDLDefinedMethods() throws CodeGenerationException { 
    d_writer.backTab();
    d_writer.println("public:");
    d_writer.tab();

    Iterator m = null;
    m = d_ext.getStaticMethods(false).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      Cxx.generateMethodSignature( d_writer, method, 
                                   "user defined static method", 
                                   Cxx.FILE_ROLE_IMPL);
    }
    d_writer.println();

    m = d_ext.getNonstaticMethods(false).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      if ( !method.isAbstract() ) { 
	  Cxx.generateMethodSignature( d_writer, method, 
                                       "user defined non-static method.", 
				       Cxx.FILE_ROLE_IMPL );
      }
    }
  }
}
