/* 
 *   Creation Date: <2001/05/06 22:27:09 samuel>
 *   Time-stamp: <2001/05/24 14:07:01 samuel>
 *   
 *	<fs.c>
 *	
 *	Generic file system access
 *   
 *   Copyright (C) 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"
#include "fs.h"

static void		native_close_fs( fs_ops_t *fs );
static file_desc_t	dummy_search_rom( fs_ops_t *fs );
static file_desc_t 	dummy_search_file( fs_ops_t *fs, char *name );
static file_desc_t 	native_open_path( fs_ops_t *fs, char *path );
static char		*native_get_path( file_desc_t fd, char *buf, int len );
static void		native_close( file_desc_t fd );
static ssize_t		native_read( file_desc_t fd, void *buf, size_t count );
static int		native_lseek( file_desc_t fd, off_t offset, int whence );

static fs_ops_t native_ops = {
	close_fs:	native_close_fs,
	open_path:	native_open_path,
	get_path:	native_get_path,
	close:		native_close,
	read:		native_read,
	lseek:		native_lseek,

	search_rom:	dummy_search_rom,
	search_file:	dummy_search_file
};


/************************************************************************/
/*	fs_open								*/
/************************************************************************/

fs_ops_t *
fs_open( int fs_type, char *path, ullong offs )
{
	switch( fs_type ){
	case FS_HFS:
		return fs_hfs_open( offs, path );
	case FS_HFS_PLUS:
		return fs_hfsp_open( offs, path );
	case FS_NATIVE:
		return fs_native_open();
	}
	return NULL;
}


/************************************************************************/
/*	Native filesystem wrapper					*/
/************************************************************************/

typedef struct {
	int	fd;
	char	*path;
} native_fd_t;

fs_ops_t *
fs_native_open( void )
{
	fs_ops_t *fs = malloc( sizeof(fs_ops_t) );
	*fs = native_ops;
	return fs;
}

static void
native_close_fs( fs_ops_t *fs ) 
{
	free( fs );
}

static file_desc_t
native_open_path( fs_ops_t *fs, char *path )
{
	native_fd_t *p;
	int fd;

	if( (fd=open( path, O_RDONLY )) < 0 )
		return NULL;
	p = malloc( sizeof(native_fd_t) );
	p->fd = fd;
	p->path = strdup(path);
	return (file_desc_t)p;
}

static int
native_lseek( file_desc_t fd, off_t offset, int whence )
{
	native_fd_t *p = (native_fd_t*)fd;
	return lseek( p->fd, offset, whence );
}

static ssize_t
native_read( file_desc_t fd, void *buf, size_t count )
{
	native_fd_t *p = (native_fd_t*)fd;
	return read( p->fd, buf, count );
}


static void
native_close( file_desc_t fd )
{
	native_fd_t *p = (native_fd_t*)fd;
	close( p->fd );
	free( p->path );
	free( p );
}

static char *
native_get_path( file_desc_t fd, char *buf, int len )
{
	native_fd_t *p = (native_fd_t*)fd;
	strncpy0( buf, p->path, len );
	return buf;
}

static file_desc_t
dummy_search_rom( fs_ops_t *fs )
{
	return NULL;
}

static file_desc_t
dummy_search_file( fs_ops_t *fs, char *name )
{
	return NULL;
}

