/*
 * Copyright 2006 Ricoh Corporation.
 * 
 * 
 * APACHE LICENSE VERSION 2.0
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * 
 * RICOH DEVELOPER PROGRAM SUPPORT:
 * 
 * Support for this software is available only to "Premier Plus" members
 * of the Ricoh Developer Program (RiDP).  You may find out more 
 * information about the Program at
 * 
 *      http://americas.ricoh-developer.com
 * 
 * Premier plus members may find answers and ask questions through the
 * RiDP customer help website at
 * 
 *      https://ridp.custhelp.com
 * 
 * Developers who are not RiDP members may still use this software as
 * stipulated in the license terms given above.
 *
 */ 

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URLConnection;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Map;

import com.hp.hpl.sparta.Document;
import com.hp.hpl.sparta.Element;
import com.hp.hpl.sparta.Node;
import com.hp.hpl.sparta.ParseException;
import com.hp.hpl.sparta.Parser;

public class XHTMLFormFactory
{	
	/**
	 *	Make an empty form that may be programatically populated with controls.
	 */
	public static XHTMLForm create()
	{
		return new XHTMLForm();
	}
	
	/**
	 *	Create an XHTMLForm object initialized by the contents of a
	 *	page read from the provided input stream.
	 */
	public static XHTMLForm loadForm(String referenceString, InputStream inStream, String formName) throws IOException, ParseException
	{
		Document xmlDoc = Parser.parse(referenceString, new BufferedReader(new InputStreamReader(inStream)));
		
		XHTMLForm result = XHTMLFormFactory.getFormFromDocument(xmlDoc, formName);
		XHTMLFormFactory.setFormTitleFromDocument(xmlDoc, result);
		XHTMLFormFactory.setFormMessageFromDocument(xmlDoc, result);
		
		return result;
	}
	
	public static XHTMLForm getFormFromDocument(Document xmlDoc, String formName) throws ParseException
	{
		return XHTMLFormFactory.getFormFromFormElement(xmlDoc.xpathSelectElement("//form[@name='" + formName + "']"));
	}

	public static void setFormTitleFromDocument(Document xmlDoc, XHTMLForm theForm)
	{
		try
		{
			System.out.println("Try to get title from document");
			Element title = xmlDoc.xpathSelectElement("/html/head/title");
			if(title != null)
			{
				System.out.println("The title is " + title.toString());
				theForm.setTitle(title.toString());
			}
		}
		catch(Exception e)
		{
		}
	}

	public static void setFormMessageFromDocument(Document xmlDoc, XHTMLForm theForm)
	{
		try
		{
			System.out.println("Try to find message in document");
			Element message = xmlDoc.xpathSelectElement("//span[@id='message']");
			if(message != null)
			{
				String theMessage = message.toString().trim();
				System.out.println("Trimmed message is " + theMessage);
				theForm.setMessage(theMessage);
			}
		}
		catch(Exception e)
		{
		}
	}
	

	public static XHTMLForm getFormFromFormElement(Element formElement) throws ParseException
	{
		XHTMLForm result = new XHTMLForm();
		Map labels = new Hashtable();
		
		if(formElement != null)
		{
			//
			// we expect to see an 'enctype' attribute whose value is either URLENCODED or MULTIPART.
			// if it is MULTIPART, then call setMultipartEncoded(true);
			//
			String encodingType = formElement.getAttribute("enctype");
			if((encodingType != null) && (encodingType.equalsIgnoreCase(XHTMLForm.MULTIPART)))
			{
				System.out.println("the form is a multipart encoded form");
				result.setMultipartEncoded(true);
			}

			String action = formElement.getAttribute("action");
			if(action != null)
			{
				System.out.println("the form action is " + action);
				result.setAction(action);
			}
			
			findLabels(formElement, labels);
			parseTree(formElement, result, labels);
		}
		else
			System.out.println("No form element found in getFormFromFormElement");
		
		System.out.flush();
			
		return result;
	}
	
	/* package-private */
	static void addInputItem(XHTMLForm theForm, String inputItemName, XHTMLForm.FormItem.ItemType inputItemType, String itemDescription, Element oneElement, String selectedAttribute) throws ParseException
	{
		System.out.println("adding an input element of type " + inputItemType.toString() + " named " + inputItemName + " with label " + itemDescription);
		String initialValue = oneElement.getAttribute("value");
		System.out.println("Its value is " + initialValue);
		if(itemDescription.length() == 0)
		{
			itemDescription = oneElement.toString();
			System.out.println("Its description is " + itemDescription);
		}
		boolean itemIsSelected = true;
		if(oneElement.getAttribute(selectedAttribute) == null)
			itemIsSelected = false;
		
		theForm.setItemValue(inputItemName, inputItemType, initialValue, itemDescription, itemIsSelected);		
	}
	
	
	private static void findLabels(Element current, Map labels) throws ParseException 
	{
		if(current != null)
		{
			String tagName = current.getTagName();
			Node child = current.getFirstChild();
					
			if(tagName.equalsIgnoreCase("label"))
			{
				String labelFor = current.getAttribute("for");
				String labelDescription = current.toString();
				System.out.println("Description for item " + labelFor + " is " + labelDescription);	
				labels.put(labelFor, labelDescription);
			}
			
		
			while (child != null )
			{
				if (child instanceof Element)
					findLabels((com.hp.hpl.sparta.Element)child, labels);
					
				child = child.getNextSibling();
			}
		}		
	}
	
	private static void parseTree(Element current, XHTMLForm theForm, Map labels) throws ParseException 
	{
		if(current != null)
		{
			String tagName = current.getTagName();
			String inputItemName = current.getAttribute("name");
			String itemDescription = "";
			if((inputItemName != null) && (labels.containsKey(inputItemName)))
				itemDescription = (String)labels.get(inputItemName);
			Node child = current.getFirstChild();
					
			//System.out.println("Parsing "+tagName);
			
			if(tagName.equalsIgnoreCase("input"))
			{
				//boolean itemIsSelected = true;
				XHTMLForm.FormItem.ItemType inputItemType = XHTMLForm.FormItem.ItemType.valueOf(current.getAttribute("type"));
				
				if((inputItemName != null) || (inputItemType == XHTMLForm.FormItem.ItemType.SUBMIT))
				{
					System.out.println("found input element named " + inputItemName);
					XHTMLFormFactory.addInputItem(theForm, inputItemName, inputItemType, itemDescription, current, "checked");
				}
			}
			else if(tagName.equalsIgnoreCase("select"))
			{
				XHTMLForm.FormItem.ItemType inputItemType = XHTMLForm.FormItem.ItemType.DROPDOWN;

				while (child != null )
				{
					if( (child instanceof Element) && (((Element)child).getTagName().equalsIgnoreCase("option")))
					{
						XHTMLFormFactory.addInputItem(theForm, inputItemName, inputItemType, "", (Element)child, "selected");
					}
					child = child.getNextSibling();
				}

				theForm.setItemLabel(inputItemName, itemDescription);		
				
			}
			
			while (child != null )
			{
				if (child instanceof Element)
					parseTree((com.hp.hpl.sparta.Element)child, theForm, labels);
					
				child = child.getNextSibling();
			}
		}		
	}

}
