/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.cnd.repository.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.netbeans.modules.cnd.repository.disk.StorageAllocator;
import org.netbeans.modules.cnd.repository.spi.Key;
import org.netbeans.modules.cnd.repository.spi.Persistent;
import org.netbeans.modules.cnd.repository.testbench.Stats;

/**
 * Hybrid repository
 *
 * @author Sergey Grinev
 */
public class SoftRefHybridRepository extends SoftRefRepository {
    
    private final Map<Key, Persistent> hardMap = new HashMap<Key, Persistent>();
    private final ReadWriteLock rwLock = new ReentrantReadWriteLock(true);
    
    public void hang(final Key key, final Persistent obj) {
        try {
            rwLock.writeLock().lock();
            hardMap.put(key, obj);
        } finally {
            rwLock.writeLock().unlock();
        }
    }
    
    public void put(final Key key, final Persistent obj) {
        try {
            rwLock.writeLock().lock();
            
            hardMap.remove(key);
            super.put(key, obj);
            
            if (Stats.isDebug && Stats.debugPut%4096==0) {
                Stats.report(hardMap.size(), super.size());
            }
        } finally {
            rwLock.writeLock().unlock();
        }
    }
    
    public Persistent get(final Key key) {
        try {
            rwLock.readLock().lock();
            final Persistent obj = hardMap.get(key);
            if (obj != null) {
                ++Stats.debugGotFromHardCache;
            }
            return obj != null ? obj : super.get(key);
        } finally {
            rwLock.readLock().unlock();
        }
    }
    
    public void remove(final Key key) {
        try {
            rwLock.writeLock().lock();
            
            if (hardMap.remove(key) == null) {
                super.remove(key);
            }
        } finally {
            rwLock.writeLock().unlock();
        }
    }
    
    public void closeUnit(final String unitName, final boolean cleanRepository, Set<String> requiredUnits) {
        try {
            rwLock.writeLock().lock();
            
            final Set<Key> keys = new HashSet<Key>(hardMap.keySet());
            for (Key key : keys) {
                if (unitName.equals(key.getUnit())) {
                    
                    if (!cleanRepository) {
                        super.put(key, hardMap.get(key));
                    }
                    
                    hardMap.remove(key);
                }
            }
            
            super.closeUnit(unitName, cleanRepository, null);
            
        } finally {
            rwLock.writeLock().unlock();
        }
    }
    
    public void shutdown() {
        try {
            rwLock.writeLock().lock();
            super.shutdown();
        } finally {
            rwLock.writeLock().unlock();
        }
    }
    
    public void debugClear() {
        try {
            rwLock.writeLock().lock();
            super.debugClear();
        } finally {
            rwLock.writeLock().unlock();
        }
    }    
}
