/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.editor.hints;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;
import org.openide.ErrorManager;
import org.openide.filesystems.FileLock;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.util.WeakSet;

/**
 *
 * @author Jan Lahoda
 */
public class PersistentCache {
    
    private static PersistentCache INSTANCE = new PersistentCache();
    
    public static PersistentCache getDefault() {
        return INSTANCE;
    }
    
    private static Map<FileObject, Set<FileObject>> fo2ContainedErrorFiles = new WeakHashMap<FileObject, Set<FileObject>>(); //TODO: persistent! (?)
    private static Set<FileObject> knownObjects = new WeakSet<FileObject>();
    
    /** Creates a new instance of PersistentCache */
    private PersistentCache() {
    }
    
    public void addErrorFile(FileObject key, FileObject error) {
        Set<FileObject> contained = fo2ContainedErrorFiles.get(key);
        
        if (contained == null) {
            fo2ContainedErrorFiles.put(key, contained = new HashSet<FileObject>());
        }
        
        if (contained.add(error)) {
            //TODO: fireFileStatusChanged(key);
        }
        
        knownObjects.add(error);
    }
    
    public void removeErrorFile(FileObject key, FileObject error) {
        Set<FileObject> contained = fo2ContainedErrorFiles.get(key);
        
        if (contained == null) {
            fo2ContainedErrorFiles.put(key, contained = new HashSet<FileObject>());
        }
        
        if (contained.remove(error)) {
            //TODO: fireFileStatusChanged(key);
        }
        
        knownObjects.add(error);
    }
    
    public boolean hasErrors(FileObject file) {
        Set<FileObject> contained = fo2ContainedErrorFiles.get(file);
        
        return contained != null && !contained.isEmpty();
    }
    
    public boolean isKnown(FileObject file) {
        return knownObjects.contains(file);
    }
    
    private FileObject cache = null;
    
    private synchronized FileObject cacheFile() {
        if (cache == null) {
            try {
                String nbuser = System.getProperty("netbeans.user");
                File userDir  = new File(nbuser);
                FileObject userDirFO = FileUtil.toFileObject(userDir);
                
                cache = FileUtil.createFolder(userDirFO, "var/cache/errors");
            } catch (IOException e) {
                ErrorManager.getDefault().notify(e);
            }
        }
        
        return cache;
    }
    
    private FileObject getCacheFileFor(FileObject file) throws IOException {
        File f = FileUtil.toFile(file);
        
        if (f == null)
            return null;
        
        String path = f.getAbsolutePath().replace(':', '-').replace(File.separatorChar, '-');
        FileObject directory = FileUtil.createFolder(cacheFile(), path);
        FileObject errors = directory.getFileObject("errors");
        
        if (errors == null)
            errors = directory.createData("errors");
        
        return errors;
    }
    
    void saveCache() {
        for (FileObject file : fo2ContainedErrorFiles.keySet()) {
            try {
                FileObject cache = getCacheFileFor(file);
                
                if (cache == null)
                    continue;
                
                FileLock lock = cache.lock();
                OutputStream out = cache.getOutputStream(lock);
                
                try {
                    Set<FileObject> errors = fo2ContainedErrorFiles.get(file);
                    
                    for (Iterator e = errors.iterator(); e.hasNext(); ) {
                        FileObject f = (FileObject) e.next();
                        
                        out.write(FileUtil.toFile(f).getAbsolutePath().getBytes());
                        out.write('\n');
                    }
                } finally {
                    out.close();
                    lock.releaseLock();
                }
            } catch (IOException e) {
                ErrorManager.getDefault().notify(e);
            }
        }
    }
    
    void loadCache() {
        loadCacheRecursive("");
    }

    private void loadCacheRecursive(String path) {
        FileObject f = cacheFile().getFileObject(path);
        FileObject[] children = f.getChildren();
        
        for (int cntr = 0; cntr < children.length; cntr++) {
            FileObject c = children[cntr];
            
            if (c.isFolder()) {
                loadCacheRecursive(path + "/" + c.getNameExt());
            }
        }
        
        try {
            FileObject errors = f.getFileObject("errors");
            
            if (errors == null)
                return ;
            
            InputStream ins = errors.getInputStream();
            BufferedReader reader = new BufferedReader(new InputStreamReader(ins));
            Set<FileObject> errorFiles = new HashSet<FileObject>();
            
            try {
                String line;
                
                while ((line = reader.readLine()) != null) {
                    File resolved = new File(line);
                    FileObject resolvedFO = FileUtil.toFileObject(FileUtil.normalizeFile(resolved));
                    
                    if (resolvedFO != null) {
                        errorFiles.add(resolvedFO);
                    }
                }
            }finally {
                reader.close();
            }
            
            FileObject original = FileUtil.toFileObject(new File("/", path));
            
            if (original != null) {
                fo2ContainedErrorFiles.put(original, errorFiles);
                knownObjects.add(original);
            }
        } catch (IOException e) {
            ErrorManager.getDefault().notify(e);
        }
    }
    
}
