/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.bpel.core.validation;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.WeakHashMap;

import org.netbeans.modules.bpel.model.api.BpelModel;
import org.netbeans.modules.bpel.model.api.events.ChangeEvent;
import org.netbeans.modules.bpel.model.api.events.ChangeEventListenerAdapter;
import org.netbeans.modules.xml.xam.Model.State;
import org.netbeans.modules.xml.xam.spi.Validation;
import org.netbeans.modules.xml.xam.spi.Validation.ValidationType;
import org.netbeans.modules.xml.xam.spi.Validator.ResultItem;

/**
 * Listens to the BPEL model changes and invokes the validators
 * when appropriate.
 * Clients like the editor, navigator can register a listner with
 * this class to listen to changes to ValidationResults.
 *
 * @author Praveen
 * @author changed by ads
 */
public class BPELValidationController extends ChangeEventListenerAdapter {
    
    /**
     * Creates a new instance of BPELValidationController
     */
    public BPELValidationController(BpelModel bpelModel) {
        myWeaklisteners = new WeakHashMap<BPELValidationListener, Object>();
        myBpelModel = bpelModel;
        myTrigger = new ExternalModelsValidationTrigger( this );
    }

    public void attach() {
        if(myBpelModel != null) {
            myBpelModel.addEntityChangeListener(this);
            myBpelModel.addEntityChangeListener( getTrigger() );
            getTrigger().loadImports( );
        }
    }

    public void detach() {
        if(myBpelModel != null) {
            myBpelModel.removeEntityChangeListener(this);
            myBpelModel.removeEntityChangeListener( getTrigger() );
            getTrigger().clearTrigger();
        }
    }

    
    /**
     *  Add a validation listener.
     *  Listeners are maintained as weaklisteners to clients should maintain
     *  a strong reference to the listener.
     *
     */
    public void addValidationListener(BPELValidationListener listener) {
        synchronized(myWeaklisteners) {
            myWeaklisteners.put(listener, null);
        }
    }
    
    /**
     * Remove a validation listener. Although listners are maintained as
     * weak listeners, clients can explicity unregister a listener.
     */
    public void removeValidationListener(BPELValidationListener listener) {
        synchronized(myWeaklisteners) {
            myWeaklisteners.remove(listener);
        }
    }
    
    BpelModel getModel() {
        return myBpelModel;
    }
    
    /**
     *  Return current validation results.
     */
    private List<ResultItem> getFastValidationResults() {
        return myLatestFastValidationResult;
    }
    
    /**
     *  Return current validation results.
     */
    private List<ResultItem> getSlowValidationResults() {
        return myLatestSlowValidationResult;
    }
    
    /**
     *  Called when the model has changed.
     */
    private void modelChanged(ChangeEvent event) {
        // Validate on the last event in a chain.
        if(!event.isLastInAtomic()) {
            return;
        }
        startValidation();
    }
    
    /**
     *  Use this for clients who want to initiate validation.
     *  For example when initially opening the editor. Make sure you
     *  attach a listener first and then call this method, so that you are
     *  notified with the validation results.
     */
    public void triggerValidation() {
        triggerValidation( false );
    }
    
    /**
     * Initiate validation.
     * 
     * @param checkExternallyTriggered if true then validation will be 
     * started only in the case external artifact was modifed 
     */
    public void triggerValidation( boolean checkExternallyTriggered ) {
        if ( checkExternallyTriggered && getTrigger().isTriggerDirty()) {
            startValidation();
        }
        else if ( !checkExternallyTriggered) {
            startValidation();
        }
    }
    
    public void notifyCompleteValidationResults(List<ResultItem> results) {
        //System.out.println(" Complete validation results obtained.");
        // Filter to keep only slow validator results here.
        myLatestSlowValidationResult = new ArrayList<ResultItem>();
        myLatestFastValidationResult = new ArrayList<ResultItem>();
        
        for (ResultItem result: results){
            ( ValidationUtil.isSlowValidationResult(result)?
                myLatestSlowValidationResult :
                myLatestFastValidationResult).add(result);
            
        }
        notifyListeners();
    }
    
    /**
     *  Listener to listen to Object model changes.
     */
    @Override
    public void notifyEvent(ChangeEvent changeEvent) {
        if ( !State.VALID.equals(myBpelModel.getState())){
            return;
        }
        modelChanged(changeEvent);
    }
    
    private void startValidation() {
        synchronized(lock) {
            final TimerTask timerTask= new TimerTask() {
                public void run() {

                    // Run at a low priority.
                    Thread.currentThread().setPriority(Thread.NORM_PRIORITY);

                    Validation validation = new Validation();
                    validation.validate(myBpelModel, ValidationType.PARTIAL);
                    myLatestFastValidationResult = validation
                            .getValidationResult();

                    notifyListeners();
                }
            };
            myTimer.cancel(); // This removes any tasks on the queue while
                            // allowing any already running task to complete.
            
            myTimer = new Timer();
            myTimer.schedule(timerTask, DELAY);
        }
    }
    
    /**
     *  Listeners are notified about change in ValidationResult.
     *  Happens on a non-AWT thread.
     */
    private void notifyListeners() {
        synchronized(myWeaklisteners) {
            List<ResultItem> mergedResults = new ArrayList<ResultItem>();
            if(getFastValidationResults() != null)
                mergedResults.addAll(getFastValidationResults());
            if(getSlowValidationResults() != null)
                mergedResults.addAll(getSlowValidationResults());
            //System.out.println(" MergedResults. count= " + mergedResults.size());
            for(BPELValidationListener listener: myWeaklisteners.keySet()) {
                if(listener != null)
                    listener.validationUpdated(mergedResults);
            }
        }
    }
    
    private ExternalModelsValidationTrigger getTrigger() {
        return myTrigger;
    }
    
    private ExternalModelsValidationTrigger myTrigger;
    private Map<BPELValidationListener, Object> myWeaklisteners;
    private Object lock = new Object();
    private BpelModel myBpelModel;
    private Timer myTimer = new Timer();
    
    private List<ResultItem> myLatestSlowValidationResult;
    private List<ResultItem> myLatestFastValidationResult;
    
    private static final int DELAY = 4000;
    
}
