/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.sun.manager.jbi.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.util.logging.Logger;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

/**
 *  Provides access to the ESB Repository for remote clients.  The scope of
 *  this class is limited to operations which require the transfer of
 *  archive data to/from the ESB Central Administration Server.  All other
 *  repository operations can be accessed through the appropriate ESB MBean.
 *
 *  The default chunk size for archive transfer is 8K.  This setting can be
 *  adjusted using <code>setChunkSize</code> if necessary.
 */
public class FileTransferManager {
    /** Name of the ArchiveDownload MBean */
    private static final String DOWNLOAD_MBEAN_NAME =
            "com.sun.jbi.esb:ServiceType=ArchiveDownload"; // NOI18N
    
    /** Name of the ArchiveUpload MBean */
    private static final String UPLOAD_MBEAN_NAME =
            "com.sun.jbi.esb:ServiceType=ArchiveUpload"; // NOI18N
    
    /** Name of the ArchiveUpload MBean */
    private static final String REFERENCE_UPLOAD_MBEAN_NAME =
            "com.sun.jbi:Target=domain,ServiceName=FileTransferService,ServiceType=Upload"; // NOI18N
    
    /** Default chunk size for archive transfer. */
    private static final int DEFAULT_CHUNK_SIZE = 8 * 1024;
    
    /** Upload MBean operations. */
    private static final String UPLOAD_INITIATE     = "initiateUpload"; // NOI18N
    private static final String UPLOAD_TRANSFER     = "uploadBytes"; // NOI18N
    private static final String GET_ARCHIVE_URL     = "getArchiveURL"; // NOI18N
    private static final String REFERENCE_GET_ARCHIVE_URL     = "getArchiveFilePath"; // NOI18N
    private static final String UPLOAD_TERMINATE    = "terminateUpload"; // NOI18N
    private static final String REMOVE_ARCHIVE      = "removeArchive"; // NOI18N
    
    /** Download MBean operations. */
    private static final String DOWNLOAD_INITIATE   = "initiateDownload"; // NOI18N
    private static final String DOWNLOAD_TRANSFER   = "downloadBytes"; // NOI18N
    private static final String DOWNLOAD_TERMINATE  = "terminateDownload"; // NOI18N
    
    private MBeanServerConnection   mServer;
    private ObjectName              mDownloadMBean;
    private ObjectName              mUploadMBean;
    private int                     mChunkSize = DEFAULT_CHUNK_SIZE;
    
    private Object mLastUploadId = null;
    
    public FileTransferManager(MBeanServerConnection server)
    throws MalformedObjectNameException {
        mServer = server;
        
        mDownloadMBean  = new ObjectName(DOWNLOAD_MBEAN_NAME);
        if(true == this.isMBeanRegistered(UPLOAD_MBEAN_NAME)) {
            mUploadMBean = new ObjectName(UPLOAD_MBEAN_NAME);
        } else {
            mUploadMBean = new ObjectName(REFERENCE_UPLOAD_MBEAN_NAME);
        }
    }
    
    /**
     * Tests to see if MBean is registered
     * @param objectNameString
     * @return
     */
    boolean isMBeanRegistered(String objectNameString) {
        boolean result = false;
        ObjectName objectName = null;
        
        try {
            objectName = new ObjectName(objectNameString);
            result = this.mServer.isRegistered(objectName);
        } catch (MalformedObjectNameException e) {
        } catch (NullPointerException e) {
        } catch (IOException e) {
        }
        
        return result;
    }
    
    /**
     * Uploads the specified archive to a temporary store in the ESB repository.
     * @return CAS-local path to the archive
     */
    public String uploadArchive(File archive)
            throws Exception {
        FileInputStream fis = null;
        String result = ""; // NOI18N
        
        try {
            // setup our stream before initiating the upload session
            fis = new FileInputStream(archive);
            
            // initiate an upload session
            Object uploadId = mServer.invoke(
                    mUploadMBean,
                    UPLOAD_INITIATE,
                    new Object[] {archive.getName()},
                    new String[] {"java.lang.String"}); // NOI18N
            
            // remember the last uploaded id as it can
            // be used for remove archive later
            this.mLastUploadId = uploadId;
            
            // get the upload url from the sesssion
            Object uploadUrlObj = null;
            boolean isOldMBean = this.isMBeanRegistered(UPLOAD_MBEAN_NAME); // GlassFish 9.0
            if(true == isOldMBean) {
                uploadUrlObj = mServer.invoke(
                        mUploadMBean,
                        GET_ARCHIVE_URL,
                        new Object[] {uploadId},
                        new String[] {"java.lang.Object"}); // NOI18N
            } else {
                uploadUrlObj = mServer.invoke(
                        mUploadMBean,
                        REFERENCE_GET_ARCHIVE_URL,
                        new Object[] {uploadId},
                        new String[] {"java.lang.Object"}); // NOI18N
            }
            
            
            // transfer the archive content to the server, chunk by chunk
            byte[] chunk = new byte[mChunkSize];
            int    count = 0;
            while ((count = fis.read(chunk)) != -1) {
                mServer.invoke(
                        mUploadMBean,
                        UPLOAD_TRANSFER,
                        new Object[] {uploadId, trim(chunk, count)},
                        new String[] {"java.lang.Object", "[B"}); // NOI18N
            }
            
            // transfer complete, terminate the session
            mServer.invoke(
                    mUploadMBean,
                    UPLOAD_TERMINATE,
                    new Object[] {uploadId},
                    new String[] {"java.lang.Object"}); // NOI18N
            
            if(true == isOldMBean) {
                URL uploadURL = new URL(uploadUrlObj.toString());
                URI uploadURI = uploadURL.toURI();
                //        File uploadFile = new File(uploadURI);
                //        String uploadFilePath = uploadFile.getAbsolutePath();
                //
                ////        System.out.println("#### UPLOAD ID " + uploadId);
                ////        System.out.println("#### UPLOAD URL OBJ " + uploadUrlObj);
                ////        System.out.println("#### UPLOAD URI " + uploadURI.toString());
                ////        System.out.println("#### UPLOAD File Path " + uploadFilePath);
                //
                //        return uploadFilePath;
                result = uploadURI.getPath();
            } else {
                result = uploadUrlObj.toString();
            }
        } finally {
            if (fis != null) {
                fis.close();
            }
        }
        
        return result;
    }
    
    public Object getLastUploadId() {
        return this.mLastUploadId;
    }
    
    /**
     * this will remove any previously uploaded archive by using its uploaded id
     */
    public void removeUploadedArchive(Object uploadId) {
        if ( uploadId == null ) {
            return;
        }
        try {
            mServer.invoke(
                    mUploadMBean,
                    REMOVE_ARCHIVE,
                    new Object[] {uploadId},
                    new String[] {"java.lang.Object"}); // NOI18N
        } catch ( Exception ex) {
            // ignore any exception as the removal of the archive file is not important
            // and the server during restart anyway cleans it up.
            //TODO log the exception if you want.
            ex.printStackTrace();
        }
    }
    
    public void downloadArchive(String archiveLocation, File target)
            throws Exception {
        Object              downloadId;
        FileOutputStream    fos;
        
        // setup our stream before initiating the download session
        fos = new FileOutputStream(target);
        
        // initiate a download session
        downloadId = mServer.invoke(
                mDownloadMBean,
                DOWNLOAD_INITIATE,
                new Object[] {archiveLocation},
                new String[] {"java.lang.Object"}); // NOI18N
        
        // transfer the archive content from the server, chunk by chunk
        byte[] chunk;
        do
        {
            chunk = (byte[]) mServer.invoke(
                    mDownloadMBean,
                    DOWNLOAD_TRANSFER,
                    new Object[] {downloadId, new Integer(mChunkSize)},
                    new String[] {"java.lang.Object", "int"}); // NOI18N
            
            fos.write(chunk);
            fos.flush();
        }
        while (chunk.length > 0);
        
        // transfer complete, terminate the session
        mServer.invoke(
                mDownloadMBean,
                DOWNLOAD_TERMINATE,
                new Object[] {downloadId},
                new String[] {"java.lang.Object"}); // NOI18N
        
        fos.close();
    }
    
    /** Return the chunk size for archive transfers. */
    public int getChunkSize() {
        return mChunkSize;
    }
    
    /** Sets the chuck size for archive transfers. */
    public void setChunkSize(int size) {
        mChunkSize = size;
    }
    
    /**
     * Returns a byte array of the specified size, using the source array
     * as input.  If the length of the source array is equal to the desired
     * size, the original array is returned.
     */
    private byte[] trim(byte[] source, int size) {
        byte[] trimmed;
        
        if (source.length == size) {
            trimmed = source;
        } else {
            trimmed = new byte[size];
            System.arraycopy(source, 0, trimmed, 0, size);
        }
        
        return trimmed;
    }
}
