/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.exceptions.utils;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.NoResultException;
import javax.persistence.Persistence;
import javax.persistence.Query;
import org.netbeans.modules.exceptions.entity.Exceptions;
import org.netbeans.modules.exceptions.entity.Issue;
import org.netbeans.modules.exceptions.utils.Query.Operator;

/**
 *
 * @author Jan Horvath
 */
public final class PersistenceUtils {
    
    
    public static final String STRUTS_PU = "StrutsExceptionsPU";
    
    public static final String TEST_PU = "TestPU";

    public static String DEFAULT_PU = STRUTS_PU;
    
    public static String SETTINGS_DIRECTORY = null;
    
    public static final String JNDI_NAME = "bean/Utils";
    
    static final Logger LOG = Logger.getLogger(PersistenceUtils.class.getName());
    
    private static PersistenceUtils DEFAULT;
    
    private EntityManagerFactory emf;
    
    private Components components = null;
    
    
    
    private PersistenceUtils(String pu) {
        emf = Persistence.createEntityManagerFactory(pu, new HashMap());
    }
    
    public static synchronized void initTestPersistanceUtils(File db) {
        if (DEFAULT != null) {
            DEFAULT.close();
        }
        db.mkdirs();
        System.setProperty("derby.system.home", db.getPath());
        DEFAULT = new PersistenceUtils(TEST_PU);
    }

    synchronized static void setDefault(String pu) {
        if (DEFAULT != null) {
            throw new IllegalStateException("There already is a factory " + DEFAULT);
        }
        try {
            DEFAULT = new PersistenceUtils(pu);
        } catch (Throwable ex) {
            LOG.log(Level.SEVERE, ex.getMessage(), ex);
        }
    }
    
    public static synchronized PersistenceUtils getInstance() {
        if (DEFAULT != null) {
            return DEFAULT;
        }

        try {
            Context initCtx = new InitialContext();
            Context envCtx = (Context) initCtx.lookup("java:comp/env");
            DEFAULT = (PersistenceUtils) envCtx.lookup(JNDI_NAME);
        } catch (NamingException namingException) {
            Logger.getLogger(PersistenceUtils.class.getName()).log(
                    Level.INFO, "IMPOSSIBLE TO LOOKUP AN INSTANCE", namingException);
        }
                
        
        return DEFAULT;
    }
    
    public static void setSettingsDirectory(String def){
        SETTINGS_DIRECTORY = def;
    }
    
    public static String getSettingsDirectory(){
        return SETTINGS_DIRECTORY;
    }
    
    public Components getComponents() {
        if (components == null) {
            Components.setSettingsDirectory(SETTINGS_DIRECTORY);
            components = Components.getInstance();
        }
        return components;
    }
    
    /**
     * @returns List of results invoked by query
     * @param name name of the named query to call
     * @param params params contains query params in a map
     *  key is String of param name and value is a param object
     *  may be null if there are not any params
     */
    public List executeNamedQuery(String name, Map<String, ? extends Object> params){
        EntityManager em = getEM();
        Query query = em.createNamedQuery(name);
        setParams(query, params);
        List list = query.getResultList();
        em.close();
        return list;
    }

    /**
     * @returns List of results invoked by query
     * @param name name of the named query to call
     * @param params params contains query params in a map
     *  key is String of param name and value is a param object
     *  may be null if there are not any params
     */
    public Object executeNamedQuerySingleResult(String name, Map<String, Object> params){
        EntityManager em = getEM();
        Query query = em.createNamedQuery(name);
        setParams(query, params);
        Object result = null;
        try {
            result = query.getSingleResult();
        } catch (NoResultException e) {
        }
        em.close();
        return result;
    }
    
    /**
     * @returns List of results invoked by query
     * @param query string containg a SQL SELECT
     * @param params params contains query params in a map
     *  key is String of param name and value is a param object
     *  may be null if there are not any params
     *
     */
    public List executeQuery(String query, Map<String, ? extends Object> params){
        EntityManager em = getEM();
        Query queryRes = em.createQuery(query);
        setParams(queryRes, params);
        List list = queryRes.getResultList();
        em.close();
        return list;
    }
    
    public List executeNativeQuery(String query){
        EntityManager em = getEM();
        Query queryRes = em.createNativeQuery(query);
        List list = queryRes.getResultList();
        em.close();
        return list;
    }
    
    private void setParams(Query query, Map<String, ? extends Object> params){
        if (params != null){
            Set<String> keys = params.keySet();
            Iterator<String> iter = keys.iterator();
            while (iter.hasNext()){
                String key = iter.next();
                query.setParameter(key, params.get(key));
            }
        }
    }
    
    public <T> List<T> getAll(Class<T> entity) {
        List result = null;
        EntityManager em = getEM();
        try {
            result = em.createQuery("SELECT e FROM " + entity.getSimpleName() + " e").getResultList();
        } catch(Exception e) {
            Logger.getLogger(getClass().getName()).log(Level.SEVERE,"exception caught", e);
            throw new RuntimeException(e);
        } finally {
            em.close();
        }
        return result;
    }
    
    public Set<String> getSubcomponents(String component) {
        return getComponents().getSubcomponentsSet(component);
    }
    
    public List getDistinct(Class entity, Map<String, List<? extends Object>> params, String column) {
        List result = null;
        EntityManager em = getEM();
        StringBuffer finalQuery = new StringBuffer();
        try {
            
            finalQuery.append("SELECT DISTINCT x." + column + " ");
            finalQuery.append(createFromClause(entity, params, false, false, column, Operator.AND));
            
            Query q =  em.createQuery(finalQuery.toString());
            result =   q.getResultList();
            
        } catch(Exception e) {
            System.err.println(finalQuery);
            Logger.getLogger(getClass().getName()).log(Level.SEVERE,"exception caught", e);
            throw new RuntimeException(e);
        } finally {
            em.close();
        }
        return result;
    }
    
    public List getDistinct(Class entity, String column) {
        List result = null;
        EntityManager em = getEM();
        try {
            Query q =  em.createQuery("SELECT DISTINCT x." + column + " FROM " + entity.getSimpleName() + " x");
            result = q.getResultList();
        } catch(Exception e) {
            Logger.getLogger(getClass().getName()).log(Level.SEVERE,"exception caught", e);
            throw new RuntimeException(e);
        } finally {
            em.close();
        }
        return result;
    }
    
    private StringBuffer createWhereClause(Map<String, List<? extends Object>> params, Operator op, String var) {
        StringBuffer query = new StringBuffer();
        boolean insertOperator = false;
        for (Iterator<String> it = params.keySet().iterator(); it.hasNext();) {
            String key = it.next();
            List<? extends Object> l = params.get(key);
            if (insertOperator) {
                if (l.size() > 0) {
                    query.append(" " + op.name() + " ");
                }
                insertOperator = false;
            }
            StringBuffer sub = new StringBuffer();
            for(Iterator<? extends Object> it1 = l.iterator(); it1.hasNext(); ){
                Object val = it1.next();
                if (val instanceof String) {
                    sub.append(var + "." + key + " = '" + val + "'");
                } else {
                    sub.append(var + "." + key + " = " + val);
                }
                if (it1.hasNext()) {
                    sub.append(" OR ");
                }
            }
            if (sub.length() > 0) query.append(" (" + sub + ") ");
            if (it.hasNext()) {
                insertOperator = true;
            }
        }
        return query;
    }
    
    public StringBuffer createFromClause(Class entity, Map<String, List<? extends Object>> params, boolean duplicates, boolean original, String orderBy, Operator op) {
        StringBuffer finalQuery = new StringBuffer("FROM " + entity.getSimpleName() + " x");
        StringBuffer query = createWhereClause(params, op, "x");
        // for Exceptions - don't find in duplicates
        if (Exceptions.class.equals(entity) && original) {
            if (query.length() > 0) {
                query.append(" AND ");
            }
            query.append("x.duplicateof is null");
        }
        
        //            not working with mysql 4.0
        if (duplicates) {
            if (query.length() > 0) {
                query.append(" AND ");
            }
            query.append("SIZE(x.exceptionsCollection) > 0 ");
        }
        
        if (query.length() > 0) {
            finalQuery.append(" WHERE ");
            finalQuery.append(query);
        }
        
        if (orderBy != null) {
            finalQuery.append(" ORDER BY x." + orderBy);
        }
        return finalQuery;
    }
    
    public <T> List<T> find(Class<T> entity, Map<String, List<? extends Object>> params) {
        return find(entity, params, false, null);
    }
    public <T> List<T> find(Class<T> entity, Map<String, List<? extends Object>> params, boolean duplicates, String orderBy) {
        return find(entity, params, duplicates, true, orderBy, Operator.AND);
    }
    public <T> List<T> find(Class<T> entity, Map<String, List<? extends Object>> params, boolean duplicates, boolean originals, String orderBy) {
        return find(entity, params, duplicates, originals, orderBy, Operator.AND);
    }
    public <T> List<T> find(Class<T> entity, Map<String, List<? extends Object>> params, Operator out) {
        return find(entity, params, false, true, null, out);
    }
    public <T> List<T> find(Class<T> entity, Map<String, List<? extends Object>> params, boolean duplicates, boolean originals, String orderBy, Operator op) {
        List result = null;
        EntityManager em = getEM();
        try {
            StringBuffer select = new StringBuffer();
            select.append("SELECT x ");
            select.append(createFromClause(entity, params, duplicates, originals, orderBy, op));
            Query q =  em.createQuery(select.toString());
            result = q.getResultList();
        } catch(Exception e) {
            Logger.getLogger(getClass().getName()).log(Level.SEVERE,"exception caught", e);
            throw new RuntimeException(e);
        } finally {
            em.close();
        }
        return result;
    }
    
    private String createFromClauseForExceptions(Map<String, List<? extends Object>> params) {
        StringBuffer select = new StringBuffer();
        if (params == null || params.isEmpty()) {
                select.append("FROM Exceptions e ");
            } else {
                StringBuffer w1 = createWhereClause(params, Operator.AND, "e");
                StringBuffer w2 = createWhereClause(params, Operator.AND, "ex");
                if (w1.length() > 0) {
                    w1.append(" AND ");
                    w2.append(" AND ");
                }
                select.append("FROM Exceptions e ");
                select.append("WHERE EXISTS (SELECT ex FROM Exceptions ex WHERE " + w2 + "ex.duplicateof=e) ");
                select.append("OR (" + w1 + "e.duplicateof is null) ");
            }
        return select.toString();
    }
    public List<Exceptions> findExceptions(Map<String, List<? extends Object>> params, String orderBy) {
        List result = null;
        EntityManager em = getEM();
        try {
            StringBuffer select = new StringBuffer("SELECT e ");
            select.append(createFromClauseForExceptions(params));
            if (orderBy != null) {
                    select.append(" ORDER BY e." + orderBy);
                }
            Query q =  em.createQuery(select.toString());
            result = q.getResultList();
        } catch(Exception e) {
            Logger.getLogger(getClass().getName()).log(Level.SEVERE,"exception caught", e);
            throw new RuntimeException(e);
        } finally {
            em.close();
        }
        return result;
    }
    
    public Long countExceptions(Map<String, List<? extends Object>> params) {
        Long result = null;
        EntityManager em = getEM();
        try {
            StringBuffer select = new StringBuffer("SELECT COUNT(e) ");
            select.append(createFromClauseForExceptions(params));
            Query q =  em.createQuery(select.toString());
            result =  (Long) q.getSingleResult();
        } catch(Exception e) {
            Logger.getLogger(getClass().getName()).log(Level.SEVERE,"exception caught", e);
            throw new RuntimeException(e);
        } finally {
            em.close();
        }
        return result;
    }
    
    public List<Exceptions> findUnmapped() {
        List<Exceptions> result = null;
        EntityManager em = getEM();
        try {
            Query q =  em.createQuery("SELECT e FROM Exceptions e WHERE e.issuezillaid = 0 AND e.duplicateof is NULL");
            result = q.getResultList();
        } catch(Exception e) {
            Logger.getLogger(getClass().getName()).log(Level.SEVERE,"exception caught", e);
            throw new RuntimeException(e);
        } finally {
            em.close();
        }
        return result;
    }

    /**
     * @return list of candidates to be inserted into issuezilla
     * @param duplicatesNo minimal number of duplicates, that issue need
     * to have to become a candidate
     */ 
    public List<Exceptions> findExceptionCandidates(long duplicatesNo) {
        List<Exceptions> unmapped = findUnmapped();
        List<Exceptions> candidates = new ArrayList<Exceptions>();
        for (Exceptions unmap : unmapped) {
            if (unmap.getDuplicates() > duplicatesNo) candidates.add(unmap);
        }
        return candidates;
    }
    
    /**
     * @returns number of entities satisfying the condition
     * @param entity entity class to count
     * @param condition condition to satisfy you can reference the entity by "entity"
     *
     */
    public int count(Class entity, String condition){
        return getSingleResult("SELECT COUNT(entity) FROM " + entity.getSimpleName() + " entity WHERE " + condition);
    }
    
    public int count(Class entity) {
        return getSingleResult("SELECT COUNT(e) FROM " + entity.getSimpleName() + " e");
    }
    
    public int max(Class entity, String field) {
        EntityManager em = getEM();
        Query query = em.createQuery("SELECT MAX(e." + field + ") FROM " + entity.getSimpleName() + " e");
        Integer result = (Integer) query.getSingleResult();
        em.close();
        if (result == null) return 0;
        else return result;
    }
   
    /**
     * @returns number of distinct items in column satisfying the condition
     * @param entity entity class that should be counted
     * @param column column name that values are going to be counted
     * @param condition condition to satisfy you can reference the entity by "entity"
     *
     */
    public int countDistinct(Class entity, String column, String condition){
        return getSingleResult("SELECT COUNT(distinct entity."+column+") FROM " + entity.getSimpleName() + " entity WHERE " + condition);
    }
    
    private int getSingleResult(String query){
        EntityManager em = getEM();
        Query que = em.createQuery(query);
        Long result = (Long) que.getSingleResult();
        em.close();
        if (result == null) return 0;
        else return result.intValue();
    }
    
    public Long count(Class entity, Map<String, List<? extends Object>> params) {
        return count(entity, params, true, false);
    }
    public Long count(Class entity, Map<String, List<? extends Object>> params, boolean originals, boolean duplicatesFilter) {
        Long result = null;
        EntityManager em = getEM();
        StringBuffer finalQuery = new StringBuffer();
        try {
            
            finalQuery.append("SELECT COUNT(x) ");
            finalQuery.append(createFromClause(entity, params, duplicatesFilter, originals, null, Operator.AND));
            
            Query q =  em.createQuery(finalQuery.toString());
            result =  (Long) q.getSingleResult();
            
        } catch(Exception e) {
            System.err.println(finalQuery);
            Logger.getLogger(getClass().getName()).log(Level.SEVERE,"exception caught", e);
            throw new RuntimeException(e);
        } finally {
            em.close();
        }
        return result;
    }
    
    public <T> T getEntity(Class<T> entity, Object key) {
        Object result = null;
        EntityManager em = getEM();
        try {
            result =  em.find(entity, key);
        } catch(Exception e) {
            Logger.getLogger(getClass().getName()).log(Level.SEVERE,"exception caught", e);
            throw new RuntimeException(e);
        } finally {
            em.close();
        }
        return (T)result;
    }
    
    public final EntityManager createEntityManager() {
        return emf.createEntityManager();
    }
    
    private final EntityManager getEM() {
        return emf.createEntityManager();
    }
    
    public void persist(Object object) {
        EntityManager em = getEM();
        em.getTransaction().begin();
        em.persist(object);
        em.getTransaction().commit();
        em.close();
    }
    
    public void merge(Object object) {
        EntityManager em = getEM();
        em.getTransaction().begin();
        em.merge(object);
        em.getTransaction().commit();
        em.close();
    }
    
    public void remove(Object object) {
        EntityManager em = getEM();
        em.getTransaction().begin();
        em.remove(object);
        em.getTransaction().commit();
        em.close();
    }

    private void close() {
        emf.close();
    }

    public void updateComponents(Map<String, Object> params){
        String update = "UPDATE Exceptions e set e.component = :component, e.subcomponent = :subcomponent" +
                " WHERE e.component = :oldcomponent AND e.subcomponent = :oldsubcomponent";
        EntityManager em = getEM();
        em.getTransaction().begin();
        Query query = em.createQuery(update);
        setParams(query, params);
        query.executeUpdate();
        em.getTransaction().commit();
        if (em.isOpen()){
            em.close();
        }
    }
    
    public boolean issueIsFixed(Integer issueId) {
        Issue issue = getEntity(Issue.class, issueId);
        if (issue != null) {
            String status = issue.getIssueStatus();
            String resolution = issue.getResolution();
            if (("RESOLVED".equals(status) ||                          // NOI18N
                    "VERIFIED".equals(status) ||                      // NOI18N
                    "CLOSED".equals(status)) && !"DUPLICATE".equals(resolution))  {                       // NOI18N
                return true;             
            } 
        }
        return false;
    }
    
    public static Exceptions getOriginal(Exceptions exc) {
        Exceptions orig = exc.getDuplicateof();
        return orig == null ? exc : orig;
    }
    
}
