/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.scripting.php.dbginterface;

import java.io.IOException;
import java.net.ConnectException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.netbeans.api.debugger.ActionsManager;
import org.netbeans.api.debugger.DebuggerManager;
import org.netbeans.modules.scripting.php.dbginterface.api.DbgSourceMap;
import org.netbeans.modules.scripting.php.dbginterface.api.DebuggerStartException;
import org.netbeans.modules.scripting.php.dbginterface.api.VariableNode;
import org.netbeans.modules.scripting.php.dbginterface.breakpoints.BreakpointModel;
import org.netbeans.modules.scripting.php.dbginterface.models.CallStackModel;
import org.netbeans.modules.scripting.php.dbginterface.models.DebugFrame;
import org.netbeans.modules.scripting.php.dbginterface.models.ThreadsModel;
import org.netbeans.modules.scripting.php.dbginterface.models.Variable;
import org.netbeans.modules.scripting.php.dbginterface.models.VariablesModel;
import org.netbeans.modules.scripting.php.dbginterface.models.WatchesModel;
import org.netbeans.spi.debugger.ActionsProviderSupport;
import org.netbeans.spi.debugger.ContextProvider;
import org.netbeans.spi.debugger.DebuggerEngineProvider;
import org.netbeans.spi.viewmodel.TableModel;
import org.netbeans.spi.viewmodel.TreeModel;
import org.openide.ErrorManager;
import org.openide.loaders.DataObject;
import org.openide.util.RequestProcessor;

/**
 *
 * @author marcow
 */
public class DbgDebuggerImpl extends ActionsProviderSupport {
    public static final String BREAKPOINTS_VIEW_NAME = "BreakpointsView";
    public static final String CALLSTACK_VIEW_NAME = "CallStackView";
    public static final String LOCALS_VIEW_NAME = "LocalsView";
    public static final String SESSIONS_VIEW_NAME = "SessionsView";
    public static final String THREADS_VIEW_NAME = "ThreadsView";
    public static final String WATCHES_VIEW_NAME = "WatchesView";

    private static final int DBG_DEFAULT_PORT = 7869;
    
    private ContextProvider contextProvider;
    private ServerSocket server = null;
    private int port;
    private DbgServerLoop dbgServerLoop;
    private DbgServerHandler currentScriptContext;
    private Boolean finished = Boolean.FALSE;
    
    /** Creates a new instance of DbgDebuggerImpl */
    // XXX probably a ref back to the project/source file beeing debugged!
    public DbgDebuggerImpl(ContextProvider contextProvider) {
        this.contextProvider = contextProvider;
    }
    
    public void start() {
        createServerSocket();
    
       
        if (dbgServerLoop == null) {
            dbgServerLoop = new DbgServerLoop();
        }
        
        RequestProcessor.getDefault().post(dbgServerLoop);
    }

    
    public void stop() {
        synchronized (finished) {
            if (finished.booleanValue()) {
                return;
            }
            
            finished = Boolean.TRUE;
        }
        
        if (dbgServerLoop != null) {
            dbgServerLoop.setStop();
            dbgServerLoop.notifyWait();
            
            dbgServerLoop = null;
        }
        
        ((DbgEngineProvider)contextProvider.lookupFirst(null, DebuggerEngineProvider.class)).getDestructor().killEngine();
    }
    
    
   public void closeServerSocket() throws IOException {
        if (server != null) {
            stop();
            server = null;
        }
    }
    
    public int getPort() {
        return port;
    }

    public void waitRunning() throws DebuggerStartException {
        start();
    }

    public String getHost() {
        // XXX For now
        return "localhost";
    }

    public CallStackModel getCallStackModel() {
        return (CallStackModel)contextProvider.lookupFirst(CALLSTACK_VIEW_NAME, TreeModel.class);
    }

    public ThreadsModel getThreadsModel() {
        return (ThreadsModel)contextProvider.lookupFirst(THREADS_VIEW_NAME, TreeModel.class);
    }
    
    public VariablesModel getVariablesModel() {
        return (VariablesModel)contextProvider.lookupFirst(LOCALS_VIEW_NAME, TreeModel.class);
    }

    public WatchesModel getWatchesModel() {
        return (WatchesModel)contextProvider.lookupFirst(WATCHES_VIEW_NAME, TreeModel.class);
    }
    
    public BreakpointModel getBreakpointModel() {
        Iterator it = DebuggerManager.getDebuggerManager().lookup(BREAKPOINTS_VIEW_NAME, TableModel.class).iterator();

        while(it.hasNext()) {
            TableModel model = (TableModel)it.next();
            if (model instanceof BreakpointModel) {
                return (BreakpointModel) model;
            }
        }

        return null;
    }
    
    public VariableNode evaluateExpr(DebugFrame frame, String expr) {
        Context c = getCurrentScriptContext();
        
        if (c == null) {
            return null;
        }
        
        return c.evaluateExpr(frame, expr);
    }
    
    public DataObject getDataObject(String sourceFile) {
        DbgSourceMap sourceMap = (DbgSourceMap)contextProvider.lookupFirst(null, DbgSourceMap.class);
        
        if (sourceMap == null) {
            return null;
        }
        
        return sourceMap.mapToSourceFileDataObject(sourceFile);
    }
    
    ////////////////////////////////////////////////////////////////////////////
    ////////////////////////////// ActionProvider //////////////////////////////
    ////////////////////////////////////////////////////////////////////////////

    public void setEnableContActions(boolean s) {
        setEnabled(ActionsManager.ACTION_CONTINUE, s);
        setEnabled(ActionsManager.ACTION_STEP_INTO, s);
        setEnabled(ActionsManager.ACTION_STEP_OVER, s);
        setEnabled(ActionsManager.ACTION_STEP_OUT, s);
    }
    
    private static final Set<Object> ACTIONS = new HashSet<Object>();

    static {
        ACTIONS.add(ActionsManager.ACTION_KILL);
        ACTIONS.add(ActionsManager.ACTION_CONTINUE);
        ACTIONS.add(ActionsManager.ACTION_START);
        ACTIONS.add(ActionsManager.ACTION_STEP_INTO);
        ACTIONS.add(ActionsManager.ACTION_STEP_OVER);
        ACTIONS.add(ActionsManager.ACTION_STEP_OUT);
        // ACTIONS.add(ActionsManager.ACTION_RUN_TO_CURSOR);
    }

    public Set getActions() {
        return ACTIONS;
    }

    public void doAction(Object action) {
        System.err.println("mw DbgDebuggerImpl.doAction(" + action + ")");
        if (action == ActionsManager.ACTION_KILL) {
            stop();
        }
        else {
            DbgDebuggerImpl.Context context = getCurrentScriptContext();

            System.err.println("mw DbgDebuggerImpl.doAction() context= " + context);
            
            if(context == null) {
                return;
            }

            if (action == ActionsManager.ACTION_CONTINUE) {
                getVariablesModel().clearModel();
                getWatchesModel().setStackFrame(null);
                context.resume();
            }
            else if (action == ActionsManager.ACTION_STEP_INTO) {
                context.stepInto();
            }
            else if (action == ActionsManager.ACTION_STEP_OUT) {
                context.stepOut();
            }
            else if (action == ActionsManager.ACTION_STEP_OVER) {
                context.stepOver();
            }
            /*
            else if (action == ActionsManager.ACTION_RUN_TO_CURSOR) {
                try {
                    Node[] nodes = TopComponent.getRegistry().getActivatedNodes();

                    if (null == nodes || nodes.length == 0) {
                        return;
                    }
                    
                    DataObject doCookie = (DataObject)nodes[0].getCookie(DataObject.class);
                    String logicalPath = getApplicationPath(doCookie.getPrimaryFile());

                    Step step = createStepToCursor(logicalPath);
                    if(null != step) {
                        context.setCurrentStep(step);
                        context.resume();
                    }
                } catch(Exception e) {
                    Debug.debugNotify(e);
                }
            }
             */
        }
    }

    public Context[] getScriptContexts() {
        Set<DbgServerHandler> set = dbgServerLoop.getHandlerSet().keySet();
        
        return set.toArray(new Context[set.size()]);
    }
    
    public void closeScriptContext(Context context) {
        DbgServerHandler handler = (DbgServerHandler)context;
        
        handler.stop();
        dbgServerLoop.getHandlerSet().remove(handler);
        
        
        if (currentScriptContext == handler) {
            DbgServerHandler replacement = null;
            
            if (!dbgServerLoop.getHandlerSet().isEmpty()) {
                replacement = (DbgServerHandler)dbgServerLoop.getHandlerSet().keySet().toArray()[0];
                setCurrentScriptContext(replacement);
            }
            else {
                stop();
                
                return;
            }
        }
        else {
            getThreadsModel().setNeedsRefresh();
        }

        // Update threads model immediately because the thread view should update
        // visually even when application is runnning.
        getThreadsModel().refresh(false);
    }


    public Context getCurrentScriptContext() {
        return currentScriptContext;
    }

    public void setCurrentScriptContext(Context context) {
        if(currentScriptContext == context) {
            return;
        }

        if(currentScriptContext != null) {
            currentScriptContext.hideCurrentLine();
        }

        currentScriptContext = (DbgServerHandler)context;

        if(currentScriptContext != null) {
            currentScriptContext.handleCurrentLine();
            setEnabled(ActionsManager.ACTION_KILL, true);
        }

        getThreadsModel().setNeedsRefresh();
        getCallStackModel().setNeedsRefresh();
    }
    
    public DbgSourceMap getSourceMap() {
        return (DbgSourceMap)contextProvider.lookupFirst(null, DbgSourceMap.class);
    }
    
    public static abstract class Context {
        private DbgDebuggerImpl server;
        
        protected Context(DbgDebuggerImpl impl) {
            this.server = impl;
        }
        
        public DbgDebuggerImpl getServer() {
            return server;
        }
        
        public abstract boolean isSuspended();
        
        public abstract List<DebugFrame> getCallStack();
        
        public abstract Variable getScopeVariables(DebugFrame frame);

        public abstract void setVariableValue(DebugFrame frame, Variable v, Object value);
        
        public abstract Variable evaluateExpr(DebugFrame frame, String expr);
        public abstract void resume();
        
        public abstract void stepInto();
        
        public abstract void stepOut();
        
        public abstract void stepOver();
        
        public boolean isCurrent() {
            return server.getCurrentScriptContext() == this;
        }
    }
    
    private void createServerSocket() {
        if (server == null) {
            port = findFreePort();

            if (port == -1) {
                ErrorManager.getDefault().
                        log(ErrorManager.WARNING, "DbgServer.createServerSocket(): could not find free port!");
            
                return;
            }
        
            try {
                server = new ServerSocket(port);
                server.setSoTimeout(60000); // Just set this to a 1 minute timeout.
            }
            catch (IOException ioe) {
                ErrorManager.getDefault().
                        log(ErrorManager.WARNING, "DbgServer.createServerSocket(): could not create server socket!");
                ErrorManager.getDefault().
                        notify(ErrorManager.WARNING, ioe);
            
                stop();
            }
        }
    }
    

    private int findFreePort() {
        for (int port = DBG_DEFAULT_PORT; port < DBG_DEFAULT_PORT + 100; port++) {
            Socket testClient = null;
            
            try {
                testClient = new Socket("localhost", port); // NOI18N
            }
            catch (ConnectException ce) {
                return port;
            }
            catch (IOException ioe) {
                ioe.printStackTrace(System.err);
                // Something else went wrong, we don't care.
            }
            finally {
                if (testClient != null) {
                    // something listenend on that socket. It's not useful for us.
                    try {
                        testClient.close();
                    }
                    catch (IOException ioe) {
                        // We don't care here.
                    }
                }
            }
        }

        return -1;
    }
    
    
    private class DbgServerLoop implements Runnable {
        private boolean stop;
        private Map<DbgServerHandler, Boolean> handlerSet = new ConcurrentHashMap<DbgServerHandler, Boolean>();

        public DbgServerLoop() {
            stop = false;
        }
        
        public Map<DbgServerHandler, Boolean> getHandlerSet() {
            return handlerSet;
        }
        
        public synchronized void setStop() {
            stop = true;
            
            // If the server socket is blocked in the accept()!
            try {
                server.close();
            }
            catch (IOException ioe) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ioe);
            }
        }
        
        public synchronized void notifyWait() {
            notify();
        }
        
        public void run() {
            try {
                // The Server main loop
                while (!stop && !server.isClosed()) {
                    Socket handlerSocket = null;
                
                    try {
                        handlerSocket = server.accept();
                        System.err.println("DbgServerLoop.run().Accepted! : " + handlerSocket.toString());
                    }
                    catch (SocketTimeoutException ste) {
                        // That's OK, we wake up periodically
                    }
                    catch (SocketException se) {
                        // That probably means we are about to be stopped.
                        ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, se);
                        
                        return;
                    }
                    catch (IOException ioe) {
                        ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ioe);

                        return;
                    }
                    
                    if (handlerSocket != null) {
                        // XXX Needs some info about the project etc!
                        DbgServerHandler h = new DbgServerHandler(DbgDebuggerImpl.this, handlerSocket);
                        
                        handlerSet.put(h, Boolean.TRUE);
                        RequestProcessor.getDefault().post(h);
                        DbgDebuggerImpl.this.setCurrentScriptContext(h);
                    }
                }
            }
            catch (Exception ex) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ex);
            } finally {
                try {
                    for (DbgServerHandler h : handlerSet.keySet()) {
                        h.stop();
                    }
                    
                    server.close();
                }
                catch (IOException ioe) {
                    ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ioe);
                    
                    return;
                }
                
                //System.out.println("Socket loop finished.");
            }
        }
    }
}
