/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.scripting.php.dbginterface;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.netbeans.modules.scripting.php.dbginterface.models.DebugError;
import org.netbeans.modules.scripting.php.dbginterface.models.DebugFrame;
import org.netbeans.modules.scripting.php.dbginterface.models.Variable;
import org.openide.ErrorManager;
import org.openide.text.Line;

/**
 *
 * @author marcow
 */
public class DbgPacket {
    private static final int PACKET_HEADER_SIZE = 16;
    private static final int FRAME_HEADER_SIZE = 8;
    
    private int cmd;
    private int flags;
    private int size;
    private int currentRawId;
    private List<Frame> frames = new ArrayList<Frame>();
    
    /** Creates a new instance of DbgPacket */
    public DbgPacket(int cmd) {
        this.cmd = cmd;
    }
    
    public void addFrame(Frame frame) {
        frames.add(frame);
        size += frame.getSize();
    }
    
    public void setFlags(int flags) {
        this.flags = flags;
    }
    
    public void send(OutputStream out) throws IOException {
        System.err.println("mw DbgPacket.send(" + cmd + ")");
        byte[] header = new byte[PACKET_HEADER_SIZE];
        
        copyToByteArray(header, 0, DbgConstants.DBG_SYNC);
        copyToByteArray(header, 4, cmd);
        copyToByteArray(header, 8, flags);
        copyToByteArray(header, 12, size);
        
        out.write(header);
        out.flush();

        for (Frame f : frames) {
            f.send(out);
        }
    }
    
    public boolean isWaitAck() {
        return (flags & DbgConstants.DBGF_WAITACK) != 0;
    }
    
    public static DbgPacket receive(InputStream in) throws IOException {
        byte[] header = new byte[PACKET_HEADER_SIZE];
        
        if (in.read(header) < PACKET_HEADER_SIZE) {
            return null;
        }
            
        if (intFromByteArray(header, 0) != DbgConstants.DBG_SYNC) {
            return null;
        }
            
        DbgPacket ret = new DbgPacket(intFromByteArray(header, 4));
            
        ret.flags = intFromByteArray(header, 8);
        ret.size = intFromByteArray(header, 12);
        ret.frames = Frame.receiveFrames(in, ret.size);

        return ret;
    }
    
    public String toString() {
        String ret = "DbgPacket([" + cmd + ", " + flags + ", " + size + "]";
        
        if (frames != null) {
            for (Frame f : frames) {
                ret += "\n\t" + f.toString();
            }
        }
        
        return ret + ")";
    }
    
    public Frame getFirstFrame() {
        if (frames == null || frames.size() == 0) {
            return null;
        }
        
        return frames.get(0);
    }

    public void handle(DbgServerHandler handler) {
        System.err.println("mw packet returned= " + this);
        
        if (cmd == DbgConstants.DBGC_STARTUP && flags == (DbgConstants.DBGF_STARTED | DbgConstants.DBGF_WAITACK)) {
            handler.setBreakpoints();
            handler.resume();
            
            return;
        }

        if (cmd == DbgConstants.DBGF_FINISHED) {
            handler.finish();
            
            return;
        }
        
        if (frames != null) {
            for (Frame f : frames) {
                f.handle(handler);
            }
        }
        
        if (cmd == DbgConstants.DBGC_BREAKPOINT || cmd == DbgConstants.DBGC_STEPINTO_DONE ||
                cmd == DbgConstants.DBGC_STEPOUT_DONE || cmd == DbgConstants.DBGC_STEPOVER_DONE) {
            updateCallStack(handler);
            handler.setSuspended(true);
        }
        
        if (cmd == DbgConstants.DBGC_ERROR) {
            handleError(handler);
        }
    }
    
    public List<Variable> getEvalValue() {
        if (frames == null) {
            return null;
        }
        
        for (Frame f : frames) {
            if (f instanceof FrameEval) {
                return ((FrameEval)f).getEvalValue();
            }
        }
        
        return null;
    }
    
    protected int getNewRawId() {
        return ++currentRawId;
    }
    
    private static void copyToByteArray(byte[] target, int offset, int value) {
        target[offset + 0] = (byte)((value >> 24) & 0xff);
        target[offset + 1] = (byte)((value >> 16) & 0xff);
        target[offset + 2] = (byte)((value >>  8) & 0xff);
        target[offset + 3] = (byte)((value >>  0) & 0xff);   
    }
    
    private static int intFromByteArray(byte[] target, int offset) {
        int ret = 0;
        
        ret |= ((int)target[offset + 0] & 0xff) << 24;
        ret |= ((int)target[offset + 1] & 0xff) << 16;
        ret |= ((int)target[offset + 2] & 0xff) <<  8;
        ret |= ((int)target[offset + 3] & 0xff) <<  0;
        
        return ret;
    }
    
    private void updateCallStack(DbgServerHandler handler) {
        List<DebugFrame> dFrames = new ArrayList<DebugFrame>();
        boolean firstFrame = true;
        
        for (Frame frame : frames) {
            if (frame instanceof FrameStack) {
                FrameStack sFrame = (FrameStack)frame;
                String desc = sFrame.getDesc();
                String method = "";
                int idx;
                
                if ((idx = desc.lastIndexOf("::")) >= 0) {
                    method = desc.substring(idx + 2);
                }
                else {
                    method = desc;
                }
                
                System.err.println("mw DbgPacket.updateCallStack() method= " + method);
                
                Line line = findLine(sFrame, handler);
                
                dFrames.add(new DebugFrame(handler, line, method, sFrame.getScope()));
        
                if (firstFrame) {
                    handler.setCurrentLine(line);
                    handler.handleCurrentLine();
                    firstFrame = false;
                }
            }
        }
        
        handler.setCallStack(dFrames);
    }
    
    private void handleError(DbgServerHandler handler) {
        Line line = null;
        boolean canContinue = true;
        boolean canResume = true;
        
        for (Frame f : frames) {
            if (f instanceof FrameStack) {
                line = findLine((FrameStack)f, handler);
                
                break;
            }
        }
        
        for (Frame f : frames) {
            if (f instanceof FrameError) {
                FrameError eFrame = (FrameError)f;
                int type = eFrame.getType();
                String desc = eFrame.getDesc();
                
                handler.addError(new DebugError(line, type, desc));

                if (type == DbgConstants.ERR_CORE_ERROR || type == DbgConstants.ERR_ERROR ||
                        type == DbgConstants.ERR_PARSE) {
                    canContinue = false;
                    canResume = false;
                }
                else if (type == DbgConstants.ERR_CORE_WARNING) {
                    canResume = false;
                }
            }
        }
        
        if (canResume) {
            handler.resume();
            
            return;
        }
        
        if (canContinue) {
            handler.setSuspended(true);
        }
    }
    
    private Line findLine(FrameStack sFrame, DbgServerHandler handler) {
        int modNo = sFrame.getModNo();
        
        String sourceFileName = handler.getSourceFileInfo(modNo);
        
        if (sourceFileName == null) {
            String s = sFrame.getDesc();
            int idx;
                
            if ((idx = s.lastIndexOf("::")) >= 0) {
                sourceFileName = s.substring(0, idx);
            }
            else {
                handler.updateSourceTree();
                sourceFileName = handler.getSourceFileInfo(modNo);
            }
        }
        
        System.err.println("mw DbgPacket.updateCallStack() sourceFileName= " + sourceFileName);
            
        return Utils.getLine(handler.getSourceMap().mapToSourceFileDataObject(sourceFileName),
                    sFrame.getLineNo());
    }
    
    public static class Frame {
        static private Map<Integer, Frame> map;
        
        protected byte[] buffer;
        protected int type;           // It's called framename but it's more an type
        
        private Frame() {
        }
        
        public int getSize() {
            int ret = FRAME_HEADER_SIZE;
            
            if (buffer != null) {
                ret += buffer.length;
            }
            
            return ret;
        }
        
        public void send(OutputStream out) throws IOException {
            byte[] header = new byte[FRAME_HEADER_SIZE];
        
            copyToByteArray(header, 0, type);
            copyToByteArray(header, 4, (buffer != null ? buffer.length : 0));

            try {
                out.write(header);
                
                if (buffer != null) {
                    out.write(buffer);
                }
                
                out.flush();
            }
            catch (Exception e) {
                ErrorManager.getDefault().notify(ErrorManager.WARNING, e);
            }
        }

        public static List<Frame> receiveFrames(InputStream in, int size) throws IOException {
            if (size == 0) {
                return null;
            }
            
            if (map == null) {
                initMap();
            }
            
            byte[] recBuffer = new byte[size];
            int offset = 0;
            
            if (in.read(recBuffer) < size) {
                // We did not get all the promised bytes!
                return null;
            }
            
            List<Frame> ret = new ArrayList<Frame>();
            List<FrameRawdata> raw = new ArrayList<FrameRawdata>();
            
            while (offset < size) {
                // Frame header
                int recType = intFromByteArray(recBuffer, offset);
                int recFrameSize = intFromByteArray(recBuffer, offset + 4);
                offset += 8;
                
                Frame f = map.get(recType).receive(recBuffer, offset, recFrameSize);
                
                if (recType == DbgConstants.FRAME_RAWDATA) {
                    raw.add((FrameRawdata)f);
                }
                else {
                    ret.add(f);
                }
                
                offset += recFrameSize;
            }
            
            for (Frame f : ret) {
                f.resolveRawRefs(raw);
            }
            
            assert raw.isEmpty();
            
            return ret;
        }
                
        // This had to be overridden by the different Frame types. 
        protected Frame receive(byte[] buffer, int offset, int size) {
            return null;
        }
        
        protected void resolveRawRefs(List<FrameRawdata> list) {
        }
        
        protected String bufferToString() {
            if (buffer == null) {
                return "";
            }
            
            String ret = "[";
            for (int i = 0; i < buffer.length; i++) {
                ret += "" + buffer[i] + ", ";
            }
            
            ret += "]";
            
            return ret;
        }
        
        protected void handle(DbgServerHandler handler) {
        }
        
        private static void initMap() {
            map = new HashMap<Integer, Frame>();
            
            map.put(DbgConstants.FRAME_STACK, new FrameStack());
            map.put(DbgConstants.FRAME_SOURCE, new FrameSource());
            map.put(DbgConstants.FRAME_SRC_TREE, new FrameSrcTree());
            map.put(DbgConstants.FRAME_RAWDATA, new FrameRawdata());
            map.put(DbgConstants.FRAME_ERROR, new FrameError());
            map.put(DbgConstants.FRAME_EVAL, new FrameEval());
            map.put(DbgConstants.FRAME_BPS, new FrameBps());
            map.put(DbgConstants.FRAME_BPL, new FrameBpl());
            map.put(DbgConstants.FRAME_VER, new FrameVer());
            map.put(DbgConstants.FRAME_SID, new FrameSid());
            map.put(DbgConstants.FRAME_SRCLINESINFO, new FrameSrcLinesInfo());
            map.put(DbgConstants.FRAME_SRCCTXINFO, new FrameSrcCtxInfo());
            map.put(DbgConstants.FRAME_LOG, new FrameLog());
            map.put(DbgConstants.FRAME_PROF, new FrameProf());
            map.put(DbgConstants.FRAME_PROF_C, new FrameProfC());
            map.put(DbgConstants.FRAME_SET_OPT, new FrameSetOpt());
        }
    }
    
    public static class FrameVer extends Frame {
        private int major;
        private int minor;
        private int iDesc;
        private FrameRawdata desc;
        
        public FrameVer() {
            type = DbgConstants.FRAME_VER;
        }
                
        public String toString() {
            return "DbgPacket.FrameVer(" + major + "." + minor + ", iDesc= " + iDesc + ", desc= " + desc + ")"; 
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            // assert size == 12;
            System.err.println("mw DbgPacket.FrameVer.receive() size= " + size);
            
            FrameVer ret = new FrameVer();
            
            ret.major = intFromByteArray(buffer, offset);
            ret.minor = intFromByteArray(buffer, offset + 4);
            ret.iDesc = intFromByteArray(buffer, offset + 8);

            return ret;
        }

        protected void resolveRawRefs(List<FrameRawdata> list) {
            Iterator<FrameRawdata> it = list.iterator();
            
            while (it.hasNext()) {
                FrameRawdata f = it.next();
                
                if (f.rawid == iDesc) {
                    desc = f;
                    it.remove();
                    
                    break;
                }
            }
        }
    }

    public static class FrameSid extends Frame {
        private int iSid;
        private FrameRawdata sid;
        private int sesstype;
                
        protected FrameSid() {
            type = DbgConstants.FRAME_SID;
        }
        
        public String toString() {
            return "DbgPacket.FrameSid(iSid= " + iSid + ", sesstype= " + sesstype + ", sid= " + sid + ")";
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 8;
            
            FrameSid ret = new FrameSid();
            
            ret.iSid = intFromByteArray(buffer, offset);
            ret.sesstype = intFromByteArray(buffer, offset + 4);

            return ret;
        }
        
        protected void resolveRawRefs(List<FrameRawdata> list) {
            Iterator<FrameRawdata> it = list.iterator();
            
            while (it.hasNext()) {
                FrameRawdata f = it.next();
                
                if (f.rawid == iSid) {
                    sid = f;
                    it.remove();
                    
                    break;
                }
            }
        }
        
        protected void handle(DbgServerHandler handler) {
            handler.setSessionId(sid.getString());
        }
    }

    public static class FrameStack extends Frame {
        private int lineNo;
        private int modNo;
        private int scope;
        private int iDesc;
        private FrameRawdata desc;
        
        public FrameStack() {
            type = DbgConstants.FRAME_STACK;
        }
        
        public String toString() {
            return "DbgPacket.FrameStack(lineNo= " + lineNo + ", modNo= " + modNo + ", scope= " + scope +
                    ", iDesc= " + iDesc + ", desc= " + desc + ")";
        }
        
        protected String getDesc() {
            return desc.getString();
        }
        
        protected int getLineNo() {
            return lineNo;
        }
        
        protected int getScope() {
            return scope;
        }
        
        protected int getModNo() {
            return modNo;
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 16;
         
            FrameStack ret = new FrameStack();
            
            ret.lineNo = intFromByteArray(buffer, offset);
            ret.modNo = intFromByteArray(buffer, offset + 4);
            ret.scope = intFromByteArray(buffer, offset + 8);
            ret.iDesc = intFromByteArray(buffer, offset + 12);
            
            return ret;
        }
        
        protected void resolveRawRefs(List<FrameRawdata> list) {
            Iterator<FrameRawdata> it = list.iterator();
            
            while (it.hasNext()) {
                FrameRawdata f = it.next();
                
                if (f.rawid == iDesc) {
                    desc = f;
                    it.remove();
                    
                    break;
                }
            }
        }
    }
    
    public static class FrameSource extends Frame {
        private int modNo;
        private int filePos;
        private int error;
        private int fullSize;
        private int iModName;
        private FrameRawdata modName;
        private int iText;
        private FrameRawdata text;
        
        protected FrameSource() {
            type = DbgConstants.FRAME_SOURCE;
        }
        
        public FrameSource(int _modNo, int _filePos) {
            this();
            buffer = new byte[8];
            copyToByteArray(buffer, 0, _modNo);
            copyToByteArray(buffer, 4, _filePos);
        }
        
        public String toString() {
            return "DbgPacket.FrameSource(modNo= " + modNo + ", filePos= " + filePos +
                    ", error= " + error + ", fullSize= " + fullSize + ", modName= " + modName +
                    ", text= " + text + ")" +
                    "\n\t\t" + bufferToString(); 

        }
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 24;
            
            FrameSource ret = new FrameSource();
            
            ret.modNo = intFromByteArray(buffer, offset);
            ret.filePos = intFromByteArray(buffer, offset + 4);
            ret.error = intFromByteArray(buffer, offset + 8);
            ret.fullSize = intFromByteArray(buffer, offset + 12);
            ret.iModName = intFromByteArray(buffer, offset + 16);
            ret.iText = intFromByteArray(buffer, offset + 20);
            
            return ret;
        }

        protected void resolveRawRefs(List<FrameRawdata> list) {
            Iterator<FrameRawdata> it = list.iterator();
            
            while (it.hasNext()) {
                FrameRawdata f = it.next();
                
                if (f.rawid == iModName) {
                    modName = f;
                    it.remove();
                }
                if (f.rawid == iText) {
                    text = f;
                    it.remove();
                }
            }
        }

    }
    
    public static class FrameSrcTree extends Frame {
        private int parentModNo;
        private int parentLineNo;
        private int modNo;
        private int iModName;
        private FrameRawdata modName;
        
        public FrameSrcTree() {
            type = DbgConstants.FRAME_SRC_TREE;
        }
        
        public String toString() {
            return "DbgPacket.FrameSrcTree(parentModNo= " + parentModNo + ", parentLineNo= " + parentLineNo +
                    ", modNo= " + modNo + ", modName= " + modName + ")"; 
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 16;
            
            FrameSrcTree ret = new FrameSrcTree();
            
            ret.parentModNo = intFromByteArray(buffer, offset);
            ret.parentLineNo = intFromByteArray(buffer, offset + 4);
            ret.modNo = intFromByteArray(buffer, offset + 8);
            ret.iModName = intFromByteArray(buffer, offset + 12);
            
            return ret;
        }
    
        protected void resolveRawRefs(List<FrameRawdata> list) {
            Iterator<FrameRawdata> it = list.iterator();
            
            while (it.hasNext()) {
                FrameRawdata f = it.next();
                
                if (f.rawid == iModName) {
                    modName = f;
                    it.remove();
                    
                    break;
                }
            }
        }
        
        protected void handle(DbgServerHandler handler) {
            handler.addSourceFileInfo(modNo, modName.getString());
        }
    }
   
     public static class FrameRawdata extends Frame {
        private int rawid;
        private int dataSize;
        private byte[] data;
        
        protected FrameRawdata() {
            type = DbgConstants.FRAME_RAWDATA;
        }
        
        public FrameRawdata(DbgPacket packet, String str) {
            this();
            data = new byte[0];
            
            try {
                data = str.getBytes("ISO8859_1");
            }
            catch (UnsupportedEncodingException uee) {
                // Should not happen
            }
            
            rawid = packet.getNewRawId();
            buffer = new byte[8 + data.length + 1]; // '\0'
            copyToByteArray(buffer, 0, rawid);
            copyToByteArray(buffer, 4, data.length);
            int i;
            
            for (i = 0; i < data.length; i++) {
                buffer[i + 8] = data[i];
            }
            
            buffer[i + 8] = 0;
        }
        
        public int getRawId() {
            return rawid;
        }
        
        public String toString() {
            return "DbgPacket.FrameRawdata(rawid= " + rawid + ", dataSize= " + dataSize + ", data= " + getString() + ")";
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size >= 8;
            
            FrameRawdata ret = new FrameRawdata();
            
            ret.rawid = intFromByteArray(buffer, offset);
            ret.dataSize = intFromByteArray(buffer, offset + 4);
            
            ret.data = new byte[ret.dataSize - 1]; // '\0'
            
            for (int i = 0; i < ret.dataSize - 1; i++) {
                ret.data[i] = buffer[offset + 8 + i];
            }
            
            return ret;
        }
        
        protected String getString() {
            try {
                return new String(data, "ISO8859_1");
            }
            catch (UnsupportedEncodingException uee) {
                // Should not happen
            }
            
            return "";
        }
    }
     
    public static class FrameError extends Frame {
        private int errorType;
        private int iMessage;
        private FrameRawdata message;
        
        protected FrameError() {
            type = DbgConstants.FRAME_ERROR;
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 8;
            
            FrameError ret = new FrameError();
            
            ret.errorType = intFromByteArray(buffer, offset);
            ret.iMessage = intFromByteArray(buffer, offset + 4);
            
            return ret;
        }

        protected void resolveRawRefs(List<FrameRawdata> list) {
            Iterator<FrameRawdata> it = list.iterator();
            
            while (it.hasNext()) {
                FrameRawdata f = it.next();
                
                if (f.rawid == iMessage) {
                    message = f;
                    it.remove();
                    
                    break;
                }
            }
        }
        
        protected int getType() {
            return errorType;
        }
        
        protected String getDesc() {
            return message.getString();
        }
    }    

    public static class FrameEval extends Frame {
        private int iStr;
        private FrameRawdata str;
        private int iResult;
        private FrameRawdata result;
        private int iError;
        private FrameRawdata error;
        private List<Variable> vars;
        
        protected FrameEval() {
            type = DbgConstants.FRAME_EVAL;
        }
        
        public FrameEval(int _iStr, int _scope) {
            this();
            buffer = new byte[8];
            copyToByteArray(buffer, 0, _iStr);
            copyToByteArray(buffer, 4, _scope);
        }
        
        protected List <Variable> getEvalValue() {
            if (vars != null) {
                return vars;
            }
            
            if (result != null && result.data != null) {
                vars = new DbgEvalValues(result.getString(), "").getVars();
            }
 
            return vars;
        }
        
        public String toString() {
            return "DbgPacket.FrameEval(str= " + str + ", result= " + result +
                    ", error= " + error + ")" +
                    "\n\t\t" + getEvalValue() + 
                    "\n\t\t" + bufferToString(); 

        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            // assert size == 12;
            System.err.println("mw DbgPacket.FrameEval.receive() size= " + size);
            FrameEval ret = new FrameEval();
            
            ret.iStr = intFromByteArray(buffer, offset);
            ret.iResult = intFromByteArray(buffer, offset + 4);
            ret.iError = intFromByteArray(buffer, offset + 8);
            
            return ret;
        }

        protected void resolveRawRefs(List<FrameRawdata> list) {
            Iterator<FrameRawdata> it = list.iterator();
            
            while (it.hasNext()) {
                FrameRawdata f = it.next();
                
                if (f.rawid == iStr) {
                    str = f;
                    it.remove();
                }
                if (f.rawid == iResult) {
                    result = f;
                    it.remove();
                }
                if (f.rawid == iError) {
                    error = f;
                    it.remove();
                }
            }
        }

    }

    public static class FrameBpl extends Frame {
        private int modNo;
        private int lineNo;
        private int iModName;
        private FrameRawdata modName;
        private int state;
        private int isTemp;
        private int hitCount;
        private int skipHits;
        private int iCondition;
        private FrameRawdata condition;
        private int bpNo;
        private int isUnderHit;
        
        protected FrameBpl() {
            type = DbgConstants.FRAME_BPL;
        }
        
        public FrameBpl(int _bpNo) {
            this();
            buffer = new byte[4];
            copyToByteArray(buffer, 0, _bpNo);
        }

        public String toString() {
            return "DbgPacket.FrameBpl(state= " + state + "  isTemp= " + isTemp +
                    "  hitCount= " + hitCount + "  skipHits= " + skipHits + "  condition= " + condition +
                    "  bpNo= " + bpNo + "  isUnderHit= " + isUnderHit + ")"; 
        }
        
        public int getBpNo() {
            return bpNo;
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 40;
            
            FrameBpl ret = new FrameBpl();
            
            ret.modNo = intFromByteArray(buffer, offset);
            ret.lineNo = intFromByteArray(buffer, offset + 4);
            ret.iModName = intFromByteArray(buffer, offset + 8);
            ret.state = intFromByteArray(buffer, offset + 12);
            ret.isTemp = intFromByteArray(buffer, offset + 16);
            ret.hitCount = intFromByteArray(buffer, offset + 20);
            ret.skipHits = intFromByteArray(buffer, offset + 24);
            ret.iCondition = intFromByteArray(buffer, offset + 28);
            ret.bpNo = intFromByteArray(buffer, offset + 32);
            ret.isUnderHit = intFromByteArray(buffer, offset + 36);
            
            return ret;
        }

        protected void resolveRawRefs(List<FrameRawdata> list) {
            Iterator<FrameRawdata> it = list.iterator();
            
            while (it.hasNext()) {
                FrameRawdata f = it.next();
                
                if (f.rawid == iModName) {
                    modName = f;
                    it.remove();
                }
                if (f.rawid == iCondition) {
                    condition = f;
                    it.remove();
                }
            }
        }
    }    

    public static class FrameBps extends Frame {
        protected FrameBps() {
            type = DbgConstants.FRAME_BPS;
        }
        
        public FrameBps(int bpNo, int lineNo, int modNo, int iModName, int iCondition, int skipHits, int isTemp, int state) {
            // FrameBpl response layout!!
            this();
            buffer = new byte[40];
            copyToByteArray(buffer, 0, modNo);
            copyToByteArray(buffer, 4, lineNo);
            copyToByteArray(buffer, 8, iModName);
            copyToByteArray(buffer, 12, state);
            copyToByteArray(buffer, 16, isTemp);
            // hitCount
            copyToByteArray(buffer, 24, skipHits);
            copyToByteArray(buffer, 28, iCondition);
            copyToByteArray(buffer, 32, bpNo);
            // isUnderHit
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 0;
            
            return new FrameBps();
        }
    }
    
    public static class FrameSrcLinesInfo extends Frame {
        private int modNo;
        private int startLineNo;
        private int linesCount;
        private int ctxId;
        
        protected FrameSrcLinesInfo() {
            type = DbgConstants.FRAME_SRCLINESINFO;
        }
        
        public FrameSrcLinesInfo(int _modNo) {
            this();
            buffer = new byte[4];
            copyToByteArray(buffer, 0, _modNo);
        }
        
        public String toString() {
            return "DbgPacket.FrameSrcLinesInfo(modNo= " + modNo + ", startLineNo= " + startLineNo +
                    ", linesCount= " + linesCount + ", ctxId= " + ctxId + ")"; 
        }

        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 16;
            
            FrameSrcLinesInfo ret = new FrameSrcLinesInfo();
            
            ret.modNo = intFromByteArray(buffer, offset);
            ret.startLineNo = intFromByteArray(buffer, offset + 4);
            ret.linesCount = intFromByteArray(buffer, offset + 8);
            ret.ctxId = intFromByteArray(buffer, offset + 12);
            
            return ret;
        }
    }
    
    public static class FrameSrcCtxInfo extends Frame {
        private int modNo;
        private int ctxId;
        private int iFunctionName;
        private FrameRawdata funtionName;
        
        protected FrameSrcCtxInfo() {
            type = DbgConstants.FRAME_SRCCTXINFO;
        }
        
        public FrameSrcCtxInfo(int _modNo) {
            this();
            buffer = new byte[4];
            copyToByteArray(buffer, 0, _modNo);
        }
        
        public String toString() {
            return "DbgPacket.FrameSrcCtxInfo(modNo= " + modNo + ", ctxId= " + ctxId +
                    ", funtionName= " + funtionName + ")"; 
        }

        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 12;
            System.err.println("mw DbgPacket.FrameSrcCtxInfo.receive() size= " + size);
            
            FrameSrcCtxInfo ret = new FrameSrcCtxInfo();
            
            ret.modNo = intFromByteArray(buffer, offset);
            ret.ctxId = intFromByteArray(buffer, offset + 4);
            ret.iFunctionName = intFromByteArray(buffer, offset + 8);
            
            return ret;
        }

        protected void resolveRawRefs(List<FrameRawdata> list) {
            Iterator<FrameRawdata> it = list.iterator();
            
            while (it.hasNext()) {
                FrameRawdata f = it.next();
                
                if (f.rawid == iFunctionName) {
                    funtionName = f;
                    it.remove();
                    
                    break;
                }
            }
        }

    }
    
    public static class FrameLog extends Frame {
        private int iLog;
        private FrameRawdata log;
        private int logType;
        private int modNo;
        private int lineNo;
        private int iModName;
        private FrameRawdata modName;
        private int extInfo;
        
        protected FrameLog() {
            type = DbgConstants.FRAME_LOG;
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 24;
            
            FrameLog ret = new FrameLog();
            
            ret.iLog = intFromByteArray(buffer, offset);
            ret.logType = intFromByteArray(buffer, offset + 4);
            ret.modNo = intFromByteArray(buffer, offset + 8);
            ret.lineNo = intFromByteArray(buffer, offset + 12);
            ret.iModName = intFromByteArray(buffer, offset + 16);
            ret.extInfo = intFromByteArray(buffer, offset + 20);
            
            return ret;
        }

        protected void resolveRawRefs(List<FrameRawdata> list) {
            Iterator<FrameRawdata> it = list.iterator();
            
            while (it.hasNext()) {
                FrameRawdata f = it.next();
                
                if (f.rawid == iLog) {
                    log = f;
                    it.remove();
                }
                if (f.rawid == iModName) {
                    modName = f;
                    it.remove();
                }
            }
        }

    }

    public static class FrameProf extends Frame {
        private int modNo;
        private int lineNo;
        private int hitCount;
        private int tm_min_lo;
        private int tm_min_hi;
        private int tm_max_lo;
        private int tm_max_hi;
        private int tm_sum_lo;
        private int tm_sum_hi;
        
        protected FrameProf() {
            type = DbgConstants.FRAME_PROF;
        }
        
        public FrameProf(int _modNo) {
            this();
            buffer = new byte[4];
            copyToByteArray(buffer, 0, _modNo);
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 36;
            
            FrameProf ret = new FrameProf();
            
            ret.modNo = intFromByteArray(buffer, offset);
            ret.lineNo = intFromByteArray(buffer, offset + 4);
            ret.hitCount = intFromByteArray(buffer, offset + 8);
            ret.tm_min_lo = intFromByteArray(buffer, offset + 12);
            ret.tm_min_hi = intFromByteArray(buffer, offset + 16);
            ret.tm_max_lo = intFromByteArray(buffer, offset + 20);
            ret.tm_max_hi = intFromByteArray(buffer, offset + 20);
            ret.tm_sum_lo = intFromByteArray(buffer, offset + 20);
            ret.tm_sum_hi = intFromByteArray(buffer, offset + 20);
            
            return ret;
        }
    }

    public static class FrameProfC extends Frame {
        private int tm_freq_lo;
        private int tm_freq_hi;
        private int tm_diff_min;
        private int tm_diff_max;
        private int tm_diff_m;
        
        protected FrameProfC() {
            type = DbgConstants.FRAME_PROF_C;
        }
        
        public FrameProfC(int testLoops) {
            this();
            buffer = new byte[4];
            copyToByteArray(buffer, 0, testLoops);
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 20;
            
            FrameProfC ret = new FrameProfC();
            
            ret.tm_freq_lo = intFromByteArray(buffer, offset);
            ret.tm_freq_hi = intFromByteArray(buffer, offset + 4);
            ret.tm_diff_min = intFromByteArray(buffer, offset + 8);
            ret.tm_diff_max = intFromByteArray(buffer, offset + 12);
            ret.tm_diff_m = intFromByteArray(buffer, offset + 16);
            
            return ret;
        }
    }

    public static class FrameSetOpt extends Frame {
        protected FrameSetOpt() {
            type = DbgConstants.FRAME_SET_OPT;
        }
        
        public FrameSetOpt(int opt) {
            this();
            buffer = new byte[4];
            copyToByteArray(buffer, 0, opt);
        }
        
        protected Frame receive(byte[] buffer, int offset, int size) {
            assert size == 0;
        
            return new FrameSetOpt();
        }
    }

}
