/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */
package org.netbeans.modules.visualweb.insync.jsf;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.WeakHashMap;

import javax.swing.Timer;
import javax.swing.text.Document;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;

import org.netbeans.api.xml.parsers.DocumentInputSource;
import org.netbeans.modules.visualweb.api.designer.markup.MarkupService;
import org.netbeans.modules.visualweb.insync.ParserAnnotation;
import org.netbeans.modules.visualweb.insync.markup.MarkupUnit;
import org.netbeans.modules.visualweb.insync.models.FacesModel;
import org.openide.ErrorManager;
import org.openide.cookies.EditorCookie;
import org.openide.cookies.LineCookie;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.text.Annotation;
import org.openide.text.Line;
import org.xml.sax.EntityResolver;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

// TODO Move it to insync, it doesn't have to do anything with designer,
// (the unnecessary usage of webform was removed).
// The parsing should be controlled solely by insync.
/**
 * The source monitor watches the source tab for a given
 * file, and when the tab is opened, attaches document listeners
 * which tracks source changes. Shortly after a source edit,
 * a parse is attempted, and any errors found are added to the
 * document as source annotations.
 * <p>
 *
 * @author Tor Norbye
 */
public class SourceMonitor implements javax.swing.event.DocumentListener, org.xml.sax.ErrorHandler {

    /** Maps <code>FacesModel</code> to <code>SourceMonitor</code>, keys are weak. */
    private static final Map sourceMonitors = new WeakHashMap();
    private static final Object LOCK_SOURCE_MONITORS = new Object();

    private WeakReference<FacesModel> facesModelReference;
    private Document document;
    private Map annotations;
    private Timer runTimer;

    /**
     * Attach a source monitor to the given data object.
     */
    private SourceMonitor(FacesModel facesModel) {
        this.facesModelReference = new WeakReference<FacesModel>(facesModel);
    }


    public static SourceMonitor getSourceMonitorForFacesModel(FacesModel facesModel) {
        synchronized (LOCK_SOURCE_MONITORS) {
            SourceMonitor sourceMonitor = (SourceMonitor)sourceMonitors.get(facesModel);
            if(sourceMonitor == null) {
                sourceMonitor = new SourceMonitor(facesModel);
                sourceMonitors.put(facesModel, sourceMonitor);
            }
            return sourceMonitor;
        }
    }


    // Update the tree occasionally
    private void docChanged() {
        // Stop our current timer; the previous node has not
        // yet been scanned; too brief an interval
        if (runTimer != null) {
            runTimer.stop();
            runTimer = null;
        }

        int delay = 4000; // ms

        if ((annotations != null) && (annotations.size() > 0)) {
            // Use a shorter parse delay when the source is in
            // error state. Thus if the user fixes the problem there is
            // more immediate feedback that things are good...
            delay = 2000;
        }

        runTimer =
            new Timer(delay,
                new ActionListener() {
                    public void actionPerformed(ActionEvent evt) {
                        runTimer = null;
                        reparse(false);
                    }
                });
        runTimer.setRepeats(false);
        runTimer.setCoalesce(true);
        runTimer.start();
    }

    /**
     * Reparse the current source file, and update the annotations set
     */
    private void reparse(boolean fullSync) {
        // Clear out old annotations
        detachAnnotations();
        
        FacesModel facesModel = facesModelReference.get();
        if (facesModel == null) {
            return;
        }

        annotations = new HashMap(20);

        if (fullSync) {
            facesModel.sync();
            MarkupUnit unit = facesModel.getMarkupUnit();
            addErrors(unit.getErrors());
        } else {
            org.xml.sax.InputSource is = new DocumentInputSource(document);

            try {
                // No need to do CSS parsing for source error parsing!
                DocumentBuilder parser = MarkupService.createRaveSourceDocumentBuilder(false);
                parser.setErrorHandler(this);

                // TODO: only set this to empty if a first parse fails?
                parser.setEntityResolver(new EntityResolver() {
                        public org.xml.sax.InputSource resolveEntity(String pubid, String sysid) {
                            // XXX I should be able to have a shared
                            // instance for this
                            return new org.xml.sax.InputSource(new ByteArrayInputStream(new byte[0]));
                        }
                    });
                parser.parse(is);
            } catch (ParserConfigurationException pce) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, pce);
            } catch (IOException ioe) {
                // deliberately swallowed exception - we're looking for parse errors!!!
                // XXX What the above was supposed to mean?
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, ioe);
            } catch (SAXException se) {
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, se);
            }
        }

        // Add annotations to the editor
        attachAnnotations();
    }

    private void attachAnnotations() {
        try {
            FacesModel facesModel = facesModelReference.get();
            if (facesModel == null) {
                return;
            }
            
            // Add annotations to the editor
            DataObject dobj = DataObject.find(facesModel.getMarkupFile());
            LineCookie cookie = (LineCookie)dobj.getCookie(LineCookie.class);
            Line.Set lines = cookie.getLineSet();

            for (Iterator it = annotations.values().iterator(); it.hasNext();) {
                ParserAnnotation annotation = (ParserAnnotation)it.next();
                annotation.attachToLineSet(lines);
            }
        } catch(DataObjectNotFoundException dnfe) {
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, dnfe);
        }
    }

    /** Remove all current parse annotations from the editor */
    private void detachAnnotations() {
        if (annotations == null) {
            return;
        }

        for (Iterator it = annotations.values().iterator(); it.hasNext();) {
            Annotation annotation = (Annotation)it.next();
            annotation.detach();
        }

        annotations = null;
    }

    /** Add a parser error to the hashmap for the given sax error */
    private void addError(ParserAnnotation anno) {
        // This is trying to ensure that annotations on the same
        // line are "chained" (so we get a single annotation for
        // multiple errors on a line).
        // If we knew the errors were sorted by file & line number, 
        // this would be easy (and we wouldn't need to do the hashmap
        // "sort")
        Integer lineInt = new Integer(anno.getLine());
        ParserAnnotation prev = (ParserAnnotation)annotations.get(lineInt);

        if (prev != null) {
            prev.chain(anno);
        } else {
            annotations.put(lineInt, anno);
        }
    }

    /** Add a parser error to the hashmap for the given sax error */
    private void addError(SAXParseException exception) {
        String message = exception.getMessage();
        int line = exception.getLineNumber();
        // If file is empty then saved, we get -1
        if (line < 0)
            line = 1;
        int column = exception.getColumnNumber();
        if (column < 0)
            column = 0;

        //String publicId = exception.getPublicId();
        //String systemId = exception.getSystemId();
        FacesModel facesModel = facesModelReference.get();
        if (facesModel == null) {
            return;
        }
        ParserAnnotation anno = new ParserAnnotation(message, facesModel.getMarkupFile(), line, column);
        addError(anno);
    }

    private void addErrors(ParserAnnotation[] annotations) {
        for (int i = 0; i < annotations.length; i++) {
            addError(annotations[i]);
        }
    }

    // ---- Implements org.xml.sax.ErrorHandler ----------
    public void error(SAXParseException exception) {
        addError(exception);
    }

    public void fatalError(SAXParseException exception) {
        addError(exception);
    }

    public void warning(SAXParseException exception) {
        addError(exception);
    }

    public void shown() {
        if (runTimer != null) {
            runTimer.stop();
            runTimer = null;
        } else {
            if (document == null) {
                try {
                    FacesModel facesModel = facesModelReference.get();
                    if (facesModel == null) {
                        return;
                    }
                    DataObject dobj = DataObject.find(facesModel.getMarkupFile());
                    document = getDocumentForDataObject(dobj);
                } catch(DataObjectNotFoundException dnfe) {
                    ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, dnfe);
                }
            }
        }

        if(document != null) {
            document.addDocumentListener(this);
        }
        docChanged(); // Cause initial parse. Should I even do it

        // immediately? (Call reparse() instead, so there's no delay?)
    }

    public void hidden() {
        if (runTimer != null) {
            runTimer.stop();
            runTimer = null;
        }

        if(document != null) {
            document.removeDocumentListener(this);
        }
        detachAnnotations();
    }

    public void activated() {
        reparse(true);
    }

    public void opened() {
    }

    public void closed() {
    }

    // ----- Implements javax.swing.event.DocumentListener -------
    public void changedUpdate(javax.swing.event.DocumentEvent documentEvent) {
        // Only attribute changes - no point in reparsing
        // In fact, there's a danger in reparsing here; when annotations
        // are added (as when we get an error), that causes a
        // changedUpdate event! This in turns causes another parse,
        // which causes the annotation to be readded, which causes the
        // changedUpdate again, looping forever.
    }

    public void insertUpdate(javax.swing.event.DocumentEvent documentEvent) {
        docChanged();
    }

    public void removeUpdate(javax.swing.event.DocumentEvent documentEvent) {
        docChanged();
    }
    // ------ End od Implements DocumentListener
    
    /** Gets the document corresponding to a given data object.
     * @return The document, or null if the data object is not openable */
    private static Document getDocumentForDataObject(DataObject dobj) {
        if(dobj == null) {
            return null;
        }
        
        EditorCookie edit = (EditorCookie)dobj.getCookie(EditorCookie.class);

        if (edit == null) {
            return null;
        }

        try {
            return edit.openDocument();
        } catch (java.io.IOException ioe) {
            ErrorManager.getDefault().notify(ioe);
        }

        return null;
    }
    
}
