/* ------------------------------------------------------------------------
 * $Id: Quaternion.hh,v 1.4 2001/08/16 11:32:20 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * Reference from
 * <http://www.gamasutra.com/features/19980703/quaternions_01.htm>
 * ------------------------------------------------------------------------
 * File created 2001-06-15 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Quaternion_hh_
#define _Quaternion_hh_

// -- System Includes
#include <cmath>

// -- Local Includes
#include "Vector3D.hh"
#include "Matrix3D.hh"
#include "Math.hh"

// -- Class Declarations

/**
 * Quaternion class. Quaternions are four-dimensional entities which
 * are particularly useful for representing 3D rotations.
 **/
class Quaternion {
public:
    
    /**
     * Constructor.
     **/
    Quaternion(void) : _w(0.0f) { }
    
    /**
     * Constructor.
     **/
    Quaternion(float x, float y, float z, float w) : _w(w), _v(x, y, z) { }
    
    /**
     * Constructor.
     **/
    Quaternion(const Vector3D &u, float w) : _w(w), _v(u) { }

    /**
     * Copy constructor.
     **/
    Quaternion(const Quaternion &q) : _w(q._w), _v(q._v) { }
    
    /**
     * Quarternial addition.
     **/
    Quaternion operator + (const Quaternion &q) const {
	return Quaternion(_v + q._v, _w + q._w);
    }

    /**
     * Quarternial subtraction.
     **/
    Quaternion operator - (const Quaternion &q) const {
	return Quaternion(_v - q._v, _w - q._w);
    }
    
    /**
     * Quarternial multiplication. Note that multiplicating two
     * rotation quaternions result in concatenating the rotations in
     * the given order.
     **/
    Quaternion operator * (const Quaternion &q) const {
	return Quaternion(Vector3D::cross(_v, q._v) + q._v * _w + _v * q._w,
			  _w * q._w - Vector3D::dot(_v, q._v));
    }
    
    /**
     * Assignment operator.
     **/
    Quaternion &operator = (const Quaternion &q) {
	_v = q._v; _w = q._w;
	return *this;
    }

    /**
     * Multiplication assignment operator.
     **/
    Quaternion &operator *= (const Quaternion &q) {
	*this = *this * q;
	return *this;
    }
    
    /**
     * Compute quaternion norm.
     **/
    float norm(void) const {
	return (_w * _w + _v.x() * _v.x() + _v.y() * _v.y() + _v.z() * _v.z());
    }
    
    /** 
     * Is this a unit quaternion?
     **/
    bool isUnit(void) const {
	return norm() == 1.0f;
    }
    
    /**
     * Turn the quaternion into its conjugate.
     **/
    void conjugate(void) {
	_v = -_v; 
    }
    
    /**
     * Invert the quaternion. 
     **/
    void invert(void) {
	conjugate();
	_v /= norm();
    }
    
    /**
     * Normalize the quaternion.
     **/
    void normalize(void) {
	float norm_val = norm();
	_w /= norm_val; _v /= norm_val;
    }
    
    /**
     * Set the quaternion to addition identity.
     **/
    void addIdentity() {
	_w = 0.0f; _v.x() = 0; _v.y() = 0; _v.z() = 0;
    }
    
    /** 
     * Set the quaternion to multiplication identity.
     **/
    void mulIdentity() {
	_w = 1.0f; _v.x() = 0; _v.y() = 0; _v.z() = 0;
    }
    
    /**
     * Retrieve scalar component. Read-only.
     **/
    float w() const { return _w; }
    
    /**
     * Retrieve scalar component.
     **/
    float &w() { return _w; }

    /**
     * Retrieve vector component. Read-only.
     **/
    const Vector3D &v() const { return _v; }
    
    /**
     * Retrieve vector component.
     **/
    Vector3D &v() { return _v; }
    
    /**
     * Convert to quaternion from angle and axis representation.
     * Angles are represented in degrees (and will be converted to
     * radians internally).
     *
     * @param angle angle to rotate (in degrees).
     * @param axis axis to rotate around.
     **/
    void fromAngleAxis(float angle, const Vector3D &axis) {
	//_w = (float) cos(angle); _v = axis * (float) sin(angle); 
	float half_rad = Math::degToRad(angle * 0.5f);
	_w = (float) cos(half_rad); _v = axis * (float) sin(half_rad); 
    }
    
    /**
     * Convert from quaternion to angle and axis representation.
     **/
    void toAngleAxis(float &angle, Vector3D &axis) const;
    
    /**
     * Convert to quaternion from matrix representation.
     **/
    void fromMatrix(float matrix[4][4]);

    /**
     * Convert from quaternion to matrix representation.
     **/
    void toMatrix(float matrix[4][4]) const;
    
    /**
     * Convert from quaternion to matrix representation.
     *
     * @return a rotation matrix representing the quaternion.
     **/
    Matrix3D toMatrix() const;
    
    /**
     * Convert to quaternion from Euler angle representation.
     **/
    void fromEuler(float x, float y, float z);
    
    /**
     * Perform spherical linear interpolation between two
     * quaternions, returning the resulting quaternion.
     *
     * @param from originating quaternion.
     * @param to destination quaternion.
     * @param t parameter in the range of [0..1].
     **/
    static Quaternion computeSLERP(const Quaternion &from,
				   const Quaternion &to, float t);

    /**
     * Quaternion output operator.
     *
     * @param f output stream to use.
     * @param q quaternion to output to the stream.
     * @return output stream.
     **/
    friend std::ostream &operator << (std::ostream &f, const Quaternion &q);
    
private:
    
    // Scalar component
    float _w;
    
    // Vector component
    Vector3D _v;
};

#endif /* Quaternion.hh */
