/* ------------------------------------------------------------------------
 * $Id: Zorn.hh,v 1.3 2001/06/25 12:28:34 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-18 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Zorn_hh_
#define _Zorn_hh_

// -- System Includes
#include <string>
#include <vector>

// -- Zorn Includes
#include "Value.hh"
#include "FunctionSet.hh"
#include "RuleSet.hh"

namespace Zorn {

    // -- Forward Declarations
    class Store;
    class EvalVisitor;

    // -- Class Declarations
        
    /**
     * Zorn interpreter class. This is a facade that hides the
     * internal details of parsing and evaluating rules.
     **/
    class Interpreter {
    public:

	/**
	 * Constructor.
	 **/
	Interpreter();

	/**
	 * Destructor.
	 **/
	~Interpreter();
	
	/**
	 * Parse a given Zorn rule file and add any valid rules to the
	 * existing set of rules contained in the interpreter.
	 *
	 * @param filename Zorn rule file name.
	 **/
	bool parse(const std::string &filename);

	/**
	 * Add a function object to the interpreter function set. All
	 * implemented Zorn functions in the function object will be
	 * mapped into the function set, and the reference count for
	 * the function object will be bumped for each.
	 *
	 * @param fobj function object to add.	
	 **/
	void addFunctionObject(FunctionObject *fobj);

	/**
	 * Perform an evaluation of the Zorn rules that have been
	 * added to the interpreter given a list of new premises. The
	 * premises will be temporarily added to the memory store
	 * during the evaluation and then removed.
	 *
	 * @param premises list of new premises.
	 **/
	void evaluate(const std::vector<NamedValue> &premises);

	/**
	 * Error output function type definition
	 **/
	typedef void (*OutputFunction)(const char *msg);

	/**
	 * Set the output function to use for reporting parse errors
	 * in the interpreter.
	 *
	 * @param output output function pointer.
	 **/
	void setErrorOutput(OutputFunction output) { 
	    _output = output;
	}

	/**
	 * Retrieve the number of primary rules in the interpreter's
	 * rule set. Primary rules are at the top of the rule files,
	 * i.e. does not include nested rules. All primary rules must
	 * be evaluated for each input event, so this number should be
	 * kept low.
	 *
	 * @return number of primary rules.
	 **/
	int getRuleCount() const { return _ruleset->getRuleCount(); }
	
    private:

	/**
	 * Error output method.
	 *
	 * @param s string message to output.
	 **/
	static void errorOutput(const char *s);

	bool _initialized;
	FunctionSet::Instance _functionset;
	RuleSet::Instance _ruleset;
	Store *_store;
	EvalVisitor *_evaluator;
	OutputFunction _output;
    };

};

#endif /* Zorn.hh */
