/* ------------------------------------------------------------------------
 * VNCInterface.cc
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-09-23 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- Standard Includes
#include <iostream>
#include <string.h>
#include <stdio.h>
#include <cmath>

// -- X11 Includes
#include <X11/Xmd.h>

// -- VNC Includes
extern "C" {
#include "Garbo/rfbproto.h"
#include "Garbo/vncauth.h"
}

// -- 3Dwm Includes
#include "Celsius/Mutex.hh"
#include "Celsius/debug.hh"
#include "Garbo/VNCInterface.hh"

// -- Defines

// @@@Need to make this portable!
int endianTest = 1;

#define Swap16IfLE(s) \
    (*(char *)&endianTest ? ((((s) & 0xff) << 8) | (((s) >> 8) & 0xff)) : (s))

#define Swap32IfLE(l) \
    (*(char *)&endianTest ? ((((l) & 0xff000000) >> 24) | \
			     (((l) & 0x00ff0000) >> 8)  | \
			     (((l) & 0x0000ff00) << 8)  | \
			     (((l) & 0x000000ff) << 24))  : (l))

//#define DEBUG
#define MAX_ENCODINGS	10

// -- Local Function Prototypes
static int countBits(int mask);

// -- Code Segment

VNCInterface::VNCInterface(const char *host, int port, const char *password)
    : _host(host), _port(port), _incremental(false)
{
    DPRINT("VNCInterface::VNCInterface\n");
    // Set up pixel format (in VNC, we get to define it ourselves)
    _pf.depth      = 24; _pf.size       = 32;
    _pf.red_bits   = 8;  _pf.red_mask   = 0xff;     _pf.red_shift   = 0;
    _pf.green_bits = 8;  _pf.green_mask = 0xff00;   _pf.green_shift = 8;
    _pf.blue_bits  = 8;  _pf.blue_mask  = 0xff0000; _pf.blue_shift  = 16;

    // Connect to the VNC server
    connectToVNCServer(password);

    // Create framebuffer    
    int framebuffer_size = _width * _height * (_pf.size / 8);
    _framebuffer = new char [framebuffer_size];
    memset(_framebuffer, 0, framebuffer_size);
    
    // Inform the VNC server of pixel formats and encodings
    setVNCPixelFormat();
    setVNCEncodings();
}

VNCInterface::~VNCInterface()
{
    // Shutdown and close the VNC socket
    _sock.shutdown();
    _sock.close();
}
    
void VNCInterface::run()
{
    DPRINT("VNCInterface::run()\n");
    // Loop forever
    while (true) {
	
	// Handle a server message
	handleVNCServerMessage();
    }
}

void VNCInterface::updateFramebuffer(int x, int y, int w, int h)
{
    rfbFramebufferUpdateRequestMsg fur;
    DPRINT("VNCInterface::updateFramebuffer\n");
    // Initialize the message
    fur.type = rfbFramebufferUpdateRequest;
    fur.incremental = _incremental ? 1 : 0;
    fur.x = Swap16IfLE(x);
    fur.y = Swap16IfLE(y);
    fur.w = Swap16IfLE(w);
    fur.h = Swap16IfLE(h);

    // Send the message
    writeData(&fur, sz_rfbFramebufferUpdateRequestMsg);

    // This is no longer the first time we update, we want incremental
    // framebuffer updates
    _incremental = true;
}

void VNCInterface::pointerEvent(int x, int y, int button_mask)
{
    rfbPointerEventMsg pe;

    // Initialize the message
    pe.type = rfbPointerEvent;
    pe.buttonMask = button_mask;
    if (x < 0) x = 0;
    if (y < 0) y = 0;
    pe.x = Swap16IfLE(x);
    pe.y = Swap16IfLE(y);
    
    // Send the message
    writeData(&pe, sz_rfbPointerEventMsg);
}

void VNCInterface::keyEvent(unsigned int key, bool pressed = true)
{
    rfbKeyEventMsg ke;

    // Initialize the message
    ke.type = rfbKeyEvent;
    ke.down = pressed ? 1 : 0;
    ke.key = Swap32IfLE(key);
    
    // Send the message
    writeData(&ke, sz_rfbKeyEventMsg);
}

// -- VNC utility methods (protected) -------------------------------------

void VNCInterface::readData(void *data, int len)
{
    char *buf = (char *) data;
    
    // Read data until we're done
    while (len > 0) {
	
	// Read data from the socket
	int bytes_read = _sock.receive(buf, len);

	// Update buffer position
	len -= bytes_read;
	buf += bytes_read;
    }
}

void VNCInterface::writeData(const void *data, int len) 
{
    char *buf = (char *) data;

    while (len > 0) {
	
	// Write data to the socket
	int bytes_sent = _sock.send(buf, len);

	// Update buffer position
	len -= bytes_sent;
	buf += bytes_sent;
    }
}

void VNCInterface::handleVNCVersion()
{
    rfbProtocolVersionMsg pv;
    int msglen = sz_rfbProtocolVersionMsg;
    
    // Read the message and zero-terminate the result
    readData(pv, msglen);
    pv[msglen] = '\0';
    
    // Extract server version from protocol version message
    if (sscanf(pv, rfbProtocolVersionFormat, &_serverMajor, &_serverMinor)!=2)
	throw "Not a valid VNC server!";
    
    // Form and write the client protocol version message
    sprintf(pv, rfbProtocolVersionFormat,
	    rfbProtocolMajorVersion, rfbProtocolMinorVersion);
    writeData(pv, msglen);

#ifdef DEBUG
    std::cerr << "Server uses protocol version " << _serverMajor << "." 
	      << _serverMinor << std::endl;
#endif 
}

bool VNCInterface::handleVNCAuthentication(const char *pwd)
{
    #define MAX_PASSWORD_SIZE 64
    uint32 scheme, result, reasonLen;
    uint8 challenge[CHALLENGESIZE];
    char password[MAX_PASSWORD_SIZE];
    char *reason;

    // Copy the password into mutable array
    strncpy(password, pwd, MAX_PASSWORD_SIZE);
    
    // Which authentication scheme to use?
    readData(&scheme, 4);
    scheme = Swap32IfLE(scheme);

#ifdef DEBUG
    std::cerr << "Authentication scheme " << scheme << std::endl;
#endif 
    
    // What do we do with it?
    switch (scheme) {

    case rfbConnFailed:
	
	// Why did it fail? 
	readData((void *) &reasonLen, 4);
	reason = (char *) malloc(reasonLen);
	readData(reason, reasonLen);
	std::cerr << "Connection failed: " << reason << std::endl;
	free(reason);
	return false;

    case rfbNoAuth:
	// No authentication needed
	break;

    case rfbVncAuth:
	
	// VNC password authentication
	readData(challenge, CHALLENGESIZE);

	// Take care of the password
	if (password == 0 || strlen(password) == 0)
	    return false;
	if (strlen(password) > 8)
	    password[8] = '\0';
	
	// Encrypt it! (using the vncauth lib)
	vncEncryptBytes(challenge, password);

	// @@@ Lose it from memory like in vncviewer?

	// Send it back to the server and get the response
	writeData(challenge, CHALLENGESIZE);
	readData(&result, 4);
	result = Swap32IfLE(result);

	// How'd it go?
	switch (result) {

	case rfbVncAuthOK:

	    // Everything is fine, return with success
	    return true;

	case rfbVncAuthFailed:
	case rfbVncAuthTooMany:
	default:
	    
	    // Failed to authenticate, return with failure
	    return false;
	}
	
    default:
	return false;
    }
    
    // We should never reach this position...
    return true;
}

void VNCInterface::handleVNCInitialization()
{
    rfbClientInitMsg ci;
    rfbServerInitMsg si;

    // Send the client initialization message
    ci.shared = 1;
    writeData(&ci, sz_rfbClientInitMsg);
    
    // Read the server initialization message
    readData(&si, sz_rfbServerInitMsg);

    // Extract the information we need
    _width        = Swap16IfLE(si.framebufferWidth);
    _height       = Swap16IfLE(si.framebufferHeight);
    
    int red_max   = Swap16IfLE(si.format.redMax);
    int green_max = Swap16IfLE(si.format.greenMax);
    int blue_max  = Swap16IfLE(si.format.blueMax);
    
    si.nameLength = Swap32IfLE(si.nameLength);

    // Set up pixel format
    _native_pf.depth       = si.format.depth;
    _native_pf.size        = si.format.bitsPerPixel;

    _native_pf.red_shift   = si.format.redShift;
    _native_pf.red_mask    = red_max << _native_pf.red_shift;
    _native_pf.red_bits    = countBits(red_max);

    _native_pf.green_shift = si.format.greenShift;
    _native_pf.green_mask  = green_max << _native_pf.green_shift;
    _native_pf.green_bits  = countBits(green_max);

    _native_pf.blue_shift  = si.format.blueShift;
    _native_pf.blue_mask   = blue_max << _native_pf.blue_shift;
    _native_pf.blue_bits   = countBits(blue_max);
    
#ifdef DEBUG
    std::cerr << "Desktop size: " << _width << "x" << _height << "x" 
	      << _native_pf.depth << "." << std::endl;
    std::cerr << "Pixel format: red (" << _native_pf.red_bits << ", " 
	      << _native_pf.red_shift << "), green (" << _native_pf.green_bits
	      << ", " << _native_pf.green_shift  << "), blue ("
	      << _native_pf.blue_bits << ", " << _native_pf.blue_shift << ")."
	      << std::endl;
#endif 

    // Read the name of the desktop
    char *desktopName = (char *) malloc(si.nameLength + 1);
    readData(desktopName, si.nameLength);
    desktopName[si.nameLength] = '\0';
    _name = desktopName;
    free(desktopName);

#ifdef DEBUG
    std::cerr << "Desktop name: " << _name << std::endl;
#endif 
}

void VNCInterface::connectToVNCServer(const char *password)
{
    // Connect the socket
    _sock.connect(InetAddress(_host.c_str(), _port));
    
    // Read and write version information
    handleVNCVersion();
    
    // Authenticate client
    if (handleVNCAuthentication(password) == false)
	throw "Authentication failed.";
    
    // Initialize the connection
    handleVNCInitialization();
}

void VNCInterface::setVNCPixelFormat()
{
    rfbSetPixelFormatMsg spf;
    
    // Set up the message
    spf.type                = rfbSetPixelFormat;

    // Pixel format: 24 depth, 32 bpp, true color, RGB 8:8:8
    // @@@ Need to add support for bigendian architectures!
    spf.format.bitsPerPixel = _pf.size;
    spf.format.depth        = _pf.depth;
    spf.format.bigEndian    = false;
    spf.format.trueColour   = true;

    spf.format.redMax     = Swap16IfLE((1 << _pf.red_bits) - 1);
    spf.format.redShift   = _pf.red_shift;
    
    spf.format.greenMax   = Swap16IfLE((1 << _pf.green_bits) - 1);
    spf.format.greenShift = _pf.green_shift;
    
    spf.format.blueMax    = Swap16IfLE((1 << _pf.blue_bits) - 1);
    spf.format.blueShift  = _pf.blue_shift;

    // Send it to the server
    writeData(&spf, sz_rfbSetPixelFormatMsg);
}
    
void VNCInterface::setVNCEncodings()
{   
    char buf[sz_rfbSetEncodingsMsg + MAX_ENCODINGS * 4];
    rfbSetEncodingsMsg *se = (rfbSetEncodingsMsg *) buf;
    CARD32 *encs = (CARD32 *) (&buf[sz_rfbSetEncodingsMsg]);
    int len = 0;

    // Initialize message
    se->type = rfbSetEncodings;
    se->nEncodings = 0;

    // @@@ We just want raw encoding for now!
    encs[se->nEncodings++] = Swap32IfLE(rfbEncodingRaw);

    /*    
    if (_sock.isSameHost() == true) {
	fprintf(stderr,"Same machine: preferring raw encoding\n");
	encs[se->nEncodings++] = Swap32IfLE(rfbEncodingRaw);
    }

    // Set up the encodings we support
    encs[se->nEncodings++] = Swap32IfLE(rfbEncodingCopyRect);
    encs[se->nEncodings++] = Swap32IfLE(rfbEncodingHextile);
    encs[se->nEncodings++] = Swap32IfLE(rfbEncodingCoRRE);
    encs[se->nEncodings++] = Swap32IfLE(rfbEncodingRRE);
    */
    
    len = sz_rfbSetEncodingsMsg + se->nEncodings * 4;
    se->nEncodings = Swap16IfLE(se->nEncodings);
    
    // Send the message to the server
    writeData(buf, len);
}

void VNCInterface::handleVNCFramebufferUpdate()
{
    rfbFramebufferUpdateMsg fu;

    // Read the message (bar the initial type byte)
    readData((char *) &fu + 1, sz_rfbFramebufferUpdateMsg - 1);
    int num_rectangles = Swap16IfLE(fu.nRects);

    // Read the rectangles and update the framebuffer
    for (int i = 0; i < num_rectangles; i++) {
	
	rfbFramebufferUpdateRectHeader rect;
	int linesToRead, bytesPerLine;

	// Read a rectangle header from the server
	readData(&rect, sz_rfbFramebufferUpdateRectHeader);
	
	// Convert results to something we can use
	rect.r.x = Swap16IfLE(rect.r.x);
	rect.r.y = Swap16IfLE(rect.r.y);
	rect.r.w = Swap16IfLE(rect.r.w);
	rect.r.h = Swap16IfLE(rect.r.h);
	
	rect.encoding = Swap32IfLE(rect.encoding);

	// Sanity checking
	if (rect.r.x + rect.r.w > _width || rect.r.y + rect.r.h > _height)
	    throw "Rectangle too large.";
	if ((rect.r.h * rect.r.w) == 0)
	    continue;
	
	// Store away the rectangle for later update
	int x = rect.r.x; int y = rect.r.y;
	int w = rect.r.w; int h = rect.r.h;
	
	// Now, do different things depending on the encoding
	switch (rect.encoding) {
	    
	case rfbEncodingRaw:
	    
	    // How many lines can we copy?
	    bytesPerLine = rect.r.w * (_pf.size / 8);
	    linesToRead = BUFFER_SIZE / bytesPerLine;
	    
	    // Read the entire rectangle from the server
	    while (rect.r.h > 0) {
		
		// Clamp value if too large
		if (linesToRead > rect.r.h)
		    linesToRead = rect.r.h;
		
		// Read data from the server
		readData(buffer, bytesPerLine * linesToRead);

		// Copy the rectangle buffer to the server
		copyRectToFramebuffer(buffer, rect.r.x, rect.r.y, rect.r.w,
				      linesToRead);
		
		// Update readbuffer and framebuffer positions
		rect.r.h -= linesToRead;
		rect.r.y += linesToRead;
	    }
	    break;

	default:
	    throw "Unknown rectangle encoding.";
	}

	// Add the rectangle to the update rectangle queue
	addUpdate(x, y, w, h);
    }
    
    // Finally, we send a framebuffer update request (incremental)
    // @@@ Remove this? 
    updateFramebuffer(0, 0, _width, _height);
}

void VNCInterface::handleVNCServerCutText()
{
    rfbServerCutTextMsg sct;
    
    // Read the message (bar the initial type byte)
    readData((char *) &sct + 1, sz_rfbServerCutTextMsg - 1);
    sct.length = Swap32IfLE(sct.length);
    
    // Retrieve the actual text
    char *text = (char *) malloc(sct.length + 1);
    readData(text, sct.length);
    text[sct.length] = '\0';
    free(text);
}

void VNCInterface::handleVNCServerMessage()
{
    uint8 msg;

    DPRINT("reading VNC message\n");
    // Retrieve the message type
    readData(&msg, 1);

    // What do we do with it?
    switch (msg) {
	
    case rfbFramebufferUpdate: 
	handleVNCFramebufferUpdate();
	break;
	
    case rfbBell:
	break;
	
    case rfbServerCutText:
	handleVNCServerCutText();
	break;

    default:
	throw "Unknown message type from server.";
    }
}

int countBits(int mask)
{
    int bits = 0;
    for (int i = 1; mask & i; i <<= 1, bits++);
    return bits;
}
