/* ------------------------------------------------------------------------
 * WSInterface.cc
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-09-24 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Celsius/debug.hh"
#include "Garbo/WSInterface.hh"

// -- Code Segment
WindowingSystemInterface::WindowingSystemInterface()
    : _framebuffer(0)
{
    DPRINT("WindowingSystemInterface::WindowingSystemInterface\n");
}

WindowingSystemInterface::~WindowingSystemInterface() 
{ 
    // Deallocate the framebuffer if it exists
    if (_framebuffer != 0) delete _framebuffer;
}

void WindowingSystemInterface::copyRectToFramebuffer(char *buf, int x, int y,
						     int w, int h)
{
    int bpp = _pf.size / 8;
    char *fbptr = _framebuffer + (y * _width + x) * bpp;
    
    // Copy the contents of the rectangle buffer
    while (h > 0) {
	
	// Copy memory
	memcpy(fbptr, buf, w * bpp);

	// Update buffer positions
	fbptr += _width * bpp;
	buf   += w * bpp;
	h     -= 1;
    }
}

bool WindowingSystemInterface::getFramebufferUpdate(Rectangle &r)
{
    Guard<Mutex> guard(_mutex);

    // Maybe there is no update?
    if (_rectQueue.empty() == true) return false;

    // Yes, there is, update argument and then remove it
    r = _rectQueue.front();
    _rectQueue.pop_front();
    return true;
}

void WindowingSystemInterface::addUpdate(int x, int y, int w, int h)
{
    Guard<Mutex> guard(_mutex);

    // Construct a new rectangle
    Rectangle r;
    r.x = x; r.y = y; r.width = w; r.height = h;
    
    // Let's move through the rectangle queue
    std::list<Rectangle>::iterator i = _rectQueue.begin();
    while (i != _rectQueue.end()) {
	
	// Get a temporary iterator and move the main iterator forward
	std::list<Rectangle>::iterator temp = i++;
	
	// Can we merge these two rectangles?
	if (isMergeable(r, *temp) == true) {

	    // Then merge and remove the original rectangle
	    r = merge(r, *temp);
	    _rectQueue.erase(temp);
	}
    }
    
    // Finally, add the new rectangle to the end of the queue
    _rectQueue.push_back(r);
}

bool WindowingSystemInterface::isMergeable(const Rectangle &r1,
					   const Rectangle &r2)
{
    // @@@ NOTE: The implementation of this function is quite
    // arbitrary, we just need a heuristic that yields good
    // performance when merging rectangles.
    
    int x1 = r1.x + r1.width;  int x2 = r2.x + r2.width;
    int y1 = r1.y + r1.height; int y2 = r2.y + r2.height;

    // Do the rectangles coincide horizontally?
    if ((r1.x >= r2.x && r1.x <= x2) ||
	(r1.x <= r2.x && x1 >= r2.x)) {

	// Yes? Then what about vertically?
	if ((r1.y >= r2.y && r1.y <= y2) ||
	    (r1.y <= r2.y && y1 >= r2.y))
	    return true;
    }
    
    // Not mergeable
    return false;
}

WindowingSystemInterface::Rectangle
WindowingSystemInterface::merge(const Rectangle &r1,
				const Rectangle &r2)
{
    Rectangle r;
    
    // Find the resulting rectangle that encloses both rectangles
    r.x = std::min(r1.x, r2.x); r.y = std::min(r1.y, r2.y);
    int x1 = r1.x + r1.width;  int x2 = r2.x + r2.width;
    int y1 = r1.y + r1.height; int y2 = r2.y + r2.height;
    r.width = x1 > x2 ? r1.width : r2.width;
    r.height = y1 > y2 ? r1.height : r2.height;

    return r;
}
