/* ------------------------------------------------------------------------
 * $Id: SDLfbconPlatform.cc,v 1.6 2001/07/18 07:48:37 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-05-05 by Steve Houston.
 *
 * Copyright (c) 2000 Antony Suter <antony@mira.net>.
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2001 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- OpenGL Includes
#include <GL/gl.h>
#include <GL/glu.h>

// -- 3Dwm Includes
#include "Celsius/Exception.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Thread.hh"
#include "Celsius/Logger.hh"
#include "Polhem/LogGroup.hh"
#include "Polhem/InputDevice.hh"
#include "Polhem/RendererImpl.hh"
#include "Polhem/SDLfbconPlatform.hh"

#include "Polhem/InterTraxDevice.hh"

// Helper functions
// If the number of helper functions grows, use namespace to control scope
// rather than static.
static void FlipAndReorder(SDL_Surface* out, SDL_Surface* in);

// -- Constants

// Maximum number of input events in the event queue
const int maxInputEvents = 64;

// Include the SDL event listener class! (it's the same for all SDL
// platforms)
#include "SDLInputListener.inl"

// -- Code Segment

/**
 * Constructor
 *
 * Currently SDLfbcon only works at 32bpp. Ignore the depth argument.
 **/
SDLfbconPlatform::SDLfbconPlatform(int &argc, char **argv, int w, int h, int d)
    : _width(w), _height(h), _depth(32), _eventQueue(maxInputEvents)
{

    // Initialize the SDL library
    if (SDL_Init(SDL_INIT_VIDEO) < 0) {
	throw Exception(SDL_GetError());
    }
    
    // Set up two buffers. offScreen for the renderer to write into and
    // onScreen for displaying.

    _onScreen = SDL_SetVideoMode(_width, _height, _depth, 
				 SDL_HWSURFACE | SDL_DOUBLEBUF);
    _offScreen = SDL_CreateRGBSurface(SDL_SWSURFACE, _width, _height, 
				      _depth, 0xFF, 0xFF00, 0xFF0000, 0x0);
		
    if ( (_offScreen == 0) || (_onScreen == 0) ) {
	throw Exception(SDL_GetError());
    }
    
    // Configure SDL
    SDL_WM_SetCaption("3Dwm - fbcon", "3Dwm");
    SDL_EventState(SDL_QUIT, SDL_ENABLE);
    SDL_EventState(SDL_ACTIVEEVENT, SDL_IGNORE);
    SDL_EnableKeyRepeat(500, 10);

    // Create and add the desired input devices to the event queue
    addInputDevice(new SDLInputListener(&_eventQueue));
    //addInputDevice(new InterTraxDevice(&_eventQueue));
}

SDLfbconPlatform::~SDLfbconPlatform()
{
    // Kill and deallocate the device threads
    for (std::vector<Thread *>::iterator i = _deviceThreads.begin();
	 i != _deviceThreads.end(); i++) {
	(*i)->kill();
	delete (*i);
    }

    // Deallocate the input devices as well
    for (std::vector<InputDevice *>::iterator i = _devices.begin();
	 i != _devices.end(); i++)
	delete (*i);
    
    // Free the SDL surfaces
    SDL_FreeSurface(_offScreen);
    SDL_FreeSurface(_onScreen);
    
    // Cleanup the SDL lib
    SDL_Quit();
}

RendererImpl *SDLfbconPlatform::createRenderer()
{
    RendererImpl *r;

    _mesaContext = OSMesaCreateContext( OSMESA_RGBA, NULL ); 
    if (!_mesaContext)
	return 0;

    OSMesaMakeCurrent( _mesaContext, _offScreen->pixels, GL_UNSIGNED_BYTE, 
		       _width, _height );

    r = new RendererImpl(_width, _height);
    return r;
}

InputEvent SDLfbconPlatform::getNextInputEvent()
{
    return _eventQueue.pop();
}

// TODO: chase these remaining gl calls out of here

void SDLfbconPlatform::clearDisplay()
{
    // Clear the buffers
    glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
}

void SDLfbconPlatform::swapDisplay()
{
    // Flush GL rendering pipeline and swap the buffers
    glFlush();

    // Lock the onScreen surface and copy the OSMesa buffer into it
    SDL_LockSurface(_onScreen);
    FlipAndReorder(_onScreen, _offScreen);
    SDL_UnlockSurface(_onScreen);

    SDL_Flip(_onScreen);
}

void SDLfbconPlatform::addInputDevice(InputDevice *device)
{
    Logger::log(LogGroup::Platform) 
	<< "Adding input device \"" << device->getDeviceName() << "\"" 
	<< std::endl;
    
    // First, add the device to the list of devices. Note that the
    // devices in this list are owned by us, so we will be the ones
    // that deallocate them.
    _devices.push_back(device);

    // Create a thread for the device and start it!
    Thread *device_thread = new Thread(device);
    device_thread->start();
    _deviceThreads.push_back(device_thread);
}

/**
 * FlipAndReorder()
 *
 *  There are 2 incompatibilities between the image buffer written by OSMesa 
 *  and the image buffer displayed by SDL.
 *
 *  The first is OSMesa writes bottom-up whereas SDL displays top-down. 
 *  A simple vertical flip resolves this.
 *
 *  The other is OSMesa writes out each RGBA pixel as 4 bytes, whereas
 *  SDL reads in a single 32-bit value. On little-endian machines this
 *  messes up the pixel color so we also correct for that here.
 *
 *  TODO: check for and handle big-endian machines
 **/

static void FlipAndReorder(SDL_Surface* out, SDL_Surface* in)
{
    int x, y, i;
    // Only handle 32bpp for now
    unsigned char *pin = (unsigned char*) in->pixels + 
	                 (in->pitch * (in->h - 1));
    unsigned int *pout = (unsigned int*) out->pixels;
    
    for (y = 0; y < in->h; y++) {
	i = 0;
	for (x = 0; x < in->w; x++) {
	    pout[x] = pin[i] << 16 | pin[i+1] << 8 | pin[i+2];
	    i += 4;
	}
	pin -= in->pitch;
	pout += (out->pitch / 4);
    }
}
