// ---------------------------------------------------------------------------
// - Key.cpp                                                                 -
// - afnix cryptographic library - key class implementation                  -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Key.hpp"
#include "Byte.hpp"
#include "Item.hpp"
#include "Sha256.hpp"
#include "Vector.hpp"
#include "Utility.hpp"
#include "Integer.hpp"
#include "CpyUtils.hxx"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a 128 bits random key

  Key::Key (void) {
    // default type is 128 bits
    d_type = K128;
    // create buffer and initialize
    long size = getsize ();
    p_kbuf    = new t_byte[size];
    for (long i = 0; i < size; i++) p_kbuf[i] = Utility::byternd (); 
  }

  // create a 128 bits message key

  Key::Key (const String& msg) {
    // default type is 128 bits
    d_type = K128;
    // create buffer and initialize
    long size = getsize ();
    p_kbuf    = new t_byte[size];
    // compute a hash value from the message
    Sha256 hasher (msg);
    // update the key buffer
    for (long i = 0; i < size; i++) p_kbuf[i] = hasher.gethash (i);
  }
  
  // create a random key by type

  Key::Key (const t_type type) {
    // set key type
    d_type = type;
    // create buffer and initialize
    long size = getsize ();
    p_kbuf    = new t_byte[size];
    for (long i = 0; i < size; i++) p_kbuf[i] = Utility::byternd (); 
  }

  // create a key by type and data

  Key::Key (const t_type type, const t_byte* data) {
    // set key type
    d_type = type;
    // create buffer and initialize
    long size = getsize ();
    p_kbuf    = new t_byte[size];
    for (long i = 0; i < size; i++) {
      p_kbuf[i] = (data == nilp) ? nilc : data[i];
    }
  }

  // create a key by type and string

  Key::Key (const t_type type, const String& msg) {
    // set key type
    d_type = type;
    // create buffer and initialize
    long size = getsize ();
    p_kbuf    = new t_byte[size];
    // compute a hash value from the message
    Sha256 hasher (msg);
    // update the key buffer
    for (long i = 0; i < size; i++) p_kbuf[i] = hasher.gethash (i);
  }

  // copy construct this key

  Key::Key (const Key& that) {
    that.rdlock ();
    // set key type
    d_type = that.d_type;
    // create buffer and initialize
    long size = getsize ();
    p_kbuf    = new t_byte[size];
    for (long i = 0; i < size; i++) p_kbuf[i] = that.p_kbuf[i];
    that.unlock ();
  }

  // destroy this key

  Key::~Key (void) {
    delete [] p_kbuf;
  }

  // assign a key to this one

  Key& Key::operator = (const Key& that) {
    wrlock ();
    that.rdlock ();
    // clean old key
    delete [] p_kbuf;
    // set key type
    d_type = that.d_type;
    // create buffer and initialize
    long size = that.getsize ();
    p_kbuf    = new t_byte[size];
    for (long i = 0; i < size; i++) p_kbuf[i] = that.p_kbuf[i];
    that.unlock ();
    unlock ();
    return *this;
  }

  // return the class name

  String Key::repr (void) const {
    return "Key";
  }

  // return the key type

  Key::t_type Key::gettype (void) const {
    rdlock ();
    t_type result = d_type;
    unlock ();
    return result;
  }

  // return the key size in bytes

  long Key::getsize (void) const {
    rdlock ();
    long result = 0;
    switch (d_type) {
    case Key::K128:
      result = 16;
      break;
    case Key::K192:
      result = 24;
      break;
    case Key::K256:
      result = 32;
      break;
    }
    unlock ();
    return result;
  }

  // return a key byte by index

  t_byte Key::get (const long index) const {
    rdlock ();
    // get key size and check
    long size = getsize ();
    if ((index < 0) || (index >= size)) {
      unlock ();
      throw Exception ("key-error","index is out of bound");
    }
    // key key byte
    t_byte result = p_kbuf[index];
    unlock ();
    return result;
  }

  // format the key result
  
  String Key::format (void) const {
    rdlock ();
    String result = btos (p_kbuf, getsize ());
    unlock ();
    return result;
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the object eval quarks
  static const long QUARK_KEY     = String::intern ("Key");
  static const long QUARK_K128    = String::intern ("K128");
  static const long QUARK_K192    = String::intern ("K192");
  static const long QUARK_K256    = String::intern ("K256");

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 4;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_GET     = String::intern ("get");
  static const long QUARK_FORMAT  = String::intern ("format");
  static const long QUARK_GETSIZE = String::intern ("get-size");
  static const long QUARK_GETTYPE = String::intern ("get-type");
  
  // map an item to a key type
  static inline Key::t_type item_to_type (const Item& item) {
    // check for a key item
    if (item.gettid () != QUARK_KEY)
      throw Exception ("item-error", "item is not a key item");
    // map the item to the enumeration
    long quark = item.getquark ();
    if (quark == QUARK_K128) return Key::K128;
    if (quark == QUARK_K192) return Key::K192;
    if (quark == QUARK_K256) return Key::K256;
    throw Exception ("item-error", "cannot map item to key type");
  }

  // map a key type to an item
  static inline Item* type_to_item (const Key::t_type type) {
    switch (type) {
    case Key::K128:
      return new Item (QUARK_KEY, QUARK_K128);
      break;
    case Key::K192:
      return new Item (QUARK_KEY, QUARK_K192);
      break;
    case Key::K256:
      return new Item (QUARK_KEY, QUARK_K256);
      break;
    }
    return nilp;
  }

  // evaluate an object data member

  Object* Key::meval (Runnable* robj, Nameset* nset, const long quark) {
    if (quark == QUARK_K128)
      return new Item (QUARK_KEY, QUARK_K128);
    if (quark == QUARK_K192)
      return new Item (QUARK_KEY, QUARK_K192);
    if (quark == QUARK_K256)
      return new Item (QUARK_KEY, QUARK_K256);
    throw Exception ("eval-error", "cannot evaluate member",
                     String::qmap (quark));
  }

  // create a new object in a generic way
  
  Object* Key::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new Key;
    // check for 1 argument
    if (argc == 1) {
      Object* obj = argv->get (0);
      // check for a string
      String* sobj = dynamic_cast <String*> (obj);
      if (sobj != nilp) return new Key (*sobj);
      // check for an item type
      Item* iobj = dynamic_cast <Item*> (obj);
      if (iobj != nilp) {
	t_type type = item_to_type (*iobj);
	return new Key (type);
      }
      throw Exception ("type-error", "invalid object with key constructor",
		       Object::repr (obj));
    }
    // check for 2 arguments
    if (argc == 2) {
      // get the key type
      Object* obj = argv->get (0);
      Item* iobj  = dynamic_cast <Item*> (obj);
      if (iobj == nilp) {
	throw Exception ("argument-error", "invalid arguments with key");
      }
      t_type type = item_to_type (*iobj);
      // get the string and key
      String msg = argv->getstring (1);
      return new Key (type, msg);
    }
    throw Exception ("argument-error", "too many arguments with key");
  }

  // return true if the given quark is defined

  bool Key::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark
  
  Object* Key::apply (Runnable* robj, Nameset* nset, const long quark,
		      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_FORMAT)  return new String   (format  ());
      if (quark == QUARK_GETSIZE) return new Integer  (getsize ());
      if (quark == QUARK_GETTYPE) return type_to_item (d_type);
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_GET) {
	long index = argv->getint (0);
	return new Byte (get (index));
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
