// ---------------------------------------------------------------------------
// - Selector.hpp                                                            -
// - afnix:sio module - i/o select class definition                          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_SELECTOR_HPP
#define  AFNIX_SELECTOR_HPP

#ifndef  AFNIX_VECTOR_HPP
#include "Vector.hpp"
#endif

#ifndef  AFNIX_INPUT_HPP
#include "Input.hpp"
#endif

#ifndef  AFNIX_OUTPUT_HPP
#include "Output.hpp"
#endif

namespace afnix {

  /// The Select class is a input/output selector which blocks until one
  /// object change its status. For an input stream object, a status change
  /// indicates that a character can be read. For an output stream, a status
  /// change indicates that a character can be written. When the a call to
  /// the 'check' method succeds, the method returns the first available 
  /// stream. If the 'check-all' method is called, the method returns a 
  /// vector with  all ready steams.
  /// @author amaury darsch

  class Selector : public Object {
  private:
    /// the input streams vector
    Vector d_isv;
    /// the output streams vector
    Vector d_osv;
    /// the private handle
    void*  p_handle;

  public:
    /// create an empty selector
    Selector (void);

    /// destroy this selector
    ~Selector (void);

    /// @return the class name
    String repr (void) const;

    /// add an input stream to the select list
    /// @param is the input stream to add
    void add (Input* is);

    /// add an output stream to the select list
    /// @param is the input stream to add
    void add (Output* os);

    /// @return the number of input streams
    long ilength (void) const;

    /// @return the number of output streams
    long olength (void) const;

    /// @return an input stream by index
    Input* iget (const long index) const;

    /// @return an output stream by index
    Output* oget (const long index) const;

    /// wait for one stream to be ready
    /// @param tout the timeout before returning
    Object* wait (const long tout) const;

    /// get all stream that are ready
    /// @param tout the timeout before returning
    Vector* waitall (const long tout) const;

  private:
    // make the copy constructor private
    Selector (const Selector&);
    // make the assignment operator private
    Selector& operator = (const Selector&);

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
