// ---------------------------------------------------------------------------
// - XmlMime.cpp                                                             -
// - afnix:wax module - xml mime class implementation                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Input.hpp"
#include "XmlMime.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the document content type
  static const String XML_MIME_TYPE = "application/xml"; 

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default xml mime document

  XmlMime::XmlMime (void) {
    d_mime = XML_MIME_TYPE;
  }

  // create a xml mime document by name

  XmlMime::XmlMime (const String& name) : XmlDocument (name) {
    d_mime = XML_MIME_TYPE;
  }

  // create a xml mime document by name and stream

  XmlMime::XmlMime (const String& name, Input* is) : XmlDocument (name, is) {
    d_mime = XML_MIME_TYPE;
  }

  // return the class name

  String XmlMime::repr (void) const {
    return "XmlMime";
  }

  // write a node to a buffer

  void XmlMime::write (Buffer& buf) const {
    rdlock ();
    try {
      // get the xml root node
      XmlRoot* node = getroot ();
      // write into the buffer
      if (node != nilp) node->write (buf);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write a node to an output stream

  void XmlMime::write (Output& os) const {
    rdlock ();
    try {
      // get the xml root node
      XmlRoot* node = getroot ();
      // write into the stream
      if (node != nilp) node->write (os);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // create a new object in a generic way

  Object* XmlMime::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for o argument
    if (argc == 0) return new XmlMime;
    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new XmlMime (name);
    }
    // check for 2 arguments
    if (argc == 2) {
      // get the document name
      String name = argv->getstring (0);
      // get the object and check
      Object* obj = argv->get (1);
      // check for an input stream
      Input* is = dynamic_cast <Input*> (obj);
      if (is != nilp) return new XmlMime (name, is);
      throw Exception ("type-error", 
		       "invalid object with document constructor",
		       Object::repr (obj));	
    }
    // wrong arguments
    throw Exception ("argument-error", 
		     "too many arguments with xml mime constructor");
  }

  // return true if the given quark is defined

  bool XmlMime::isquark (const long quark, const bool hflg) const {
    rdlock ();
    bool result = hflg ? Mime::isquark (quark, hflg) : false;
    if (result == false) {
      result = hflg ? XmlDocument::isquark (quark, hflg) : false;
    }
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* XmlMime::apply (Runnable* robj, Nameset* nset, const long quark,
			    Vector* argv) {
    // check the mime object
    if (Mime::isquark (quark, true) == true) {
      return Mime::apply (robj, nset, quark, argv);
    }
    // default to xml document object
    return XmlDocument::apply (robj, nset, quark, argv);
  }
}
