//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006,2007,2008 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "parse.h"
#include "utils.h"

#define EMPTY_STR   	"EMPTY"

extern char tmp[MAXMAXLEN];

extern type_key* findExact(mode *mode, char *key);

extern mode       *modes;
extern mode       *currentMode;
extern mode 	  *defMode;
extern type_alias *aliases;
extern type_key   *alarms;
extern CONF       conf;

/* ----------------- Interface functions ------------------------- */

int getLog()
{
	return (conf.log >  0 ? 1 : 0);
}

int getDebug()
{
	return (conf.log == 2 ? 1 : 0);
}

char* getDevice()
{
	return conf.device;
}

int getBaudrate()
{
	return conf.baudrate;
}

int getRetrySecs()
{
	return conf.retrysecs;
}

int autoConnect()
{
	return conf.autoConnect;
}

int getAutoRepeat()
{
	return conf.autoRepeat;
}

int getUseScreen()
{
	return conf.screen;
}

int getFrontEnd()
{
	return conf.frontEnd;
}

int getHttp()
{
	return conf.http;
}

char* getCharset()
{
	return (conf.charset[0] == '\0' ? NULL : conf.charset);
}

char* getCfgDir(void) 
{
	return conf.cfgDir;
}

char* getToMainMenu()
{
	return  (conf.toMain[0] == '\0' ? NULL : conf.toMain);
}

char* getAT_CMER(int what)
{
	if (what == CMER_ON) {
		if  (conf.cmerOn[0] == '\0') {
			if (conf.model == MODEL_MOTOROLA) {
				return DEF_AT_CMER_ON_MOTOROLA;
			} else if (conf.model == MODEL_SE) {
				return DEF_AT_CMER_ON_SE;
			} else if (conf.model == MODEL_SAGEM) {
				return DEF_AT_CMER_ON_SAGEM;
			} else if (conf.model == MODEL_SIEMENS) {
				return DEF_AT_CMER_ON_SIEMENS;
			} else {
				return DEF_AT_CMER_ON_DEFAULT;
			}
		} else {
			return conf.cmerOn;
		}
	}

	if (what == CMER_OFF) {
		if  (conf.cmerOff[0] == '\0') {
			if (conf.model == MODEL_MOTOROLA) {
				return DEF_AT_CMER_OFF_MOTOROLA;
			} else if (conf.model == MODEL_SE) {
				return DEF_AT_CMER_OFF_SE;
			} else if (conf.model == MODEL_SAGEM) {
				return DEF_AT_CMER_OFF_SAGEM;
			} else if (conf.model == MODEL_SIEMENS) {
				return DEF_AT_CMER_OFF_SIEMENS;
			} else {
				return DEF_AT_CMER_OFF_DEFAULT;
			}
		} else {
			return conf.cmerOff;
		}
	}
	return NULL;
}

char *getMode (void) 
{
	return (currentMode == NULL ? NULL : currentMode->name);
}

void setModel (char *answer) 
{
	if (answer == NULL) {
		conf.model = MODEL_DEFAULT;
	} else if (strstr(answer,STR_MOTOROLA)) {
		conf.model = MODEL_MOTOROLA;
	} else if (strstr(answer,STR_SE)) {
		conf.model = MODEL_SE;
	} else if (strstr(answer,STR_SAGEM)) {
		conf.model = MODEL_SAGEM;
	} else if (strstr(answer,STR_SIEMENS)) {
		conf.model = MODEL_SIEMENS;
	} else {
		conf.model = MODEL_DEFAULT;
	}
}

int getModel (void) 
{
	return conf.model;
}

char* getAlarmInfo(int which, cmdItem ** cmds)
{
	int i;
        type_key *tk = alarms;
        
	for (i=0;i<which;i++) {
        	if (tk != NULL) {
        		tk = tk->next;
                } else {
                	break;
                }
        }
        
        if (i<which || tk == NULL) {
        	return NULL;
        }
    	*cmds = tk->cmd;
    
 	return tk->key;
}

/////////////////////////////////////////////////////////////////////////////////////////////////

char* findAlias(char *keyIn)
{
	type_alias* al = aliases;
	char key[MAXARGLEN];

	if (*keyIn == '\0') {
		strcpy(key, EMPTY_STR);
	} else {
	 	strncpy(key, keyIn, MAXARGLEN - 1);
	}

	while (al) {
		if (strcmp(key, al->key) == 0) {
			sprintf(tmp,"found alias %s=%s", key, al->alias);
			logger("DEBUG",tmp);   
	
			return al->alias;
		}
		al = al->next;
	}
	return NULL;
}

static type_key* findStartingWith(mode *mode, char *key)
{
        if (mode == NULL) {
        	return NULL;
        }
	type_key* It = mode->keys;

	while (It && It->key != NULL) {
		char *start = strstr(It->key,key);

		if (start && start == It->key) {    // We got a part of multi-key or parametrized command
			
			logger("DEBUG","Found part of multi-key or parametrized command");
			return It;
		} 
		It = (type_key*) It->next;
	}    

	return It;
}


static type_key* findItemInMode(mode *mode, char *key, int *flag, cmdParams *params) 
{
	if (mode == NULL) {
		logger("DEBUG","findItemInMode: mode is null");
		return NULL;
	}

	sprintf(tmp,"findItemInMode >%s,%s<", mode->name, key);
	logger("DEBUG",tmp);

	// Prepare to search as parametrized command, control presence of both ( and )
	char *start  = index(key,'(');
	char *finish = rindex(key,')');

	if ((start != NULL && finish == NULL) ||  // Command was read partially?
            (start == NULL && finish != NULL)) {  // Command was incorectrly formed ?
		logger("DEBUG","findItemInMode: Incorrectly formed parametrized command. One brace is absent");
		return NULL;
	}
        
	// Clean-up
	if (params != NULL) {
		params->index[0] = '\0';
		params->value[0] = '\0';
	}

	type_key* It = findExact(mode,key);	// Search exact command
	if (It) {
		//logger("DEBUG","Found exact command");
		*flag = FLAG_EXACT;
		return It;
	} 

	It = findStartingWith(mode,key);	// Search as part of multikey sequence
	if (It) {
		//logger("DEBUG","Found part of multi-key command");
		*flag = FLAG_MULTIKEY;
		return It;
	} 

	if (params == NULL) {
		logger("DEBUG","findItemInMode: No parameters suspected. Item does not found.");
		return NULL;
	}

	// Search as parametrized command

	if (start != NULL && finish != NULL && start != key) {        // Do not match "(...)=..."
		It = mode->keys;
	
		char tag  [MAXARGLEN];
		char index[4];
		char value[MAXARGLEN];
	
		strncpy(tag, key, start-key);
		*(tag+(start-key)) = '\0';
	
		char *comma = strstr(start+1,",");
		if (comma == NULL || comma > finish) { // Parametrized command from Java Client should be in a form like List(1,String1)
			logger("DEBUG","findItemInMode: Incorrectly formed parametrized command");
			return NULL;
		}

		if (comma-start > 5) {
			logger("ERROR","findItemInMode: Received incorrect index!");
			return NULL;
		}
		strncpy(index,start+1,comma-start-1);
		*(index+(comma-start-1)) = '\0';

		strncpy(value,comma+1,finish-comma-1);
		*(value+(finish-comma-1)) = '\0';

		sprintf(tmp,"Parametrized command parsed as >%s< >%s< >%s< ", tag,index,value);
		logger("DEBUG",tmp);
	
		// Try to search explicitly specified parametrized command (for ex List(1) or List(commandX))
		// By index
		strcat(tag,"(");
		strcat(tag,index);
		strcat(tag,")");
		It = findExact(mode,tag);

		if (It) {
			logger("DEBUG","Found exact (explicitly specified by index) parametrized command");
			*flag = FLAG_EXACT;
			return It;
		} 

		// Then by value
		*(tag+(start-key+1)) = '\0';
		strcat(tag,value);
		strcat(tag,")");
		It = findExact(mode,tag);

		if (It) {
			logger("DEBUG","Found exact (explicitly specified by value) parametrized command");
			*flag = FLAG_EXACT;
			return It;
		} 
	
		// Finally... Search command like ListItem($$)
		*(tag+(start-key)) = '\0';
		
		char* keyAndBrace = calloc(1,strlen(tag)+2);
		strcpy(keyAndBrace,tag);
		strcat(keyAndBrace,"(");
		
                //sprintf(tmp,"Search findStartingWith() >%s< ", keyAndBrace);
                //logger("DEBUG",tmp);
                DEBUG2("Search findStartingWith() >%s< ", keyAndBrace);
                
		It = findStartingWith(mode,keyAndBrace);
		free(keyAndBrace);
		if (It) {
	                //sprintf(tmp,"Found parametrized command >%s,%s< ", index,value);
               		//logger("DEBUG",tmp);
	                DEBUG2("Found parametrized command >%s,%s< ", index,value);

			strcpy(params->index, index);
			strcpy(params->value, value);
	
			*flag = FLAG_PARAMETR;
			return It;
		} 
	}

	logger("DEBUG","findItemInMode: Item does not found");
	return NULL;
}

type_key* findItem(char *keyIn, int *flag, cmdParams *params) 
{
	//logger("DEBUG","findItem()"); 
	type_key *tk = NULL;

	if (keyIn == NULL) {
        	return NULL;
        }

	char key[MAXARGLEN];

	if (*keyIn == '\0') {
		logger("INFO","Got empty key");   
		strcpy(key, EMPTY_STR);
	} else {
		strncpy(key, keyIn, MAXARGLEN - 1);
	}
	 
	if (currentMode != NULL) {
		tk = findItemInMode(currentMode, key, flag, params);
		if (tk == NULL && currentMode != defMode) {
			logger("DEBUG","findItem() search in default");
			tk = findItemInMode(defMode, key, flag, params);
		}
	}
	//logger("DEBUG","findItem() EXIT");
        /*if (tk && tk->cmd) {
        	printf("TRACE %s->%s:%s \n",keyIn,tk->cmd->descr,(tk->cmd->exec ? tk->cmd->exec : "NULL"));
        } else {
        	printf("TRACE %s->NOT FOUND?\n",keyIn);
        }*/
	return tk;
}

cmdItem *getCommand(type_key* item)
{
	if (!item) {
		logger("DEBUG","getCommand(): Empty type_key was got");
		return NULL;
	}
	if (item->key == NULL) {
		logger("DEBUG","Empty key was got");
		return NULL;
	}

	sprintf(tmp,"getCommand for >%s<", item->key);
	logger("DEBUG",tmp);   

	return item->cmd;
}

//
// Print cfg stuff
//

void printKeysInMode(type_key* It)
{    
        while (It && It->key != NULL) {
        	if (It->cmd) {
    			sprintf(tmp, "\t%s = ", It->key);
                        
        		cmdItem* item = It->cmd;
        		while (item) { 
        			
                                strcat(tmp, id2Cmd(item->type));
                                
                                if (item->tdata || item->descr || item->exec) {
        				strcat(tmp, "(");
                                }
                                
                                int pComma = 0;
        			if (item->descr) {
                                	//printf("descr  %s\n",item->descr);
        				strcat(tmp, item->descr);
					pComma = 1;
                                }
                                if (item->tdata ) {
                                        char iBuff[16];
                                        sprintf(iBuff,",%d",item->tdata->timeout);
                               		strcat(tmp, iBuff);
                                        sprintf(iBuff,",%d",item->tdata->times);
                               		strcat(tmp, iBuff);
                                        strcat(tmp, ",");
                                }
        			strcat(tmp, "<|>");
                                if (item->exec) {
                                        if (pComma) {
						strcat(tmp, ",");
        				}
					strcat(tmp, item->exec);
                                }
                                
                                if (item->tdata || item->descr || item->exec) {
        				strcat(tmp, ")");
        			}
        			strcat(tmp, ";");
        			item = item->next;
                                
                                if (item) {
        				strcat(tmp, "\\");
                                }
                                logger("CFG", tmp);
                                sprintf(tmp, "\t\t");
        	
        		}
    		} else {
        		sprintf(tmp, "\t%s = no command", It->key);
                        logger("CFG", tmp);
        	}    
	
		It = (type_key*)It->next;
        }
        return;
}

void printKeys()
{    
	mode *mp = modes;
	while (mp) {
		
		sprintf(tmp, "%s %s", MODE_STR, mp->name);
		logger("CFG", tmp);
		 
		type_key* It = mp->keys;
	
		printKeysInMode(It);

		sprintf(tmp, "%s %s", MODE_END_STR, mp->name);
		logger("CFG", tmp);
	
		mp = (mode*)mp->next;
	}

	return;
}

void printAlarms()
{
        logger("CFG", ALARMS_SECTION_STR);

	printKeysInMode(alarms);

        logger("CFG", SECTION_END_STR);
	return;
}

void printAliases()
{
	type_alias *al = aliases;

	logger("CFG", ALIAS_SECTION_STR);

	while(al) {
		sprintf(tmp, "\tKey %s \taliased to %s", al->key, al->alias);
		logger("CFG", tmp);

		al = al->next;
	}
	logger("CFG", SECTION_END_STR);

	return;
}

void printParams()
{
	int atMode = 1;
	if (strstr(conf.device, INET_SOCKET) != NULL || 
	    strstr(conf.device, BT_SOCKET) != NULL) {
		atMode = 0;
	}

	logger("CFG", "[Parameters]");
	if (atMode) {
		sprintf(tmp, "\tAutoConnect   %s", (conf.autoConnect?"Yes":"No"));
		logger("CFG", tmp);
	}
	sprintf(tmp, "\tAutoRepeat    %s", (conf.autoRepeat?"Yes":"No"));
	logger("CFG", tmp);
	if (atMode) {
		if (conf.cmerOn[0] != '\0') {
			sprintf(tmp, "\tCMER On       %s", conf.cmerOn);
			logger("CFG", tmp);
		} else {
			logger("CFG", "\tCMER On	   Will be determined automatically");
		}
		if (conf.cmerOff[0] != '\0') {
			sprintf(tmp, "\tCMER Off      %s", conf.cmerOff);
			logger("CFG", tmp);
		} else {
			logger("CFG", "\tCMER Off	   Will be determined automatically");  
		}
	}
	sprintf(tmp, "\tDevice        %s", conf.device);
	logger("CFG", tmp);
	sprintf(tmp, "\tLog	      %s", (conf.log == 2?"Debug":(conf.log == 1?"Yes":"No")));
	logger("CFG", tmp);
	sprintf(tmp, "\tRetry Sec     %d", conf.retrysecs);
	logger("CFG", tmp);
	sprintf(tmp, "\tScreen Menu   %s", (conf.screen?"Yes":"No"));
	logger("CFG", tmp);
	if (atMode) {
		sprintf(tmp, "\tTo Main Menu  %s", conf.toMain);
		logger("CFG", tmp);
	}
	if (conf.frontEnd >= 0) {
		sprintf(tmp, "\tWorks as back end");
		sprintf(tmp, "\tPort          %d", conf.frontEnd);
		logger("CFG", tmp);
	}
	logger("CFG", SECTION_END_STR);
} 

void printConf() 
{
        sprintf(tmp, "anyRemote %s", AR_VERSION);
        logger("CFG", tmp);
	printKeys   ();
	printAliases();
	printAlarms ();
	printParams ();

	return ;
}

//
// Clean-up stuff
//

void freeCmds(cmdItem* item)
{
	cmdItem* mi;
        while (item) {
        	mi = item->next;
        	if (item->descr != NULL) {
        		free(item->descr);
        		item->descr = NULL;
        	}
         	if (item->exec != NULL) {
        		free(item->exec);
        		item->exec = NULL;
        	}
         	if (item->tdata != NULL) {
        		free(item->tdata);
        		item->tdata = NULL;
        	}
		
		free(item);
        	item = mi;
        }
	return;
}

void freeCfgEx(mode* pmodes, type_key* palarms, type_alias *paliases)
{
	//logger("DEBUG", "freeCfgEx()");
	
	// Keys
	while(pmodes) {
		mode	 *md = pmodes;
		type_key *mk = md->keys;
	
		while (mk) {
			type_key* It = mk;
			mk = (type_key*) mk->next;
			
                        freeCmds(It->cmd);
 
 			It->cmd = NULL;
                        free(It->key);
			free(It);
		}
		pmodes = (mode*) pmodes->next;
		free(md);
	}

	// Alarms
        type_key *mk = palarms;
	while (mk) {
		type_key* It = mk;
		mk = (type_key*) mk->next;
		
                freeCmds(It->cmd);

		It->cmd = NULL;
                free(It->key);
		free(It);
	}

	// Aliases
	while(paliases) {
		type_alias *al = paliases->next;
		free(paliases); paliases = NULL;
		paliases = al;
	}
}

void freeCfg(void)
{
	freeCfgEx(modes, alarms, aliases);
        free(conf.cfgDir);
}
