//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006,2007,2008 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <unistd.h>

#include "atsend.h"
#include "btio.h"
#include "cmds.h"
#include "conf.h"
#include "common.h"
#include "parse.h"
#include "utils.h"
#include "xemulate.h"

#define TMPDISCONN_TIME 60
#define ALARM_CLEAN 	0
#define ALARM_CHECK 	1

static void mainRoutine     (void);
static int  isBemusedCommand(char *cmd);
static int  isLircCommand   (char *cmd);

char tmp	    [MAXMAXLEN];
char callerId	    [MAXLEN];
char modifierString [MAXARGLEN];
char frontEndBuf    [MAXLEN];

int  alarmTimer   = 0;
int  repeatTimer  = 0;
int  callTimer	  = 0;
int  commandTimer = 0;
int  feTimer      = 0;

int  useCallId    = 1;
int  repeatNow    = 0;
type_key*  repeatCmdPtr = NULL;
int  callOn       = 0;

int  gotSignal     = 0;

int  serverMode   = CLIENT_AT;  // By default use AT commands and devices in /dev

int  port         = 0;
int  initialized  = 0;
int  isConnected  = 0;
int  flushConf    = 0;

int openPort(char * portIn) 
{
	logger("INFO", "openPort()");
        
        char* strPort;
        int ret = 0;

        // device should be in format:
        // 1. (AT mode)     /dev/something							used with AT mode commands
        // 2. (Server mode) socket:NNNN or 							used with java client
	//                  socket:/path/to/socket						used with java client
        // 3. (Server mode) bluetooth:NN or bluetooth (port will be = DEFAULT_BT_CHANNEL) 	used with java client
        // 4. local:/some/path									used with java client, like 1, but no AT commands
        // 5. ilirc:/dev/something 								like 1, but no AT commands
        // 6. stdio								used with java client, like 1, but no AT commands

        if ((strPort = strstr(portIn, INET_SOCKET)) != NULL) {
        	
                strPort += strlen(INET_SOCKET);

		// check is it port or unix socket
		char * ch = strPort;
		int isPort = 1;
		while (*ch != '\0') {
			if (isdigit(*ch)) {
				ch++;
			} else {
				isPort = 0;
				break;
			}
		}
		
		if (isPort) {
                	port = atoi(strPort);

                	DEBUG2("TCP Server mode. Use port %d\n", port);

                	if ((ret = openSocketPort(SERVER_TCP, port, NULL)) < 0) {
                		return -1;
                	}

                	serverMode = SERVER_TCP;
		} else {

                	/*sprintf(tmp, "Unix socket Server mode. Use socket %s\n", strPort);
                	logger("DEBUG", tmp);
		
                	if ((ret = openSocketPort(SERVER_UX, -1, strPort)) < 0) {
                		return -1;
                	}
			
			serverMode = SERVER_UX;
			*/
			printf("ERROR:incorrect port\n");
                        return -1;
			
		}
        } else if ((strPort = strstr(portIn, ILIRC_SOCKET)) != NULL) {
        	
                strPort += strlen(ILIRC_SOCKET);

                DEBUG2("Unix socket client mode. Use port %s\n", strPort);

                if (unix_open_port(strPort) < 0) {
			printf("ERROR: open inputlircd socket\n");
                	return -1;
                }

                serverMode = CLIENT_ILIRC;

        } else if ((strPort = strstr(portIn, BT_SOCKET)) != NULL) {

        	strPort += strlen(BT_SOCKET);
        
                if (strstr(strPort, ":") == NULL) { // just "bluetooth
                	port = DEFAULT_BT_CHANNEL;
                } else {
                	strPort++;
                        port = atoi(strPort);
                }
                DEBUG2("Bluetooth Sockets Server mode. Use channel %d", port);

                if ((ret = openSocketPort(SERVER_BT, port, NULL)) < 0) {
                	return -1;
                }
                serverMode = SERVER_BT;

        } else if ((strPort = strstr(portIn, UNIX_SOCKET)) != NULL){

		strPort += strlen(UNIX_SOCKET);

        	DEBUG2("Serial Client mode (no AT). Use device %s", strPort);

                if (bt_open_port(strPort) < 0) {
                	printf("ERROR: open device\n");
                        return -1;
                }
                serverMode = CLIENT_NOAT;
		
        } else if ((strPort = strstr(portIn, STDIN_STREAM)) != NULL){

                serverMode = SERVER_STDIN;
                openStdin();

        } else {	//if ((strPort = strstr(portIn, AT_DEVICE)) != NULL){

        	DEBUG2("Serial Client mode. Use device %s", portIn);

                if (bt_open_port(portIn) < 0) {
                	printf("ERROR: open RFCOMM port\n");
                        return -1;
                }
                serverMode = CLIENT_AT;
        }

        if (initialized == 0) { 
        	// Run "onInit" command
                int i = 0;
                type_key *k = findItem(EVT_INIT, &i, NULL);
                if (k && i == FLAG_EXACT) {
                	logger("INFO", "Exec cmd on init");
                        handleCmdByKey(k,NULL);  // Since we not connected yet we did not interested in return code
                }
                initialized++;
        }
        return 1;
}

void createDataDir() 
{
	char dd[542];
	char *t = getenv("HOME");
	strcpy(dd, t);
	strcat(dd, "/.anyRemote");

	DIR *d = opendir(dd);
	if (d) {
        	closedir(d);
        } else {
		mkdir(dd, S_IRWXU);
	}
}

// Then port will be closed from a forked child use 0
void closePort(int final) 
{
	if (serverMode == SERVER_BT  || 
            serverMode == SERVER_TCP || 
            serverMode == SERVER_UX) {
         	closeSocketPort(serverMode);
        } else if (serverMode == CLIENT_ILIRC) {
        	if (unix_close_port() < 0) {
                	printf("Error on closing AF_UNIX socket\n");
                }
        } else if (serverMode == CLIENT_AT || serverMode == CLIENT_NOAT) {
        	if (bt_close_port(final) < 0) {
                	printf("Error on closing port\n");
                }
        } else if (serverMode == SERVER_STDIN && final) {
        	closeStdin();
        }

	if (getFrontEnd() > 0) {
		if (final)  {
			sendToFrontEnd("Exiting");
                }
                // else - just close socket from forked child
		disconnectFrontEnd();
	}
}

static void freeAndExit(void) 
{
	int i;
	type_key *k = findItem(EVT_EXIT, &i, NULL);
	if (k && i == FLAG_EXACT) {
		logger("INFO", "Exec cmd on exit");
		handleCmdByKey(k,NULL);   // Since we exiting now we did not interested in return code
	}
	
	freeDisplay();
	freeTimers();
	freeVars();
	freeCachedData();
	freeCfg();
	exit(0);
}

void sig_handle() {

	if (bt_getfd() > 0 && 
	    (serverMode == SERVER_BT  || 
	     serverMode == SERVER_TCP || 
	     serverMode == SERVER_UX)) {
	    
		writeSocketConn("Set(disconnect);", 16);
		gotSignal = 1;
	} else {	// CLIENT_AT, CLIENT_NOAT, SERVER_STDIN
		freeAndExit();
	}
}

// handle signals only once (it is enough to send disconnect message to java client)
void sig_int() {
	signal(SIGINT, SIG_DFL);   
	printf("\nProcess %d: got SIGINT\n",getpid());
	sig_handle();
}

void sig_quit() {
	signal(SIGQUIT, SIG_DFL);
	printf("\nProcess %d: got SIGQUIT\n",getpid());
	sig_handle();
}

void sig_kill() {
	signal(SIGKILL, SIG_DFL);
	printf("\nProcess %d: got SIGKILL\n",getpid());
	sig_handle();
}

int main(int argc, char *argv[])
{
	int i;
        for (i=1; i<argc; i++) {
        	if (strcmp(argv[i],"-f")==0 && i+1<argc) {
                	break;
                }
        }
	
        createDataDir();
	
        if (argv[1] && (strcmp(argv[1],"-h")==0 || strcmp(argv[1],"--help")==0)) {
		 //Just print help and exit
		 printHelp();
	         exit(0);
        } else if (argv[1] && (strcmp(argv[1],"-v")==0 || strcmp(argv[1],"--version")==0)) {
		printf("anyremote %s\n", AR_VERSION); 
		exit(0);
        }

        if (init_cfg(i<argc ? argv[++i] : NULL) != EXIT_OK) {
        	printf("Can not initialize configuration. Exiting.\n");
		
		printHelp();
                
		freeCfg();
                exit(1);
        }
                    
        if (parse_opts(argc, argv)!=1 ) {
        	printf("Invalid command line option used. Exiting.\n");
                freeCfg();
                exit(1);
        }
        
        initLog();
        
        // Not to core dump if connection will close
        signal(SIGPIPE, SIG_IGN);   
        signal(SIGCHLD, SIG_IGN);   

    	signal(SIGINT, sig_int);
    	signal(SIGQUIT,sig_quit);
   	signal(SIGKILL,sig_kill);
    
        // main cycle
        mainRoutine();

        return 0;
}

int handle_key_press(char* keyIn,int isRepeatable) 
{
	int ret;

        DEBUG2("handle_key_press() >%s<", keyIn);

        // Search an alias
        char *key = NULL;
        if((key = findAlias(keyIn)) == NULL) {
        	key = keyIn;
        }    
	
	if (getFrontEnd() > 0) {
		sendToFrontEnd(key);
	}

        // If modifier was set already it needs to verify which sequence we go
        if (modifierString[0] != '\0') {
        	strcat(modifierString," "); // Add one space first
                strcat(modifierString, key);
                DEBUG2("Modifier+sequence is: >%s<", modifierString);
        }

        // Execute appropriate command
        int i;
        cmdParams params;
        type_key *k = findItem((modifierString[0] == '\0' ? key : modifierString), &i, &params);

        if (k && i == FLAG_EXACT) {
        	if (isRepeatable && (!repeatNow) && getAutoRepeat()) {
                	logger("DEBUG", "Set auto repeat flag");
                        repeatNow    = 1;
                        repeatCmdPtr = k;
                } 
                ret = handleCmdByKey(k, NULL);
                if (ret == EXIT_ABORT || ret == EXIT_DISCON) { 
			logger("DEBUG", "handle_key_press EXIT_ABORT || EXIT_DISCON");
                	return ret;
                }
                    
                // Clean modifier string
                modifierString[0] = '\0';
                
         	if (flushConf == 1) {
                	logger("DEBUG", "Flush old configuration");
        		flushConf = 0;
                
                	flushOldConf();
        	}
                
        } else if (k && i == FLAG_MULTIKEY){
        	logger("DEBUG", "Got part of multi keys sequence. Nothing to do.");
                if (modifierString[0] == '\0') {
                	logger("DEBUG", "Start of multi key sequence");
                        strcpy(modifierString, key);
                } 

                if(serverMode == CLIENT_AT) {
                	ret = sendSeq(getToMainMenu());
                        if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
                        	return ret;
                        }
                }
        } else if (k && i == FLAG_PARAMETR){
                logger("DEBUG", "Got parametrized command");
        
                ret = handleCmdByKey(k, &params);
                
                // Clean modifier string
                modifierString[0] = '\0';
		
                if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
                	 return ret;
                }
            
        } else {
        	// User had pressed some key ... 
                // Send ToMainMenu sequence of CKPD's to show main menu again
                if (serverMode == CLIENT_AT && sendSeq(getToMainMenu()) != EXIT_OK) {
                	logger("DEBUG","Fails in sendSeq()");
                }

                logger("DEBUG", "No approprite key definition was found by findItem()");
                if (modifierString[0] != '\0') {
                	// Clean modifier string
                        modifierString[0] = '\0';
                        logger("DEBUG", "Clean modifier string");
                }
        }

        return EXIT_OK;
}

void cleanAutoRepeatFlag() 
{
	// Clear auto repeat flag
	logger("DEBUG","Clean auto repeat flag");
	repeatNow    = 0;
	repeatCmdPtr = NULL;
	repeatTimer  = 0;
}

int handle_command(char* cmd) 
{
	char *prev, *next, *key;
	int ret;

	if (cmd == NULL) {
		return EXIT_OK;
	}
	
	//skip lines starting with \n and \r
	if (cmd[0] == '\r') cmd++;
	if (cmd[0] == '\n') cmd++;


	// most common case
	if (!cmd[0]) {
		//logger("DEBUG", "handle_command() skip 1");
		return EXIT_OK;
	}
        
	DEBUG2("handle_command() >%s<", cmd);

	// if recieved multiline command - handle line by line and return
	prev = cmd;
	next = strchr(cmd, '\r');
	if (next == NULL && serverMode != CLIENT_AT) {	// Java client will send +CKEV: 1,1; +CKEV: 1,0
		next = strchr(cmd, ';');
	}
	if (next) {
		logger("DEBUG", "handle_command() multiline");

		char copy[1024];
		int len;

		do {	 
			len = next-prev;
	
			if (len > 2) {
				memcpy(copy, prev, len);
				copy[len] = 0;
			
				// use recursion
				ret = handle_command(copy);
				if (ret == EXIT_DISCON || ret == EXIT_ABORT) {
					return ret;
				}
			}
			prev = next+1;
			next = strchr(prev, '\r');
			if (next == NULL && serverMode != CLIENT_AT) {	// Java client will send +CKEV: 1,1; +CKEV: 1,0
				next = strchr(prev, ';');
			}
		} while (next) ;

		// now return
		return EXIT_OK;
	}

	if (IS_OK(cmd)) {   // OK - nothing to do
		return EXIT_OK;
	}

	if (memcmp(cmd, DEF_CLDEBUG, 12) == 0) {	    // Got debug event from Java client

		return EXIT_OK;
                
	} else if (memcmp(cmd, DEF_MSG, 4) == 0) {	    // Got event from Java client
		
		key = cmd+4;
				
		logger("DEBUG","Got event from client");
		return handle_key_press(key,0);

	} else if (memcmp(cmd, DEF_CKEV, 6) == 0) {
		    	
		// Handle different keypresses
		// Motorola: +CKEV: "1",1
		// SE	   : +CKEV: 1,1
		// Sagem: in rare ) cases it could be (between > and <) >,1:< ?

		int isSE = MODEL_SE;

		key = cmd+7;

		// Sagem test
		// key+=3;

		if (*key == '"') {	// Seems this is Motorola
			key++;
			//logger("DEBUG","+CKEV is in Motorola format");
			isSE = MODEL_MOTOROLA;
		}

		if (*key == ',') {	// Seems this is Sagem
			logger("DEBUG","+CKEV is in rare Sagem format (Empty code)");
			isSE = MODEL_SAGEM;
		}

		int i = 1;

		if (isSE == MODEL_SE) { 	// SonyEricsson & default
			while(key[i] != ',') i++;
			key[i] ='\0';
		
			i = i+1;
		} else if (isSE == MODEL_MOTOROLA) {	// Motorola
		    while(key[i] != '"') i++;
		    key[i] ='\0';

		    i = i+2;
		} else if (isSE == MODEL_SAGEM) {	// rare Sagem case
			*key ='\0';
		} else {
			logger("ERROR","Can't recognize +CKEV event !!!");
			return EXIT_NOK;
		}

		if (key[i] == '1') {
			//logger("DEBUG","Button down event");
			return handle_key_press(key,1);
		} else if (key[i] == '0') {
			//logger("DEBUG","Button up event");
			// In general we skip this
			if (repeatNow) {
				cleanAutoRepeatFlag();
			}
			return EXIT_OK;
		} else {
			DEBUG2("something wrong: key should be pressed or released: cmd=>%s< char=>%c<",cmd, key[i]);
			return EXIT_NOK;
		}
	} else if (memcmp(cmd, DEF_CLCC,6) == 0) {
	
		//logger("INFO","Caller ID received. Skip it.");
		return EXIT_OK;
		
	} else if (strstr(cmd, DEF_RING) != NULL) {	// Incoming call
		logger("INFO","Incoming call"); 	// This event sent periodically until used answered a call

		if (callOn == 0) {
			if (useCallId) {
				//logger("DEBUG", "Get CallerId on incoming call");
				ret = getClip(callerId);

				INFO2("CallerId is %s",callerId);
		
				if(ret == EXIT_EXACT) { // Active call exists
				//	logger("DEBUG","Caller ID received");
				//} else { 
					logger("DEBUG","Can't get Caller ID. Probably call was finished");
				}
			}
			//
			// May be we can get +CLIP: "+xxxxxxxxxxxx",145,,,,0< here ?
			// 

			int i = 0;
			type_key* k = findItem(EVT_INCALL, &i, NULL);

			if (k && i == FLAG_EXACT) {
				logger("DEBUG", "Exec cmd on incoming call");
				ret = handleCmdByKey(k,NULL);
				if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
					return ret;
				}
			}

			// If there is (EndCall) command,then set timer to determine when call ends
			i = 0;
			k = findItem(EVT_ENDCALL, &i, NULL);
			if (k && i == FLAG_EXACT) {
				logger("DEBUG", "Set call timer");
				callTimer = 0;
				callOn    = 1;
			}
		
			return EXIT_OK;
		} else {
			logger("DEBUG","Skip it."); // This event sent periodically until used answered a call
		}
	} else if (isLircCommand(cmd)) {	    // LIRC tricks
		return EXIT_OK;
	} else if (isBemusedCommand(cmd)) {	    // Bemused server emulation tricks
		return EXIT_OK;
	} else if (serverMode == SERVER_STDIN) {    // Just direct input, try it
        	//printf("DEBUG: Is it SERVER_STDIN ? %s\n", cmd);
                
                char *p = cmd;
        	while (*p != '\0') {
                	if (*p == '\n') {
                        	*p = '\0';
                                break;
                        }
                        p++;
                }

		return handle_key_press(cmd,1);
	}

	DEBUG2("Unhandled cmd %s", cmd);
	return EXIT_OK;
}

static int read_command(char* buf, int l) 
{
	int ch, len = 0;
	//printf("read_command\n");
	
	if (serverMode == SERVER_STDIN) {
		/* this blocks any other operation
                char* r = fgets((char*)buf, l, (FILE*)stdin);
	        printf("read_command SERVER_STDIN %d\n", len);
		if (r == NULL) {
			buf[0] = 0;
			printf("read_command SERVER_STDIN empty\n");
                	return 0;
		} else {
		        len = strlen(buf);
			printf("read_command SERVER_STDIN %d\n", len);
		}
                */

                int ret = readStdin(buf,l);
		if (ret == 0) {
			buf[0] = 0;
                	return 0;
		} else {
		        len = strlen(buf);
                        DEBUG2("read_command SERVER_STDIN %d %s\n", len, buf);
                }
	} else {

        	char* p = buf;

        	while (len < l) {
        		char c;
        		ch = bt_readchar(&c,100);
		
                	if (ch == EOF) {
                		buf[0] = 0;
                        	return EOF;
                	} else if (ch == EOF-1) {
                		break;
                	} else if (ch >= 0 && (c == '\r' || c == ';')) {
               			break;
                	} else  if (ch >= 0 && c != '\r' && c != ';') {
                		*p++ = c;
                        	len++;
			}
        	}
	}
        buf[len] = '\0';
        return len;
}

// mode: 0 - clear alarms
//       1 - if fired - process, then clear alarms
void manageAlarms(int mode)
{
	char    *pf;
        cmdItem *pg;
        int i;
        struct stat buf;
            
        i = 0;
        while (1) {
        	
                //sprintf(tmp, "Call getAlarmInfo %d", i);
                //logger("DEBUG",tmp);
                pf = getAlarmInfo(i, &pg);

                if (pf == NULL) {
                        //logger("DEBUG", "After getAlarmInfo ALL PROCESSED");
                        break;
                }
                //sprintf(tmp, "After getAlarmInfo %s", pf);
                //logger("DEBUG", tmp);
                i++;
        
                // If file exists
                int fs = stat(pf, &buf);
                if (fs == 0) {
                	if (mode == ALARM_CLEAN) {
                        	logger("DEBUG","Remove file");
                        } else {            
                        	INFO2("Alarm %s fired", pf);
                                processCommands(pg, NULL);
                        }
                        remove(pf);
                }
        }
        return;
}

int initPort()
{
	char answer[1024];
        char cmd[32];

        logger("DEBUG","initPort");

        if (serverMode == SERVER_BT  || 
	    serverMode == SERVER_TCP || 
	    serverMode == SERVER_UX) {	// Wait for incoming connection
                logger("INFO", "Server mode: Waiting connection");
                return listenAndAcceptSocketConn();
        } 

	if (serverMode == CLIENT_ILIRC || 
	    serverMode == CLIENT_NOAT  ||
	    serverMode == SERVER_STDIN) {
		logger("INFO", "Unix client: init OK");
		return 1;	// nothing to init
	}
	
        answer[0] = '\0';
        sendSyncCommand("ATZ\r", answer, sizeof(answer), NULL);
        //sendSyncCommand("AT\r", answer, sizeof(answer), NULL);
        if(!IS_OK(answer)) {
        	printf("ERROR: AT -> %s\n", answer);
                return -1;
        }
	
        sendSyncCommand("ATE0\r", answer, sizeof(answer), NULL);
        if(!IS_OK(answer)) {
        	printf("ERROR: ATE0 -> %s\n", answer);
                return -1;
        }
        char* ptr = getCharset();
        if (ptr) {
        	sprintf(cmd,"%s\"%s\"\r", DEF_AT_CSCS, ptr); 
                sendSyncCommand(cmd, answer, sizeof(answer), NULL);
                if(!IS_OK(answer)) {
                	printf("ERROR: Can't set charset to %s\n", ptr);
                }     
        }

        // Determine model
        sprintf(cmd,"%s\r", DEF_AT_CGMI); 
        sendSyncCommand(cmd, answer, sizeof(answer), NULL);

        //Set model in conf. data
        setModel(answer);
	
	if (getModel() == MODEL_MOTOROLA) {	// Motorola RIZR Z3 needs to set MODE=2 to allow AT+CLIP command
		// do not care about responce
		sendSyncCommand("AT+MODE=2\r", answer, sizeof(answer), NULL);
	}
	
        sprintf(cmd,"%s\r", DEF_AT_CLIP); 
        sendSyncCommand(cmd, answer, sizeof(answer), NULL);
        if(!IS_OK(answer)) {
            printf("ERROR: Can't set CLIP ON\n");
        }


	int ret = sendCMER(CMER_ON);
        if(ret != EXIT_OK) {
        	printf("ERROR: fails in set event reporting on");
                return -1;
        }

        // Siemens S55 needs additional AT+CMEC=2 to make AT+CKPD works
        // not sure about other Siemens phones
        if (getModel() == MODEL_SIEMENS) {
        	sprintf(cmd,"%s\r", DEF_AT_CMEC); 
                sendSyncCommand(cmd, answer, sizeof(answer), NULL);
                if(!IS_OK(answer)) {
                	printf("ERROR: ON AT+CMEC\n");
                }
        }

        // Will set global callerId [MAXLEN];
        ret = getClip(callerId);
        if(ret == EXIT_NOK) {       // Got ERROR on AT+CLCC; probably phone does not supports this command
        	printf("ERROR: fails in getClip\n");
                useCallId = 0;
        }
        return 1;
}


static void mainRoutine(void) 
{
	char answer[MAXCMDLEN];
	int ret = EXIT_OK;
	int retRead, i;
	type_key *k = NULL;

	logger("DEBUG","mainRoutine");

	setResFile();

	int dmn = autoConnect();
	int rs  = getRetrySecs();
	strcpy(callerId, "NO CALLER ID");
	
	int fe;
	if ((fe = getFrontEnd()) > 0) {
		connectFrontEnd(fe);
	}
	
	while (1){
	    
		// try to open port
		if(openPort(getDevice()) == 1 &&   // Open device
		   initPort() == 1 ) {  // Init modem: AT, ATE0, AT+CMER, in server mode waits for connection
		
			manageAlarms(ALARM_CLEAN);
			
                        isConnected = 1;
                        
			if (fe > 0) {
				sendToFrontEnd("Connected");
			}
	    	
			i = 0;    
	    		k = findItem(EVT_CONNECT, &i, NULL);
			if (k && i == FLAG_EXACT) {
		    		logger("INFO", "Exec cmd on connect");
				//printf("mainRoutine: Exec cmd on connect\n");
		    		ret = handleCmdByKey(k,NULL);
		    		if (ret == EXIT_ABORT || ret == EXIT_DISCON) {
				        printf("mainRoutine: Exec cmd on connect EXIT_ABORT/EXIT_DISCON \n");
					return;
		    		}
				if (getHttp()) {
					writeSocketConn("End();", 6);
				}
			}
			
			//printf("mainRoutine: connect OK\n");

			logger("INFO","Start event forwarding");
		
			while (1) {	 
		    		//logger("DEBUG","main loop");
		    
		    		retRead = read_command(answer, sizeof(answer));
				//printf("GOT %d\n",retRead);
				
		    		if (retRead == EOF) {
	    				logger("INFO", "Got disconnected");
					printf("Got disconnected\n");
					
					if (fe > 0) {
						sendToFrontEnd("Disconnected");
					}

					// cleanup cached data
					freeCachedData();
					
					i = 0;
	    				k = findItem(EVT_DISCONNECT, &i, NULL);
					if (k && i == FLAG_EXACT) {
			    			logger("INFO", "Exec cmd on disconnect");
			    			ret = handleCmdByKey(k,NULL);
	    		    			// No additional handling since we already disconnected
					} 

					if (gotSignal) {
			   	 		logger("INFO", "Got signal, exiting");
						closePort(1);
	    					freeAndExit();
					}
					
	    				if (!dmn && 
					    (serverMode == CLIENT_AT || 
					     serverMode == CLIENT_ILIRC)) {
					     
			   	 		logger("INFO", "Exit because of disconnect");
			    			freeAndExit();  // Exit here
	    				}
	    		    
	    				if (serverMode == CLIENT_AT    || 
					    serverMode == CLIENT_ILIRC || 
					    serverMode == CLIENT_NOAT) {  // In server mode we will reuse already opened socket
					    
	    		    			logger("INFO", "Closing the port");
			    			closePort(1);
					}   
	    				ret = EXIT_NOK;
                                        
                                        isConnected = 0;
	    		
	    				break;
		    		} else if (retRead > 0) {
					DEBUG2("-------------------- Command read -------------------- >%s<", answer);
	    	    
		    			if (serverMode == CLIENT_NOAT) {	// in IR communication a lot of 'empty' chars (= -1) could be added
						char * k2 = answer;
						while (*k2 != '\0') {
							if (*k2 == -1) {
								*k2 = '\r';
							}
							k2++;
						}
					}

					ret = handle_command(answer);
	    		
	    				if (ret == EXIT_OK) {
						if (getHttp()) {
							writeSocketConn("End();", 6);
						}
	    		    			continue;
	    				} else if (ret == EXIT_ABORT) {
			   			 freeAndExit();  // Exit here
	    				} else if (ret == EXIT_DISCON) {
	    		    			break;
	    				}
	    	    		}
		    
		    		//Is any alarm fired (timeout about 15 seconds) ?
	    	    		if (alarmTimer == 300) {
	    				manageAlarms(ALARM_CHECK);
	    				alarmTimer = 0;
	    	    		} else {
	    				alarmTimer++;
	    	    		}

		    		// Is call still active (timeout about 1 seconds) ?
		    		if (callOn) {
	    				if (callTimer >= 20) {
	    		   			 logger("DEBUG", "Get call list again");
	    		    
	    		    			// Will set global callerId     [MAXLEN];
	    		    			ret = getClip(callerId);
	    		    			if (ret == EXIT_EXACT) {
	    						DEBUG2("Call still active (%s)", callerId);
	    		    			} else {
	    						logger("DEBUG","Call was finished");
	    						callOn = 0;
	    						strcpy(callerId, "NO CALLER ID");

	    						int i = 0;
	    						type_key *k = findItem(EVT_ENDCALL, &i, NULL);
	    						if (k != NULL && i == FLAG_EXACT) {
	    			    				logger("INFO","EVT_ENDCALL found");
	    			    				ret = handleCmdByKey(k,NULL);
	    			    				if (ret == EXIT_ABORT) {
									freeAndExit();      // Exit here
	    			    				} else if (ret == EXIT_DISCON) {
	    								break;
	    			    				}
	    						}
	    		    			}

	    		   			callTimer = 0;
	    				} else {
	    		    			callTimer++;
	    				}
	    	    		}
	    	    
	    	    		// Should we auto-repeat command (timeout about 1/10 second) ?
	    	    		if (repeatNow) {
					int isOdd = ((int)repeatTimer/2)*2;
					
					if (repeatTimer > 100) {  // Check for infinite loop. Only 50 autorepeats per one press
						cleanAutoRepeatFlag();
	    				} else if (repeatTimer > 0 && isOdd == repeatTimer) {
	    					DEBUG2("Auto repeat command ... (%d)", repeatTimer);

	    		    			ret = handleCmdByKey(repeatCmdPtr,NULL);
	    		    
	    		    			if (ret == EXIT_ABORT) {
							freeAndExit();      // Exit here
	    		    			} else if (ret == EXIT_DISCON) {
	    						break;
	    		    			}
						repeatTimer++;
	    				} else {
						//logger("DEBUG","Auto repeat command ...Skip");
	    		    			repeatTimer++;
	    				}
	    	    		}

		    		// Verify commands executed by timer (timeout about 1/2 sec)
	    	    		if (commandTimer == 10) {
	    				//logger("DEBUG","Verify timer commands ...");
	    				ret = verifyTimers(1) ;
	    				commandTimer = 0;
					
	    	    		} else {
	    				commandTimer++;
	    	    		}
				
		    		// Verify commands from front-end (timeout about 1/2 sec)
				if (fe > 0) {
	    	    			if (feTimer == 10) {
	    					//logger("DEBUG","Verify front end timer...");
					
						ret = readFromFrontEnd(frontEndBuf, MAXLEN);
						if (ret == EXIT_OK) {
							if (frontEndBuf[0] != '\0') {
								DEBUG2("readFromFrontEnd() >%s<", frontEndBuf);

								ret = execDynamically(frontEndBuf);
								frontEndBuf[0] = '\0';
	    		    					if (ret == EXIT_ABORT) {
									freeAndExit();      // Exit here
	    		    					} else if (ret == EXIT_DISCON) {
	    								break;
	    		    					}			
							}
						}	
	    	    			} else {
	    					feTimer++;
	    	    			}
				}
				
	    	    		// Main loop timer (1/20 of second)
		    		usleep(50000);
		 	}  
		}
		
		printf("Connection closed or lost\n");

		// Can't open port or it closed again
		if (dmn || 
		    serverMode == SERVER_BT   || 
		    serverMode == SERVER_TCP  ||
		    serverMode == SERVER_UX   || 
		    serverMode == CLIENT_NOAT || 
		    ret == EXIT_DISCON) {
		    
	    		int timeout;
	    	
	    		if (serverMode == SERVER_BT  || 
		            serverMode == SERVER_TCP ||
		            serverMode == SERVER_UX  || 
		            serverMode == CLIENT_NOAT) {
			    
				timeout = 5;	// wait only 5 seconds  
				 
			} else if (ret == EXIT_DISCON) {
			
	    	   		timeout = TMPDISCONN_TIME;
	    	    		ret = EXIT_OK;
	    		 	    		
			} else {
	    	    		timeout = rs;
	    		}

	    		INFO2("Wait %d seconds to connect/open server socket ...", timeout);
			printf("Wait %d seconds to connect/open server socket ...\n", timeout);
			
	    		sleep(timeout);
	    	
	    	} else {	// Proceed to exit
	    		break;
	    	}
	}

	// Finish all
	logger("INFO", "Stop process ...");
	closePort(1);
	
	return;
}

//
// Bemused server limited emulation
// we could get more than one command at once
//
      
static int isBemusedCommand(char *cmdIn)
{  
	DEBUG2("isBemusedCommand: >%s<",cmdIn);

        int isBemused = 0;

        char *cmd = cmdIn;
        char *last = cmdIn + strlen(cmdIn);

        while (cmd < last) {
        	int handle = 0;
                int shift  = 4;

                char oneCmd[10];
                char paramValue[8];
                paramValue[0] = '\0';

                if (strncmp(cmd, "CHCK", 4) == 0 ||
                    strncmp(cmd, "DINF", 4) == 0 ||
                    strncmp(cmd, "EXIT", 4) == 0 ||
                    strncmp(cmd, "FADE", 4) == 0 ||
                    strncmp(cmd, "FFWD", 4) == 0 ||
                    strncmp(cmd, "GVOL", 4) == 0 ||
                    strncmp(cmd, "INF2", 4) == 0 ||
                    strncmp(cmd, "INFO", 4) == 0 ||
                    strncmp(cmd, "LIST", 4) == 0 ||
                    strncmp(cmd, "NEXT", 4) == 0 ||
                    strncmp(cmd, "PAUS", 4) == 0 ||
                    strncmp(cmd, "PLEN", 4) == 0 ||
                    strncmp(cmd, "PLST", 4) == 0 ||
                    strncmp(cmd, "PREV", 4) == 0 ||
                    strncmp(cmd, "RMAL", 4) == 0 ||
                    strncmp(cmd, "RWND", 4) == 0 ||
                    strncmp(cmd, "SHUT", 4) == 0 ||
                    strncmp(cmd, "STEN", 4) == 0 ||
                    strncmp(cmd, "STOP", 4) == 0 ||
                    strncmp(cmd, "STRT", 4) == 0 ||
                    strncmp(cmd, "VERS", 4) == 0) {
                	handle = 1;
                }

                if (strncmp(cmd, "REPT", 4) == 0 ||
                    strncmp(cmd, "SHFL", 4) == 0) {
                
			// Just "toggle" functionality
                        //unsigned char bc = ((unsigned char) *(cmd + 4));
                        //sprintf(paramValue, "%d",bc);
                
                        handle = 1;
                        shift  = 5;
                }
                if (strncmp(cmd, "VOLM", 4) == 0) {
                	unsigned char bc;
                        if (strlen(cmd) == 4) {  // read only VOLM without value to set
                        	logger("DEBUG","read one more byte");
                                char ret;
                                //if ((ret = readSocketConn((char*)buf, 2)) != 1) {
                                char c;
                                if ((ret = bt_readchar(&c,500000)) < 0) {
                                	DEBUG2("isBemusedCommand: read < 0");
                                        return 0;
                                }
                                sprintf(tmp, "isBemusedCommand: read >%c<", c);
                                logger("DEBUG",tmp);
                                bc = ((unsigned char) c);
                        } else {
                        	bc = ((unsigned char) *(cmd + 4));
                        }

                        // VOLM 0-255 -> 0%-100%
                        sprintf(paramValue, "%d", ((int)bc * 100)/255);
        
                        DEBUG2("isBemusedCommand: VOLM parameter >%s<", paramValue);
                
                        handle = 1;
                        shift  = 5;
                }
                if (strncmp(cmd, "SLCT", 4) == 0) {
		
                	unsigned char bc1 = *((unsigned char*) (cmd + 4));
                	unsigned char bc2 = *((unsigned char*) (cmd + 5));
			
			unsigned int ix = bc1*256+bc2;
                        sprintf(paramValue, "%d",ix);
			
			DEBUG2("isBemusedCommand: SLCT parameter >%s<", paramValue);
                	
			handle = 1;
                        shift  = 6;
                }
                if (strncmp(cmd, "SEEK", 4) == 0)  {    

                	sprintf(paramValue, "%d", 
                                     (((unsigned char)*(cmd + 4)) << 24) + 
                                     (((unsigned char)*(cmd + 5)) << 16) + 
                                     (((unsigned char)*(cmd + 6)) << 8)  + 
                                       (unsigned char)*(cmd + 7));
                
                        handle = 1;
                        shift  = 8;
                }
                if (strncmp(cmd, "DLST", 4) == 0 || 
                    strncmp(cmd, "DOWN", 4) == 0 || 
                    strncmp(cmd, "FINF", 4) == 0 ||    
                    strncmp(cmd, "LADD", 4) == 0 ||
                    strncmp(cmd, "PLAY", 4) == 0) {             
                    
                	unsigned char bc = *((unsigned char*) (cmd + 4));
                        sprintf(paramValue, "%d",bc);
                
                        shift  = 5 + bc;
                        handle = 1;
                }
                if (handle) {
                	isBemused = 1;
                        strncpy(oneCmd,cmd,4); 
                        oneCmd[4] = '\0';
                        if (paramValue[0] != '\0') {
                        	strcat(oneCmd,"(-1,");
                                strcat(oneCmd,paramValue);
                                strcat(oneCmd,")");
                        }
                
                        DEBUG2("isBemusedCommand: one commmand >%s<", oneCmd);
                
                        handle_key_press(oneCmd,1);
                } else {
                	break;
                }

                cmd = cmd + shift;
        }
        return isBemused;
}

//
// LIRC dev/event handling
//
      
static int isLircCommand(char *cmdIn)
{  
	// should got replay in form "2 0 KEY_1 event6"
	
	DEBUG2("isLircCommand: >%s<",cmdIn);

	/*char *f = cmdIn;
	sprintf(tmp, "isLircCommand: DUMP >%c< >%d<",*f,*f);
        logger("DEBUG",tmp);
	*/
	
        int isLirc = 1;

	char *data = NULL;
        char *cmd = strdup(cmdIn);
        char *bufPtr;
        char *token = strtok_r(cmd," ",&bufPtr);
	if (token == NULL) {
		free(cmd);
		return 0;
	} 
	
	token = strtok_r(NULL," ",&bufPtr);
	if (token == NULL) {
		free(cmd);
		return 0;
	} 

	token = strtok_r(NULL," ",&bufPtr);
	if (token && strncmp(token, "KEY_",4)==0) {
		data = token + 4;
		token = strtok_r(NULL," ",&bufPtr);
		if (token && strncmp(token, "event",5) != 0 ) {
			isLirc = 0;
		}
	} else {
		isLirc = 0;
	}

        if (isLirc) {
        	DEBUG2("isLircCommand: one commmand >%s<", data);
		handle_key_press(data,1);
	}
	 	
	free(cmd);
        return isLirc;
}
